﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>

#include <algorithm>
#include <cmath>
#include <cstdarg>
#include <cstdio>
#include <cstring>
#include <mutex>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include "SimpleGfx_PostEffect.h"
#include "SimpleGfx_TypesInternal.h"
#include "detail/SimpleGfx_ApiInternal.h"

namespace nns { namespace sgx {

namespace
{

// ポストエフェクト用頂点シェーダ
const char* VertexShaderPostEffect = R"(
#version 320 es
layout(location = 0) in highp   vec3 a_Position;
layout(location = 1) in mediump vec2 a_TexCoord;
out mediump vec2 v_TexCoord;

void main() {
    gl_Position = vec4(a_Position, 1.0);
    v_TexCoord  = a_TexCoord;
}
)";

// ブラーエフェクト用フラグメントシェーダ
const char* FragmentShaderBlurEffect = R"(
#version 320 es
layout(location = 0) out lowp vec4 o_FragColor;
in mediump vec2 v_TexCoord;

uniform      sampler2D s_Decal;
uniform lowp vec2      u_Offset;

void main() {
    lowp vec4 c = vec4(0, 0, 0, 0);

/*
    c += ( 1.0f / 64.0f) * texture(s_Decal, v_TexCoord + -3.0 * u_Offset);
    c += ( 6.0f / 64.0f) * texture(s_Decal, v_TexCoord + -2.0 * u_Offset);
    c += (15.0f / 64.0f) * texture(s_Decal, v_TexCoord + -1.0 * u_Offset);
    c += (20.0f / 64.0f) * texture(s_Decal, v_TexCoord);
    c += (15.0f / 64.0f) * texture(s_Decal, v_TexCoord + 1.0 * u_Offset);
    c += ( 6.0f / 64.0f) * texture(s_Decal, v_TexCoord + 2.0 * u_Offset);
    c += ( 1.0f / 64.0f) * texture(s_Decal, v_TexCoord + 3.0 * u_Offset);
*/
    c += (1.0f / 16.0f) * texture(s_Decal, v_TexCoord + -2.0 * u_Offset);
    c += (4.0f / 16.0f) * texture(s_Decal, v_TexCoord + -1.0 * u_Offset);
    c += (6.0f / 16.0f) * texture(s_Decal, v_TexCoord);
    c += (4.0f / 16.0f) * texture(s_Decal, v_TexCoord + 1.0 * u_Offset);
    c += (1.0f / 16.0f) * texture(s_Decal, v_TexCoord + 2.0 * u_Offset);

    o_FragColor = c;
}
)";

/**
 * @brief   Blur シェーダの attribute の位置
 */
struct BlurShaderLocations
{
    GLuint position;
    GLuint texCoord;

    GLuint offset;
};

// GL_TRIANGLE_FAN を想定した座標
const Point3D ScreenVertexBufferData[] =
{
    {{ -1.0f, -1.0f, 0.0f }},
    {{  1.0f, -1.0f, 0.0f }},
    {{  1.0f,  1.0f, 0.0f }},
    {{ -1.0f,  1.0f, 0.0f }}
};

const Point2D FullTexCoordData[] =
{
    {{ 0.0f, 0.0f }},
    {{ 1.0f, 0.0f }},
    {{ 1.0f, 1.0f }},
    {{ 0.0f, 1.0f }}
};

bool g_IsInitialized = false;

ShaderPack g_BlurShader;
BlurShaderLocations g_BlurLocation;

GLuint g_GlTextures[2];
GLuint g_GlBuffers[2];

GLuint g_DecalTexture;
GLuint g_RenderTexture;

GLuint g_Sampler;

GLuint g_VertexArray;
GLuint g_VertexBuffer;
GLuint g_TexCoordBuffer;

GLuint g_FrameBuffers[1];

/**
 * @brief   ブラーエフェクトの初期化
 */
void InitializeBlurEffect(const Rectangle& rect) NN_NOEXCEPT
{
    auto canvasSize = detail::GetCanvasSize();

    glUseProgram(g_BlurShader.programId);

    // フレームバッファを Texture 0 に転送
    glActiveTexture(GL_TEXTURE0);
    glBindTexture(GL_TEXTURE_2D, g_DecalTexture);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
    glCopyTexImage2D(
        GL_TEXTURE_2D,
        0,
        GL_RGBA,
        static_cast<GLint>(rect.x),
        static_cast<GLint>(canvasSize.height - rect.y - rect.height),
        static_cast<GLsizei>(rect.width),
        static_cast<GLsizei>(rect.height),
        0
    );

    // 頂点バッファの設定
    glBindVertexArray(g_VertexArray);
    glBindBuffer(GL_ARRAY_BUFFER, g_VertexBuffer);
    glEnableVertexAttribArray(g_BlurLocation.position);
    glVertexAttribPointer(
        g_BlurLocation.position,
        NN_ARRAY_SIZE(Point3D::v),
        GL_FLOAT,
        GL_FALSE,
        0,
        nullptr
    );

    // テクスチャ座標の設定
    glBindBuffer(GL_ARRAY_BUFFER, g_TexCoordBuffer);
    glEnableVertexAttribArray(g_BlurLocation.texCoord);
    glVertexAttribPointer(
        g_BlurLocation.texCoord,
        NN_ARRAY_SIZE(Point2D::v),
        GL_FLOAT,
        GL_FALSE,
        0,
        nullptr
    );
    glBufferData(
        GL_ARRAY_BUFFER,
        sizeof(FullTexCoordData),
        FullTexCoordData,
        GL_STATIC_DRAW
    );

#if 0
    // カーネル係数バッファの準備 (Texture 1)
    const GLfloat gaussianKernel[] =
    {
        1.0f / 16.0f,
        4.0f / 16.0f,
        6.0f / 16.0f,
        4.0f / 16.0f,
        1.0f / 16.0f
    };
    glBindBuffer(GL_TEXTURE_BUFFER, g_KernelBufferId);
    glBufferData(
        GL_TEXTURE_BUFFER,
        sizeof(gaussianKernel),
        gaussianKernel,
        GL_STATIC_DRAW
    );

    glActiveTexture(GL_TEXTURE1);
    glBindTexture(GL_TEXTURE_BUFFER, g_KernelTexture);
    glTexBuffer(GL_TEXTURE_BUFFER, GL_R32F, g_KernelBufferId);
#endif

    // 描画ターゲットの設定 (Texture 2)
    glActiveTexture(GL_TEXTURE2);
    glBindTexture(GL_TEXTURE_2D, g_RenderTexture);
    glTexImage2D(
        GL_TEXTURE_2D,
        0,
        GL_RGBA,
        static_cast<GLsizei>(rect.width),
        static_cast<GLsizei>(rect.height),
        0,
        GL_RGBA,
        GL_UNSIGNED_BYTE,
        nullptr
    );

    // 1 パス目を描画するフレームバッファを構築
    glBindFramebuffer(GL_FRAMEBUFFER, g_FrameBuffers[0]);
    glFramebufferTexture2D(
        GL_FRAMEBUFFER,
        GL_COLOR_ATTACHMENT0,
        GL_TEXTURE_2D,
        g_RenderTexture,
        0
    );

    GLuint drawBuffers[] = { GL_COLOR_ATTACHMENT0 };
    glDrawBuffers(1, drawBuffers);

#if 0
    {
        GLenum status = glCheckFramebufferStatus(GL_FRAMEBUFFER);
        if (status != GL_FRAMEBUFFER_COMPLETE)
        {
            NN_LOG("Frame buffer is corrupted (%d)\n", status);
        }
    }
#endif

}  // NOLINT(readability/fn_size)

}  // anonymous

void InitializePostEffect() NN_NOEXCEPT
{
    if (g_IsInitialized)
    {
        return;
    }

    glGenTextures(NN_ARRAY_SIZE(g_GlTextures), g_GlTextures);
    glGenBuffers(NN_ARRAY_SIZE(g_GlBuffers), g_GlBuffers);
    glGenSamplers(1, &g_Sampler);
    glGenVertexArrays(1, &g_VertexArray);
    glGenFramebuffers(NN_ARRAY_SIZE(g_FrameBuffers), g_FrameBuffers);

    g_DecalTexture   = g_GlTextures[0];
    g_RenderTexture  = g_GlTextures[1];

    g_VertexBuffer   = g_GlBuffers[0];
    g_TexCoordBuffer = g_GlBuffers[1];

    // シェーダの初期化
    detail::CreateShader(&g_BlurShader, &VertexShaderPostEffect, &FragmentShaderBlurEffect);

    //g_BlurLocation.position = detail::GetAttribLocationWithAssert(g_BlurShader, "a_Position");
    //g_BlurLocation.texCoord = detail::GetAttribLocationWithAssert(g_BlurShader, "a_TexCoord");
    //NN_LOG("a_Position: %u\n", g_BlurLocation.position);
    //NN_LOG("a_TexCoord: %u\n", g_BlurLocation.texCoord);
    g_BlurLocation.position = 0;
    g_BlurLocation.texCoord = 1;

    g_BlurLocation.offset = glGetUniformLocation(g_BlurShader.programId, "u_Offset");
    NN_ASSERT_GREATER_EQUAL(g_BlurLocation.offset, 0);

    // サンプラーの設定
    glSamplerParameteri(g_Sampler, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
    glSamplerParameteri(g_Sampler, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    glSamplerParameteri(g_Sampler, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
    glSamplerParameteri(g_Sampler, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);

    g_IsInitialized = true;
}

void FinalizePostEffect() NN_NOEXCEPT
{
    if (!g_IsInitialized)
    {
        return;
    }

    detail::DeleteShader(g_BlurShader);

    glDeleteFramebuffers(NN_ARRAY_SIZE(g_FrameBuffers), g_FrameBuffers);
    glDeleteVertexArrays(1, &g_VertexArray);
    glDeleteSamplers(1, &g_Sampler);
    glDeleteBuffers(NN_ARRAY_SIZE(g_GlBuffers), g_GlBuffers);
    glDeleteTextures(NN_ARRAY_SIZE(g_GlTextures), g_GlTextures);

    g_IsInitialized = false;
}

bool IsPostEffectInitialized() NN_NOEXCEPT
{
    return g_IsInitialized;
}

void ApplyBlurEffect(float offsetX, float offsetY) NN_NOEXCEPT
{
    auto canvasSize = detail::GetCanvasSize();
    ApplyBlurEffect(offsetX, offsetY, { { 0, 0, canvasSize.width, canvasSize.height } });
}

void ApplyBlurEffect(
    float offsetX,
    float offsetY,
    const Rectangle& rect) NN_NOEXCEPT
{
    NN_ASSERT(g_IsInitialized, "PostEffect is not initialized");

    Rectangle prevRenderArea;
    detail::GetCurrentRenderArea(&prevRenderArea);

    InitializeBlurEffect(rect);

    // 1 パス (X 方向)
    {
        // 一次処理用のフレームバッファに切り替えてクリア
        glBindFramebuffer(GL_FRAMEBUFFER, g_FrameBuffers[0]);
        glViewport(0, 0, rect.width, rect.height);
        glClear(GL_COLOR_BUFFER_BIT);

        // X 方向のぶれを設定
        glUniform2f(g_BlurLocation.offset, offsetX / rect.width, 0);

        // テクスチャの設定
        glActiveTexture(GL_TEXTURE0);
        glBindTexture(GL_TEXTURE_2D, g_DecalTexture);
        glBindSampler(0, g_Sampler);

        // 頂点の設定
        glBindVertexArray(g_VertexArray);
        glBindBuffer(GL_ARRAY_BUFFER, g_VertexBuffer);
        glBufferData(
            GL_ARRAY_BUFFER,
            sizeof(ScreenVertexBufferData),
            ScreenVertexBufferData,
            GL_STATIC_DRAW
        );

        // 描画
        glDrawArrays(GL_TRIANGLE_FAN, 0, NN_ARRAY_SIZE(ScreenVertexBufferData));
        //glFlush();
    }

    // 2 パス (Y 方向)
    {
        auto canvasSize = detail::GetCanvasSize();

        // デフォルトのフレームバッファに戻す
        glBindFramebuffer(GL_FRAMEBUFFER, 0);
        glViewport(0, 0, canvasSize.width, canvasSize.height);

        // Y 方向のぶれを設定
        glUniform2f(g_BlurLocation.offset, 0, offsetY / rect.height);

        // テクスチャの設定
        glActiveTexture(GL_TEXTURE0);
        glBindTexture(GL_TEXTURE_2D, g_RenderTexture);
        glBindSampler(0, g_Sampler);

        // 頂点の設定
        glBindVertexArray(g_VertexArray);

        // 上下反転
        float left   = 2.0f * rect.x / canvasSize.width - 1.0f;
        float right  = left + 2.0f * rect.width / canvasSize.width;
        float top    = 2.0f * (canvasSize.height - rect.y - rect.height) / canvasSize.height - 1.0f;
        float bottom = top + 2.0f * rect.height / canvasSize.height;
        const Point3D vertices[] =
        {
            { { left,  top,    0.0f } },
            { { right, top,    0.0f } },
            { { right, bottom, 0.0f } },
            { { left,  bottom, 0.0f } }
        };
        glBindBuffer(GL_ARRAY_BUFFER, g_VertexBuffer);
        glBufferSubData(GL_ARRAY_BUFFER, 0, sizeof(vertices), vertices);

        // 描画
        glDrawArrays(GL_TRIANGLE_FAN, 0, NN_ARRAY_SIZE(vertices));
        //glFlush();
    }

    // 後始末
    glBindTexture(GL_TEXTURE_2D, 0);
    glBindSampler(0, 0);
    glBindVertexArray(0);
    detail::SetRenderArea(prevRenderArea.x, prevRenderArea.y, prevRenderArea.width, prevRenderArea.height);

    // 次回描画時に強制的にシェーダを切り替える
    detail::ChangeShader(ShaderType::None);
}

}}  // nns::sgx
