﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       シークバー作成ための API の宣言
 */

#pragma once

#include "layout_Base.h"

namespace nns { namespace hidfw { namespace layout {

    /**
     * @brief       シークバーを作成するためのクラスです
     */
    class SeekBar : public BaseItem
    {
    public:
        SeekBar() NN_NOEXCEPT;
        virtual ~SeekBar() NN_NOEXCEPT {}

        virtual void Update() NN_NOEXCEPT;
        virtual void PrintText() NN_NOEXCEPT;
        virtual void Draw() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief  デフォルト値を設定
        //----------------------------------------------------------------
        virtual void SetDefault() NN_NOEXCEPT;

        void SetX(float x) NN_NOEXCEPT { m_Pos.x = x - m_Margin.x; }
        void SetY(float y) NN_NOEXCEPT { m_Pos.y = y - m_Margin.y; }
        void SetPos(float x, float y) NN_NOEXCEPT { m_Pos = NN_UTIL_FLOAT_2_INITIALIZER(x - m_Margin.x, y - m_Margin.y); }
        void SetWidth(float w) NN_NOEXCEPT { m_Size.x = w + m_Margin.x * 2; }
        void SetHeight(float h) NN_NOEXCEPT { m_Size.x = h + m_Margin.y * 2; }
        void SetSize(float w, float h) NN_NOEXCEPT { m_Size = NN_UTIL_FLOAT_2_INITIALIZER(w + m_Margin.x * 2, h + m_Margin.y * 2); }

        //----------------------------------------------------------------
        //! @brief      最大値を設定します
        //! @param[in]  value 最大値
        //----------------------------------------------------------------
        void SetMaxValue(int value) NN_NOEXCEPT { m_MaxValue = value; }

        //----------------------------------------------------------------
        //! @brief      バーの値を設定します
        //! @param[in]  設定値
        //----------------------------------------------------------------
        void SetValue(int value) NN_NOEXCEPT { m_Value = value; }

        //----------------------------------------------------------------
        //! @brief      バーの値を加算します
        //! @param[in]  value 加算値
        //----------------------------------------------------------------
        void AddValue(int value) NN_NOEXCEPT { m_Value = (m_Value + value) > m_MaxValue ? m_MaxValue : (m_Value + value) < m_MinValue ? m_MinValue : (m_Value + value); }

        //----------------------------------------------------------------
        //! @brief      バーの値を減算します
        //! @param[in]  value 減算値
        //----------------------------------------------------------------
        void SubValue(int value) NN_NOEXCEPT { m_Value = (m_Value - value) > m_MaxValue ? m_MaxValue : (m_Value - value) < m_MinValue ? m_MinValue : (m_Value - value); }

        //----------------------------------------------------------------
        //! @brief      バーの値を取得します
        //! @return     値
        //----------------------------------------------------------------
        int GetValue() const NN_NOEXCEPT { return m_Value; }

        //----------------------------------------------------------------
        //! @brief      バーの上側に現在の値を描画します
        //! @param[in]  描画カウント ( showCount / 60 秒)
        //----------------------------------------------------------------
        void ShowValue(uint64_t showCount) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      値が変化したときに自動的に現在の値を描画するか設定します
        //! @param[in]  有効か否か
        //----------------------------------------------------------------
        void EnableAutoShowValue(bool isEnable) NN_NOEXCEPT { m_IsValueAutoShowEnable = isEnable; }

        //----------------------------------------------------------------
        //! @brief      値の自動描画機能が有効か取得します
        //! @return     値の自動描画が有効な場合 true
        //----------------------------------------------------------------
        bool IsEnableAutoShowValue() const NN_NOEXCEPT { return m_IsValueAutoShowEnable; }

        //----------------------------------------------------------------
        //! @brief      値の描画が有効か
        //! @return     値を描画している最中なら true
        //----------------------------------------------------------------
        bool IsEnableShowValue() const NN_NOEXCEPT { return (m_ValueShowCount > m_FrameCount); }

    protected:
        float               m_BarSize;                  //!< バーの横幅
        float               m_BarPos;                   //!< バーの位置 (最小: 0.f 最大 1.f)

        int                 m_Value;                    //!< 現在値
        int                 m_MaxValue;                 //!< 最大値
        int                 m_MinValue;                 //!< 最小値

        nn::util::Color4u8  m_BaseColor;                //!< メータ部分の下地の色
        nn::util::Color4u8  m_BarColor;                 //!< 操作バーの色

        nn::util::Float2    m_ConvPos;                  //!< マージンを無視した表示座標
        nn::util::Float2    m_ConvSize;                 //!< マージンを無視したサイズ
        nn::util::Float2    m_TextAreaPos;              //!< マージンを含めたテキストの描画座標
        nn::util::Float2    m_TextAreaSize;             //!< マージンを含めたテキストの描画領域
        nn::util::Float2    m_Margin;                   //!< メーターは縦幅が狭くタッチ操作がし難いためマージンを設けます

        bool                m_IsValueAutoShowEnable;    //!< 現在値の自動描画のON/OFF
        uint64_t            m_ValueShowCount;           //!< 値の表示時間
        float               m_TextTransparence;         //!< テキストの透明度
    };

}}}
