﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       ダイアログの作成・表示のための API の宣言
 */

#pragma once

#include <map>

#include "../gfx.h"
#include "../hid.h"
#include "../layout/layout_Button.h"
#include "../layout/layout_ButtonGroup.h"

namespace nns { namespace hidfw { namespace layout {

    /**
     * @brief       ダイアログの表示を行います
     * @details     呼び出す前に下記2点の初期化が必要です
     *              nns::hidfw::hid::Input
     *              nns::hidfw::gfx::Graphics
     */
    class Dialog
    {
    public:
        /**
         * @brief       ダイアログの表示中に内部で呼び出される関数
         */
        typedef void(*DialogFunction)(void* executant, void* param);

        /**
         * @brief       ダイアログの表示結果
         */
        enum DialogResult
        {
            DialogResult_None,          //!< 何も設定されていません
            DialogResult_OK,            //!< OK が選択されました
            DialogResult_Cancel,        //!< キャンセル が選択されました
            DialogResult_Yes,           //!< 賛同ボタンが押された
            DialogResult_No,            //!< 拒否ボタンが押された
            DialogResult_Ignore         //!< ボタン以外の方法で終了したか結果を無視した
        };

        /**
         * @brief        ダイアログのボタン押下げ時に引数として渡される情報
         */
        struct DialogParam
        {
            DialogResult    Result;
            bool*           pOutIsExit;
            DialogResult*   pOutResult;
            uint64_t*       pOutExitCount;
        };

        static const int MaxDialogButtonCount = 8;              //!< ダイアログのボタンの種類の最大数

        /**
         * @brief       ダイアログで表示されるボタン
         */
        struct DialogButton
        {
            typedef ::nn::util::BitFlagSet<MaxDialogButtonCount, DialogButton>::Flag<0> OK;        //!< OK ボタン
            typedef ::nn::util::BitFlagSet<MaxDialogButtonCount, DialogButton>::Flag<1> Cancel;    //!< キャンセルボタン
            typedef ::nn::util::BitFlagSet<MaxDialogButtonCount, DialogButton>::Flag<2> Yes;       //!< 承認ボタン (≒OKボタン)
            typedef ::nn::util::BitFlagSet<MaxDialogButtonCount, DialogButton>::Flag<3> NO;        //!< 否定ボタン
            typedef ::nn::util::BitFlagSet<MaxDialogButtonCount, DialogButton>::Flag<7> Other;     //!< その他のボタン
        };
        typedef ::nn::util::BitFlagSet<MaxDialogButtonCount, DialogButton> DialogButtonSet;

        static Button DialogButton[MaxDialogButtonCount];           //!< ダイアログ上に表示されるボタン (デフォルト)

        /**
        * @brief        ダイアログの情報
        */
        struct DialogInfo
        {
            nn::util::Float2    Size;               //!< ダイアログのサイズ
            DialogButtonSet     Buttons;            //!< ダイアログに表示されるボタン (複数指定可能)
            std::string         Title;              //!< ダイアログのタイトル
            std::string         Message;            //!< ダイアログの本文
            DialogFunction      UpdateFunc;         //!< ダイアログの入力処理の更新直後に呼び出される関数
            DialogFunction      DrawerFunc;         //!< ダイアログのベース部分の描画直後に呼び出される関数
            void*               UpdateFuncParam;    //!< ダイアログの入力処理の更新直後に呼び出される関数に入力される引数
            void*               DrawFuncParam;      //!< ダイアログのベース部分の描画直後に呼び出される関数に入力される引数
            DialogInfo() NN_NOEXCEPT
            {
                Size = nn::util::MakeFloat2(800, 500);
                Buttons.Reset();
                Title = "Title";
                Message = "Message";
                UpdateFunc = DrawerFunc = nullptr;
                UpdateFuncParam = DrawFuncParam = nullptr;
            }
        };

    public:
        static const int OpenDialogCountMax = 4;         //!< 同時に展開できるダイアログの最大個数

    public:
        Dialog() NN_NOEXCEPT;
        ~Dialog() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief          呼び出しを行うダイアログの情報を設定します
        //! @details        DialogInfo は ShowDialog の引数に設定するか
        //!                 本関数を利用して ShowDialog の呼び出し前に設定しておく必要があります
        //! @param[in] info 呼び出すダイアログの情報
        //-------------------------------------------------------------
        void SetDialogInfo(const DialogInfo& info) NN_NOEXCEPT { m_DialogInfo = info; }

        //----------------------------------------------------------------
        //! @brief          設定されているダイアログの情報を取得します
        //! @return         設定されているダイアログの情報
        //-------------------------------------------------------------
        DialogInfo GetDialogInfo() NN_NOEXCEPT { return m_DialogInfo; }

        //----------------------------------------------------------------
        //! @brief          ダイアログを表示します
        //! @pre            ダイアログは最大 OpenDialogCountMax 個まで同時表示させる事が出来ます
        //! @details        呼び出し中は処理が停止します
        //----------------------------------------------------------------
        DialogResult ShowDialog() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief          ダイアログを表示します
        //! @details        呼び出し中は処理が停止します
        //! @param[in] info 呼び出すダイアログの情報
        //----------------------------------------------------------------
        DialogResult ShowDialog(const DialogInfo& info) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief          ダイアログを閉じます
        //! @details        関数呼び出しを行った次のフレームでダイアログの終了要求を発行します
        //----------------------------------------------------------------
        void CloseDialog() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief          ダイアログの標準ボタンを変更します
        //! @param[in] index  変更する DialogButton のインデックス
        //! @param[in] button 更新するボタン
        //----------------------------------------------------------------
        void SetDialogButton(int index, const nns::hidfw::layout::Button& button) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief          ダイアログの標準ボタンを元に戻します
        //----------------------------------------------------------------
        void ClearDialogButton() NN_NOEXCEPT;

    private:
        //----------------------------------------------------------------
        //! @brief      ダイアログのボタン押下げ時に呼び出される関数
        //! @details    開発者が呼び出すことは出来ません
        //----------------------------------------------------------------
        static void DialogFunc(void* pushButton, void* param);

    private:
        static bool                 IsInitialized;                  //!< 初期化済みか否か
        static int                  OpenDialogCount;                //!< 展開中のダイアログの数

        bool                        m_IsExit;                       //!< ダイアログの終了を許可する
        DialogResult                m_DialogResult;                 //!< ダイアログの戻り値
        DialogInfo                  m_DialogInfo;
        uint64_t                    m_ExitCount;                    //!< 0の時にダイアログの終了を試行する
        nn::util::Color4u8          m_DialogColor;                  //!< ダイアログの色
        nn::util::Color4u8          m_DialogShadowColor;            //!< ダイアログの影色
        Button m_DialogButton[MaxDialogButtonCount];                //!< ダイアログ上に表示されるボタン (コピー)
        std::map<int, nns::hidfw::layout::Button> m_CustomButton;   //!< 変更された標準ボタン

    };

}}}
