﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       振動操作のための API の宣言
 */

#pragma once

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/hid_Vibration.h>

namespace nns { namespace hidfw { namespace hid {

    /**
     * @brief       振動関連の設定、状態取得等を行うためのクラスです
     */
    class Vibration
    {
    public:
        static const int                    StyleCountMax = 5;                              //!< ハンドル取得の際に対象とするスタイルの個数
        static const int                    DeviceCountMax = 4;                             //!< 振動子の最大個数

    public:
        //----------------------------------------------------------------
        //! @brief              初期化します
        //! @details            全スタイルのハンドルを取得し、InitializeVibrationDevice を実行します
        //! @param[in] npadId   初期化する振動子を持つNpadId
        //----------------------------------------------------------------
        void Initialize(nn::hid::NpadIdType npadId) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief              振動を更新します
        //----------------------------------------------------------------
        void Update() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief              振動の送信予約をします
        //! @details            実際の振動値の送信処理は Update 内で一括して行われます。
        //! @param[in] vib      送信される振動値
        //----------------------------------------------------------------
        void Send(const nn::hid::VibrationValue& vib) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief              振動の送信予約をします
        //! @details            実際の振動値の送信処理は Update 内で一括して行われます。
        //! @param[in] leftVib  送信される振動値 (左手側)
        //! @param[in] rightVib 送信される振動値 (右手側)
        //----------------------------------------------------------------
        void Send(const nn::hid::VibrationValue& leftVib, const nn::hid::VibrationValue& rightVib) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief              振動の送信予約をします
        //! @details            実際の振動値の送信処理は Update 内で一括して行われます。
        //! @param[in] leftVib  送信される振動値 (左手側)
        //----------------------------------------------------------------
        void SendLeft(const nn::hid::VibrationValue& leftVib) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief              振動の送信予約をします
        //! @details            実際の振動値の送信処理は Update 内で一括して行われます。
        //! @param[in] rightVib 送信される振動値 (右手側)
        //----------------------------------------------------------------
        void SendRight(const nn::hid::VibrationValue& rightVib) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief              振動を停止します
        //! @details            振幅0の振動を送信します
        //----------------------------------------------------------------
        void StopVibration() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief              左手側の振動を停止します
        //! @details            振幅0の振動を送信します
        //----------------------------------------------------------------
        void StopLeftVibration() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief              右手側の振動を停止します
        //! @details            振幅0の振動を送信します
        //----------------------------------------------------------------
        void StopRightVibration() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief              送信されている実際の振動値を取得します
        //! @param[out] leftVib     実際の振動値 (左手側)
        //! @param[out] rightVib    実際の振動値 (右手側)
        //----------------------------------------------------------------
        void GetCurVibrationValue(nn::hid::VibrationValue* leftVib, nn::hid::VibrationValue* rightVib) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief              振動子のハンドル値を取得します
        //! @param[in]  style   取得対象とするスタイル
        //! @param[out] handle  ハンドルの格納先
        //! @param[in]  count   取得数
        //! @return             実際に取得できた振動子の数
        //----------------------------------------------------------------
        int GetVibrationDeviceHandle(nn::hid::NpadStyleSet style, nn::hid::VibrationDeviceHandle* handle, const int count) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief              直前のフレームで送信された振動値を取得します
        //! @param[out] vib     振動値の格納先
        //! @param[in] count    取得数
        //! @return             実際に取得できた振動の数
        //----------------------------------------------------------------
        int GetCurSendVibrationValue(nn::hid::VibrationValue* vib, const int count) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      振動を有効化します
        //! @details    本体設定は変更されません。
        //----------------------------------------------------------------
        void Enable() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      振動を無効化します
        //! @details    振動の送信を無視します、本体設定は変更されません
        //----------------------------------------------------------------
        void Disable() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      振動が有効化されているか否かを返します
        //! @details    本体設定の値は考慮されていません
        //! @return     振動が有効されている場合 true を返します
        //----------------------------------------------------------------
        bool IsEnable() NN_NOEXCEPT;

    private:
        bool                                m_IsEnable;                                     //!< この振動が有効か否か
        nn::hid::NpadIdType                 m_NpadId;
        nn::hid::NpadStyleSet               m_OldNpadStyleSet;
        nn::hid::NpadStyleSet               m_NpadStyleSet;
        nn::hid::VibrationValue             m_OldVibrationValue[DeviceCountMax];            //!< 直前のフレームで送された振動値
        nn::hid::VibrationValue             m_VibrationValue[DeviceCountMax];               //!< 振動値
        nn::hid::VibrationValue             m_CurrentVibrationValue[DeviceCountMax];        //!< 現在の振動値
        uint64_t                            m_VibrationCount[DeviceCountMax];               //!< 振動時間

        int                                 m_DeviceCount[StyleCountMax];                   //!< デバイス数
        nn::hid::VibrationDeviceHandle      m_Handle[StyleCountMax][DeviceCountMax];        //!< 振動子のハンドル
        nn::hid::VibrationDeviceInfo        m_VibrationInfo[StyleCountMax][DeviceCountMax]; //!< 振動デバイスの情報
    };

}}}
