﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "hid_Touch.h"

namespace nns { namespace hidfw { namespace hid {

    void Touch::Initialize() NN_NOEXCEPT
    {
        nn::hid::InitializeTouchScreen();
        nn::hid::InitializeGesture();
    }

    void Touch::Update() NN_NOEXCEPT
    {
        m_oldGestureState = m_GestureState[0];
        auto stateCount = nn::hid::GetGestureStates(m_GestureState, nn::hid::GestureStateCountMax);
        m_GestureCount = 0;
        for (int i = 0; i < stateCount; ++i)
        {
            if (m_oldGestureState.eventNumber < m_GestureState[i].eventNumber)
            {
                ++m_GestureCount;
            }
            else
            {
                break;
            }
        }

        m_oldTouchState = m_TouchState[0];
        m_TouchStateCount = nn::hid::GetTouchScreenStates<nn::hid::TouchStateCountMax>(m_TouchState, nn::hid::TouchScreenStateCountMax);
    }

    Touch& Touch::GetInstance() NN_NOEXCEPT
    {
        static Touch instance;
        return instance;
    }

    bool Touch::IsEnable(const nn::util::Float2& pos, const nn::util::Float2& size) NN_NOEXCEPT
    {
        nn::hid::TouchState state;
        nn::util::Float2 center = NN_UTIL_FLOAT_2_INITIALIZER(pos.x + size.x / 2.f, pos.y + size.y / 2.f);
        bool result = GetTouchState(&state, center);
        if (result)
        {
            result = (std::pow(state.x - center.x, 2) < std::pow(size.x / 2.f, 2)) && (std::pow(state.y - center.y, 2) < std::pow(size.y / 2.f, 2));
        }
        return result;
    }

    bool Touch::IsTouch(const nn::util::Float2& pos, const nn::util::Float2& size) NN_NOEXCEPT
    {
        nn::hid::GestureType gestureType;
        nn::hid::GestureState gestureState;
        if (GetGestureState(&gestureType, &gestureState, pos, size))
        {
            return (gestureType == nn::hid::GestureType_Touch);
        }
        return false;
    }

    bool Touch::IsTap(const nn::util::Float2& pos, const nn::util::Float2& size) NN_NOEXCEPT
    {
        nn::hid::GestureType gestureType;
        nn::hid::GestureState gestureState;
        if (GetGestureState(&gestureType, &gestureState, pos, size))
        {
            return (gestureType == nn::hid::GestureType_Tap);
        }
        return false;
    }

    bool Touch::IsPress(const nn::util::Float2& pos, const nn::util::Float2& size) NN_NOEXCEPT
    {
        nn::hid::GestureType gestureType;
        nn::hid::GestureState gestureState;
        if (GetGestureState(&gestureType, &gestureState, pos, size))
        {
            return (gestureType == nn::hid::GestureType_Press);
        }
        return false;
    }

    bool Touch::GetGestureState(nn::hid::GestureType* outType, nn::hid::GestureState* outState, const nn::util::Float2& pos, const nn::util::Float2& size) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(outType);
        NN_ASSERT_NOT_NULL(outState);

        bool isIdle = false;

        for (auto i = 0; i < m_GestureCount; ++i)
        {
            if (m_GestureState[i].eventNumber <= m_oldGestureState.eventNumber)
            {
                break;
            }
            else
            {
                if (static_cast<float>(m_GestureState[i].x) > pos.x && static_cast<float>(m_GestureState[i].x) < pos.x + size.x &&
                    static_cast<float>(m_GestureState[i].y) > pos.y && static_cast<float>(m_GestureState[i].y) < pos.y + size.y)
                {
                    if (m_GestureState[i].GetGestureType() != nn::hid::GestureType_Idle)
                    {
                        *outType = m_GestureState[i].GetGestureType();
                        *outState = m_GestureState[i];
                        return true;
                    }
                    else
                    {
                        if (!isIdle)
                        {
                            *outType = m_GestureState[i].GetGestureType();
                            *outState = m_GestureState[i];
                            isIdle = true;
                        }
                    }
                }
            }
        }
        return isIdle;
    }

    bool Touch::GetGestureState(nn::hid::GestureState* outState) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(outState);

        bool result = false;

        for (auto i = 0; i < m_GestureCount; ++i)
        {
            if (m_GestureState[i].eventNumber <= m_oldGestureState.eventNumber)
            {
                break;
            }
            else
            {
                if (m_GestureState[i].GetGestureType() != nn::hid::GestureType_Idle &&
                    m_GestureState[i].GetGestureType() != nn::hid::GestureType_Complete)
                {
                    *outState = m_GestureState[i];
                    return true;
                }
                else if (!result)
                {
                    *outState = m_GestureState[i];
                    result = true;
                }
            }
        }
        return result;
    }

    int Touch::GetGestureStates(nn::hid::GestureState* outState, int stateCount, const nn::util::Float2& pos, const nn::util::Float2& size) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(outState);
        NN_ASSERT_LESS_EQUAL(stateCount, nn::hid::GestureStateCountMax);

        int count = 0;

        for (int i = std::min(m_GestureCount - 1, stateCount - 1); i >= 0; --i)
        {
            if (
                (m_GestureState[i].GetGestureType() == nn::hid::GestureType_Idle) ||
                (m_GestureState[i].GetGestureType() == nn::hid::GestureType_Complete) ||
                (m_GestureState[i].GetGestureType() == nn::hid::GestureType_Cancel) ||
                (
                    (static_cast<float>(m_GestureState[i].x) > pos.x && static_cast<float>(m_GestureState[i].x) < pos.x + size.x) &&
                    (static_cast<float>(m_GestureState[i].y) > pos.y && static_cast<float>(m_GestureState[i].y) < pos.y + size.y)
                    )
                )
            {
                outState[count++] = m_GestureState[i];
            }
        }
        return count;
    }

    bool Touch::GetTouchState(nn::hid::TouchState* outState, const nn::util::Float2& nearPos) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(outState);

        bool result = false;

        double minDist = 9999999;

        for (auto i = 0; i < m_TouchStateCount; ++i)
        {
            if (m_TouchState[i].samplingNumber <= m_oldTouchState.samplingNumber)
            {
                break;
            }
            else
            {
                for (auto t = 0; t < m_TouchState[i].count; ++i)
                {
                    double dist = sqrt(pow(nearPos.x - static_cast<float>(m_TouchState[i].touches[t].x), 2) + pow(nearPos.y - static_cast<float>(m_TouchState[i].touches[t].y), 2));
                    if (dist < minDist)
                    {
                        minDist = dist;
                        *outState = m_TouchState[i].touches[t];
                        result = true;
                    }
                }
            }
        }
        return result;
    }

    Touch::Touch() NN_NOEXCEPT
    {

    }
    Touch::~Touch() NN_NOEXCEPT
    {

    }
}}}
