﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "gfx_GraphicsDrawer.h"

namespace nns { namespace hidfw { namespace gfx {

    GraphicsDrawer::~GraphicsDrawer() NN_NOEXCEPT
    {
        Finalize();
    }

    GraphicsDrawer& GraphicsDrawer::GetInstance() NN_NOEXCEPT
    {
        static nns::hidfw::gfx::GraphicsDrawer instance;
        return instance;
    }

    nns::gfx::PrimitiveRenderer::Renderer& GraphicsDrawer::GetPrimitiveRenderer() NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(m_pGraphicsSystem);
        return m_pGraphicsSystem->GetPrimitiveRenderer();
    }

    nns::hidfw::gfx::GraphicsSystem& GraphicsDrawer::GetGraphicsSystem() NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(m_pGraphicsSystem);
        return *m_pGraphicsSystem;
    }

    void GraphicsDrawer::Initialize(nns::hidfw::gfx::GraphicsSystem* system) NN_NOEXCEPT
    {
        m_pGraphicsSystem = system;
        NN_ASSERT_NOT_NULL(m_pGraphicsSystem);
        SetAntiAliasingType(AntiAliasingType_None);
        SetLineWidth(1.f);
        InitializeBuffer();
        StartScanScreen();
    }

    void GraphicsDrawer::Finalize() NN_NOEXCEPT
    {
        FinalizeBuffer();
    }

    void GraphicsDrawer::ResetRenderTarget() NN_NOEXCEPT
    {
        m_pGraphicsSystem->SetRenderTarget(m_pGraphicsSystem->GetDefaultRenderTarget());
    }

    void GraphicsDrawer::SetRenderTarget(nn::gfx::ColorTargetView* pColorTargetView) NN_NOEXCEPT
    {
        m_pGraphicsSystem->SetRenderTarget(pColorTargetView);
    }

    void GraphicsDrawer::SetRenderTarget(nn::gfx::ColorTargetView* pColorTargetView, nn::gfx::DepthStencilView* pDepthStencilView) NN_NOEXCEPT
    {
        m_pGraphicsSystem->SetRenderTarget(pColorTargetView, pDepthStencilView);
    }

    void GraphicsDrawer::SetClearColor(const nn::util::Color4u8& color) NN_NOEXCEPT
    {
        m_ClearColor = color;
    }

    nn::util::Color4u8  GraphicsDrawer::GetClearColor() const NN_NOEXCEPT
    {
        return m_ClearColor;
    }

    void GraphicsDrawer::SetColor(const nn::util::Color4u8& color) NN_NOEXCEPT
    {
        SetColor(GradationDirection::GradationDirection_None, color, color);
    }

    nn::util::Color4u8 GraphicsDrawer::GetColor() const NN_NOEXCEPT
    {
        return m_DrawColor[0];
    }

    void GraphicsDrawer::SetColor(const GradationDirection dir, const nn::util::Color4u8& color, const nn::util::Color4u8& color2) NN_NOEXCEPT
    {
        m_GradationDirection = dir;
        m_DrawColor[0] = color;
        m_DrawColor[1] = color2;
    }

    nns::hidfw::gfx::GraphicsDrawer::GradationDirection GraphicsDrawer::GetColor(nn::util::Color4u8* color, nn::util::Color4u8* color2) const NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(color);
        NN_ASSERT_NOT_NULL(color2);
        *color = m_DrawColor[0];
        *color2 = (m_GradationDirection == GradationDirection_None) ? m_DrawColor[0] : m_DrawColor[1];
        return m_GradationDirection;
    }

    void GraphicsDrawer::SetLineWidth(const float width) NN_NOEXCEPT
    {
        m_LineWidth = std::fabsf(width);
        GetGraphicsSystem().GetPrimitiveRenderer().SetLineWidth(m_LineWidth);
    }

    float GraphicsDrawer::GetLineWidth() const NN_NOEXCEPT
    {
        return m_LineWidth;
    }

    void GraphicsDrawer::BeginDraw() NN_NOEXCEPT
    {
        m_State = State_Started;
        GetGraphicsSystem().BeginDraw(m_ClearColor);
        SetLineWidth(1.f);
    }

    void GraphicsDrawer::EndDraw() NN_NOEXCEPT
    {
        GetGraphicsSystem().GetDebugFont().Draw(&(GetGraphicsSystem().GetCommandBuffer()));
        if (m_isEnableScreenScanJob)
        {
            nn::gfx::TextureCopyRegion destRegion;
            destRegion.SetDefault();
            destRegion.SetWidth(GetGraphicsSystem().GetScreenWidth());
            destRegion.SetHeight(GetGraphicsSystem().GetScreenHeight());
            nn::gfx::TextureCopyRegion workRegion;
            workRegion.SetDefault();
            workRegion.SetWidth(GetGraphicsSystem().GetScreenWidth() * m_ScreenShotScale.x);
            workRegion.SetHeight(GetGraphicsSystem().GetScreenHeight() * m_ScreenShotScale.y);
            nn::gfx::TextureCopyRegion srcRegion = destRegion;
            GetGraphicsSystem().GetCommandBuffer().BlitImage(&m_WorkBuffer, workRegion, GetGraphicsSystem().GetColorTargetTexture(), srcRegion, nn::gfx::ImageCopyFlag_LinearFilter);
            GetGraphicsSystem().GetCommandBuffer().BlitImage(&m_ScanBuffer, destRegion, &m_WorkBuffer, workRegion, nn::gfx::ImageCopyFlag_LinearFilter);
            if (m_isEnableScreenShotJob)
            {
                GetGraphicsSystem().GetCommandBuffer().BlitImage(&m_ScreenShotBuffer[m_ScanScreenSlot], destRegion, &m_WorkBuffer, workRegion, nn::gfx::ImageCopyFlag_LinearFilter);
            }
            m_isEnableScreenShotJob = false;
        }
        GetGraphicsSystem().EndDraw();
        GetGraphicsSystem().Synchronize(
            nn::TimeSpan::FromNanoSeconds(1000 * 1000 * 1000 / m_FrameRate));
        m_State = State_End;
    }

    void GraphicsDrawer::ScreenShot(const nn::util::Float2& scale) NN_NOEXCEPT
    {
        ScreenShot(ScreenShotSlot_Slot_Auto, scale);
    }

    void GraphicsDrawer::ScreenShot(ScreenShotSlot slot, const nn::util::Float2& scale) NN_NOEXCEPT
    {
        m_ScreenShotScale = scale;
        m_isEnableScreenShotJob = true;
        m_ScanScreenSlot = slot;
    }

    void GraphicsDrawer::Draw2DCircle(const nn::util::Float2& center, const float radius, const int32_t div, const float angle) NN_NOEXCEPT
    {
        Draw2DCircle(center, nn::util::MakeFloat2(radius, radius), div, angle);
    }

    void GraphicsDrawer::Draw2DCircle(const nn::util::Float2& center, const nn::util::Float2& radius, const int32_t div, const float angle) NN_NOEXCEPT
    {
        NN_ASSERT_GREATER_EQUAL(div, 3);
        NN_ASSERT_LESS(div, 512);

        nns::gfx::PrimitiveRenderer::Renderer* pRenderer = &(GetGraphicsSystem().GetPrimitiveRenderer());
        nns::gfx::PrimitiveRenderer::PrimitiveMesh mesh;

        const auto verticesCount = (m_AntiAliasingType == AntiAliasingType_Pseudo) ? div  * 2 + 1 : div + 1;        // 頂点数
        const auto indexCount = (m_AntiAliasingType == AntiAliasingType_Pseudo) ? (div + 1) * 4 : (div + 1) * 2;             // インデックス数
        if (mesh.Initialize(pRenderer->GetGpuBuffer(), verticesCount, indexCount,
            (nns::gfx::PrimitiveRenderer::VertexFormat)(nns::gfx::PrimitiveRenderer::VertexFormat_Pos | nns::gfx::PrimitiveRenderer::VertexFormat_Color | nns::gfx::PrimitiveRenderer::VertexFormat_Uv)) == false)
        {
            return;
        }
        //---------------------------
        // インデックス
        uint32_t* pIndexData = mesh.GetIndexBufferCpuAddress();
        uint32_t centerPointNumber = verticesCount - 1;
        uint32_t index = 0;
        // 内部
        for (int32_t i = 0; i <= div; i++)
        {
            pIndexData[index++] = centerPointNumber;
            pIndexData[index++] = i % div;
        }
        // 外周
        if (m_AntiAliasingType == AntiAliasingType_Pseudo)
        {
            for (int32_t i = 0; i <= div; i++)
            {
                pIndexData[index++] = (i % div);
                pIndexData[index++] = (i % div) + div;
            }
        }
        //---------------------------
        // 頂点と色
        float curDeg = -angle + 180.f;
        float addDeg = -360.f / static_cast<float>(div);
        float color = (m_GradationDirection == GradationDirection::GradationDirection_Out) ? 1.f : (m_GradationDirection != GradationDirection::GradationDirection_None) ? 0.5f : 0.f;  // [1]側の色にかかる係数

        nn::util::Float3* pPos = static_cast<nn::util::Float3*>(mesh.GetVertexBufferCpuAddress(nns::gfx::PrimitiveRenderer::VertexAttribute_Pos));
        nn::util::Float4* pColor = static_cast<nn::util::Float4*>(mesh.GetVertexBufferCpuAddress(nns::gfx::PrimitiveRenderer::VertexAttribute_Color));
        nn::util::Float2* pUv = static_cast<nn::util::Float2*>(mesh.GetVertexBufferCpuAddress(nns::gfx::PrimitiveRenderer::VertexAttribute_Uv));
        // 中心点を設定
        auto& centerVertex = pPos[centerPointNumber];
        auto& centerColor = pColor[centerPointNumber];
        auto& centerUv = pUv[centerPointNumber];

        centerVertex = nn::util::MakeFloat3(-1.f + (2.f * center.x) / GetGraphicsSystem().GetScreenWidth(), 1.f - (2.f * center.y) / GetGraphicsSystem().GetScreenHeight(), 0.f);
        nn::util::Color4f(nn::util::Color4u8::Lerp(m_DrawColor[0], m_DrawColor[1], color)).Get(&centerColor.x, &centerColor.y, &centerColor.z, &centerColor.w);
        centerUv = nn::util::MakeFloat2(0.5f, 0.5f);
        // 外周を設定
        double s, c;        // Sin, Cos 値
        auto rad = radius;  // 半径
        for (uint32_t i = 0; i < centerPointNumber; ++i)
        {
            if (i == static_cast<uint32_t>(div))
            {
                rad = nn::util::MakeFloat2(rad.x + 1.f, rad.y + 1.f);
                curDeg = -angle + 180.f;
            }
            s = nn::util::SinEst(nn::util::DegreeToRadian(curDeg));
            c = nn::util::CosEst(nn::util::DegreeToRadian(curDeg));
            pPos[i] = nn::util::MakeFloat3(-1.f + (2.f * (center.x + s * rad.x)) / GetGraphicsSystem().GetScreenWidth(), 1.f - (2.f * (center.y + c * rad.y)) / GetGraphicsSystem().GetScreenHeight(), 0.f);
            pUv[i] = nn::util::MakeFloat2((s + 1.f) / 2.f, (c + 1.f) / 2.f);
            switch (m_GradationDirection)
            {
            case nns::hidfw::gfx::GraphicsDrawer::GradationDirection_None:
                color = 0.f;
                break;
            case nns::hidfw::gfx::GraphicsDrawer::GradationDirection_Left:
                color = 1.f - ((nn::util::SinEst(nn::util::DegreeToRadian(curDeg + angle)) + 1.f) / 2.f);
                break;
            case nns::hidfw::gfx::GraphicsDrawer::GradationDirection_Up:
                color = 1.f - ((nn::util::CosEst(nn::util::DegreeToRadian(curDeg + angle)) + 1.f) / 2.f);
                break;
            case nns::hidfw::gfx::GraphicsDrawer::GradationDirection_Right:
                color = (nn::util::SinEst(nn::util::DegreeToRadian(curDeg + angle)) + 1.f) / 2.f;
                break;
            case nns::hidfw::gfx::GraphicsDrawer::GradationDirection_Down:
                color = (nn::util::CosEst(nn::util::DegreeToRadian(curDeg + angle)) + 1.f) / 2.f;
                break;
            case nns::hidfw::gfx::GraphicsDrawer::GradationDirection_In:
                color = 0.f;
                break;
            case nns::hidfw::gfx::GraphicsDrawer::GradationDirection_Out:
                color = 1.f;
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
            nn::util::Color4f(nn::util::Color4u8::Lerp(m_DrawColor[0], m_DrawColor[1], color)).Get(&pColor[i].x, &pColor[i].y, &pColor[i].z, &pColor[i].w);;
            if (i >= static_cast<uint32_t>(div)) { pColor[i].w = 0.f; }
            curDeg += addDeg;
        }
        pRenderer->SetColor(m_DrawColor[0]);
        pRenderer->DrawUserMesh(&(GetGraphicsSystem().GetCommandBuffer()), nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleStrip, &mesh);
    }

    void GraphicsDrawer::Draw2DCircleFrame(const nn::util::Float2& center, const nn::util::Float2& radius, const float weight, const int32_t div, const float angle, const float length) NN_NOEXCEPT
    {
        NN_ASSERT_GREATER_EQUAL(div, 3);
        NN_ASSERT_LESS(div, 512);

        nns::gfx::PrimitiveRenderer::Renderer* pRenderer = &(GetGraphicsSystem().GetPrimitiveRenderer());
        nns::gfx::PrimitiveRenderer::PrimitiveMesh mesh;

        auto drawLength = length > 360.f ? 1.f : length < 0.f ? 0.f : length / 360.f;
        const auto drawDiv = static_cast<int32_t>(std::roundf(div * drawLength)) + 1;

        if (drawDiv < 3) { return; }

        const auto verticesCount = (m_AntiAliasingType == AntiAliasingType_Pseudo) ? drawDiv * 4 : drawDiv * 2;                 // 頂点数
        const auto indexCount = (m_AntiAliasingType == AntiAliasingType_Pseudo) ? drawDiv * 6 : drawDiv * 2;        // インデックス数
        if (mesh.Initialize(pRenderer->GetGpuBuffer(), verticesCount, indexCount,
            (nns::gfx::PrimitiveRenderer::VertexFormat)(nns::gfx::PrimitiveRenderer::VertexFormat_Pos | nns::gfx::PrimitiveRenderer::VertexFormat_Color)) == false)
        {
            return;
        }
        //---------------------------
        // インデックス
        uint32_t* pIndexData = mesh.GetIndexBufferCpuAddress();
        uint32_t index = 0;
        uint32_t lineCount = (m_AntiAliasingType == AntiAliasingType_Pseudo) ? 3 : 1;
        // 内部
        for (uint32_t i = 0; i < lineCount; ++i)
        {
            if (i % 2 == 0)
            {
                for (int32_t j = 0; j < drawDiv; ++j)
                {
                    pIndexData[index++] = (j % drawDiv) + i * drawDiv;
                    pIndexData[index++] = (j % drawDiv) + (i + 1) * drawDiv;
                }
            }
            else
            {
                for (int32_t j = drawDiv - 1; j >= 0; --j)
                {
                    pIndexData[index++] = (j % drawDiv) + (i + 1) * drawDiv;
                    pIndexData[index++] = (j % drawDiv) + i * drawDiv;
                }
            }
        }

        //---------------------------
        // 頂点と色
        float    curDeg = -angle + 180.f;
        float    addDeg = -360.f / static_cast<float>(div);
        // 座標
        nn::util::Float3* pPos = static_cast<nn::util::Float3*>(mesh.GetVertexBufferCpuAddress(nns::gfx::PrimitiveRenderer::VertexAttribute_Pos));
        // カラー
        nn::util::Float4* pColor = static_cast<nn::util::Float4*>(mesh.GetVertexBufferCpuAddress(nns::gfx::PrimitiveRenderer::VertexAttribute_Color));
        // 中心点を設定
        float color = (m_GradationDirection != GradationDirection::GradationDirection_None) ? 0.5f : 0.f;  // [0]側の色にかかる係数

        // 外周を設定
        double s, c;                                                        // Sin, Cos 値
        const auto borderSize = (weight > 0.f) ? 1.f : -1.f;
        auto rad = (m_AntiAliasingType == AntiAliasingType_Pseudo) ? nn::util::MakeFloat2(radius.x - borderSize, radius.y - borderSize) : radius ;  // 半径
        for (int i = 0; i < verticesCount; ++i)
        {
            if (i % drawDiv == 0) { curDeg = -angle + 180.f; }
            if (i == drawDiv)
            {
                if (m_AntiAliasingType == AntiAliasingType_Pseudo)
                {
                    rad = nn::util::MakeFloat2(rad.x + borderSize, rad.y + borderSize);
                }
                else
                {
                    rad = nn::util::MakeFloat2(rad.x + weight, rad.y + weight);
                }
            }
            if (i == drawDiv * 2) { rad = nn::util::MakeFloat2(rad.x + weight, rad.y + weight); }
            if (i == drawDiv * 3) { rad = nn::util::MakeFloat2(rad.x + borderSize, rad.y + borderSize); }

            s = nn::util::SinEst(nn::util::DegreeToRadian(curDeg));
            c = nn::util::CosEst(nn::util::DegreeToRadian(curDeg));
            pPos[i] = nn::util::MakeFloat3(-1.f + (2.f * (center.x + s * rad.x)) / GetGraphicsSystem().GetScreenWidth(), 1.f - (2.f * (center.y + c * rad.y)) / GetGraphicsSystem().GetScreenHeight(), 0.f);
            switch (m_GradationDirection)
            {
            case nns::hidfw::gfx::GraphicsDrawer::GradationDirection_None:
                color = 0.f;
                break;
            case nns::hidfw::gfx::GraphicsDrawer::GradationDirection_Left:
                color = 1.f - ((nn::util::SinEst(nn::util::DegreeToRadian(curDeg + angle)) + 1.f) / 2.f);
                break;
            case nns::hidfw::gfx::GraphicsDrawer::GradationDirection_Up:
                color = 1.f - ((nn::util::CosEst(nn::util::DegreeToRadian(curDeg + angle)) + 1.f) / 2.f);
                break;
            case nns::hidfw::gfx::GraphicsDrawer::GradationDirection_Right:
                color = (nn::util::SinEst(nn::util::DegreeToRadian(curDeg + angle)) + 1.f) / 2.f;
                break;
            case nns::hidfw::gfx::GraphicsDrawer::GradationDirection_Down:
                color = (nn::util::CosEst(nn::util::DegreeToRadian(curDeg + angle)) + 1.f) / 2.f;
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
            pColor[i].x = static_cast<float>(m_DrawColor[0].GetR()) * (1.f - color) / 255.f + static_cast<float>(m_DrawColor[1].GetR()) * color / 255.f;
            pColor[i].y = static_cast<float>(m_DrawColor[0].GetG()) * (1.f - color) / 255.f + static_cast<float>(m_DrawColor[1].GetG()) * color / 255.f;
            pColor[i].z = static_cast<float>(m_DrawColor[0].GetB()) * (1.f - color) / 255.f + static_cast<float>(m_DrawColor[1].GetB()) * color / 255.f;
            pColor[i].w = ((m_AntiAliasingType == AntiAliasingType_Pseudo) && (i < drawDiv || i >= drawDiv * 3) ) ? 0.f : static_cast<float>(m_DrawColor[0].GetA()) * (1.f - color) / 255.f + static_cast<float>(m_DrawColor[1].GetA()) * color / 255.f;
            curDeg += addDeg;
        }
        pRenderer->SetColor(m_DrawColor[0]);
        pRenderer->DrawUserMesh(&(GetGraphicsSystem().GetCommandBuffer()), nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleStrip, &mesh);
    }

    void GraphicsDrawer::Draw2DCircleFrame(const nn::util::Float2& center, const float radius, const float weight, const int32_t div, const float angle, const float length) NN_NOEXCEPT
    {
        Draw2DCircleFrame(center, nn::util::MakeFloat2(radius, radius), weight, div, angle, length);
    }

    void GraphicsDrawer::Draw2DTriangle(const nn::util::Float2& center, const float radius, const float angle) NN_NOEXCEPT
    {
        Draw2DCircle(center, radius, 3, angle);
    }

    void GraphicsDrawer::Draw2DRect(const nn::util::Float2& pos, const nn::util::Float2& size, const nn::util::Float2& uvPos, const nn::util::Float2& uvSize, nn::gfx::DescriptorSlot* pSlot) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pSlot);

        nns::gfx::PrimitiveRenderer::Renderer* pRenderer = &(GetGraphicsSystem().GetPrimitiveRenderer());
        nns::gfx::PrimitiveRenderer::PrimitiveMesh mesh;

        static const uint32_t index[6] =
        {
            0, 1, 2, 0, 2, 3
        };

        if (mesh.Initialize(pRenderer->GetGpuBuffer(), 4, 6,
            (nns::gfx::PrimitiveRenderer::VertexFormat)(nns::gfx::PrimitiveRenderer::VertexFormat_Pos | nns::gfx::PrimitiveRenderer::VertexFormat_Uv)) == false)
        {
            return;
        }

        uint32_t* pIndexData = mesh.GetIndexBufferCpuAddress();

        for (size_t i = 0; i < 6; ++i)
        {
            pIndexData[i] = index[i];
        }

        nn::util::Float3* pPos = static_cast<nn::util::Float3*>(mesh.GetVertexBufferCpuAddress(nns::gfx::PrimitiveRenderer::VertexAttribute_Pos));
        pPos[0] = nn::util::MakeFloat3(
            -1.f + (2.f * (pos.x)) / GetGraphicsSystem().GetScreenWidth(),
            1.f - (2.f * (pos.y)) / GetGraphicsSystem().GetScreenHeight(),
            0);
        pPos[1] = nn::util::MakeFloat3(
            -1.f + (2.f * (pos.x + size.x)) / GetGraphicsSystem().GetScreenWidth(),
            1.f - (2.f * (pos.y)) / GetGraphicsSystem().GetScreenHeight(),
            0);
        pPos[2] = nn::util::MakeFloat3(
            -1.f + (2.f * (pos.x + size.x)) / GetGraphicsSystem().GetScreenWidth(),
            1.f - (2.f * (pos.y + size.y)) / GetGraphicsSystem().GetScreenHeight(),
            0);
        pPos[3] = nn::util::MakeFloat3(
            -1.f + (2.f * (pos.x)) / GetGraphicsSystem().GetScreenWidth(),
            1.f - (2.f * (pos.y + size.y)) / GetGraphicsSystem().GetScreenHeight(),
            0);

        nn::util::Float2* pUv = static_cast<nn::util::Float2*>(mesh.GetVertexBufferCpuAddress(nns::gfx::PrimitiveRenderer::VertexAttribute_Uv));

        pUv[0] = nn::util::MakeFloat2(
            uvPos.x,
            1.f - uvPos.y);
        pUv[1] = nn::util::MakeFloat2(
            uvPos.x + uvSize.x,
            1.f - uvPos.y);
        pUv[2] = nn::util::MakeFloat2(
            uvPos.x + uvSize.x,
            1.f - (uvPos.y + uvSize.y));
        pUv[3] = nn::util::MakeFloat2(
            uvPos.x,
            1.f - (uvPos.y + uvSize.y));

        pRenderer->SetColor(GetColor());

        pRenderer->DrawUserMesh(
            &(GetGraphicsSystem().GetCommandBuffer()),
            nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleList,
            &mesh,
            *pSlot,
            m_SamplerDescriptor
        );
    }

    void GraphicsDrawer::Draw2DRect(const nn::util::Float2& pos, const nn::util::Float2& size) NN_NOEXCEPT
    {
        nns::gfx::PrimitiveRenderer::Renderer* pRenderer = &(GetGraphicsSystem().GetPrimitiveRenderer());
        nns::gfx::PrimitiveRenderer::PrimitiveMesh mesh;

        static const uint32_t index[6] =
        {
            0, 1, 2, 0, 2, 3
        };

        if (mesh.Initialize(pRenderer->GetGpuBuffer(), 4, 6,
            (nns::gfx::PrimitiveRenderer::VertexFormat)(nns::gfx::PrimitiveRenderer::VertexFormat_Pos | nns::gfx::PrimitiveRenderer::VertexFormat_Color)) == false)
        {
            return;
        }

        uint32_t* pIndexData = mesh.GetIndexBufferCpuAddress();

        for (size_t i = 0; i < 6; ++i)
        {
            pIndexData[i] = index[i];
        }


        nn::util::Float3* pPos = static_cast<nn::util::Float3*>(mesh.GetVertexBufferCpuAddress(nns::gfx::PrimitiveRenderer::VertexAttribute_Pos));
        pPos[0] = nn::util::MakeFloat3(
            -1.f + (2.f * (pos.x)) / GetGraphicsSystem().GetScreenWidth(),
            1.f - (2.f * (pos.y)) / GetGraphicsSystem().GetScreenHeight(),
            0);
        pPos[1] = nn::util::MakeFloat3(
            -1.f + (2.f * (pos.x + size.x)) / GetGraphicsSystem().GetScreenWidth(),
            1.f - (2.f * (pos.y)) / GetGraphicsSystem().GetScreenHeight(),
            0);
        pPos[2] = nn::util::MakeFloat3(
            -1.f + (2.f * (pos.x + size.x)) / GetGraphicsSystem().GetScreenWidth(),
            1.f - (2.f * (pos.y + size.y)) / GetGraphicsSystem().GetScreenHeight(),
            0);
        pPos[3] = nn::util::MakeFloat3(
            -1.f + (2.f * (pos.x)) / GetGraphicsSystem().GetScreenWidth(),
            1.f - (2.f * (pos.y + size.y)) / GetGraphicsSystem().GetScreenHeight(),
            0);

        nn::util::Float4* pColor = static_cast<nn::util::Float4*>(mesh.GetVertexBufferCpuAddress(nns::gfx::PrimitiveRenderer::VertexAttribute_Color));

        nn::util::Color4u8 primitiveColor;

        for (size_t idxFace = 0; idxFace < 4; idxFace++)
        {
            switch (m_GradationDirection)
            {
            case nns::hidfw::gfx::GraphicsDrawer::GradationDirection_None:
                primitiveColor = m_DrawColor[0];
                break;
            case nns::hidfw::gfx::GraphicsDrawer::GradationDirection_Left:
                primitiveColor = (1.f - ((pPos[idxFace].x + 1.f) * (GetGraphicsSystem().GetScreenWidth() / 2.f) - pos.x) / size.x > 0.5f) ? m_DrawColor[1] : m_DrawColor[0];
                break;
            case nns::hidfw::gfx::GraphicsDrawer::GradationDirection_Up:
                primitiveColor = (1.f - ((1.f - pPos[idxFace].y) * (GetGraphicsSystem().GetScreenHeight() / 2.f) - pos.y) / size.y > 0.5f) ? m_DrawColor[1] : m_DrawColor[0];
                break;
            case nns::hidfw::gfx::GraphicsDrawer::GradationDirection_Right:
                primitiveColor = ((((pPos[idxFace].x + 1.f) * (GetGraphicsSystem().GetScreenWidth() / 2.f)) - pos.x) / size.x > 0.5f) ? m_DrawColor[1] : m_DrawColor[0];
                break;
            case nns::hidfw::gfx::GraphicsDrawer::GradationDirection_Down:
                primitiveColor = (((1.f - pPos[idxFace].y) * (GetGraphicsSystem().GetScreenHeight() / 2.f) - pos.y) / size.y > 0.5f) ? m_DrawColor[1] : m_DrawColor[0];
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
            for (size_t i = 0; i < NN_ARRAY_SIZE(primitiveColor.v); ++i)
            {
                pColor[idxFace].v[i] = static_cast<float>(primitiveColor.v[i]) / 255.f;
            }
        }

        pRenderer->SetColor(nn::util::Color4u8(
            std::max(m_DrawColor[0].GetR(), m_DrawColor[1].GetR()),
            std::max(m_DrawColor[0].GetG(), m_DrawColor[1].GetG()),
            std::max(m_DrawColor[0].GetB(), m_DrawColor[1].GetB()),
            std::max(m_DrawColor[0].GetA(), m_DrawColor[1].GetA())));

        pRenderer->DrawUserMesh(
            &(GetGraphicsSystem().GetCommandBuffer()),
            nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleList,
            &mesh
        );
    }

    void GraphicsDrawer::Draw2DFrame(const nn::util::Float2& pos, const nn::util::Float2& size, const float borderSize) NN_NOEXCEPT
    {
        nns::gfx::PrimitiveRenderer::Renderer* pRenderer = &(GetGraphicsSystem().GetPrimitiveRenderer());
        nns::gfx::PrimitiveRenderer::PrimitiveMesh mesh;

        static const uint32_t index[4][6] =
        {
            {0, 1, 4, 4, 1, 5},
            {5, 1, 2, 2, 6, 5},
            {2, 3, 6, 6, 3, 7},
            {3, 0, 7, 7, 0, 4}
        };

        if (mesh.Initialize(pRenderer->GetGpuBuffer(), 8, 24,
            (nns::gfx::PrimitiveRenderer::VertexFormat)(nns::gfx::PrimitiveRenderer::VertexFormat_Pos | nns::gfx::PrimitiveRenderer::VertexFormat_Color)) == false)
        {
            return;
        }

        uint32_t* pIndexData = mesh.GetIndexBufferCpuAddress();

        for (size_t i = 0; i < 4; ++i)
        {
            for (size_t j = 0; j < 6; ++j)
            {
                pIndexData[i * 6 + j] = index[i][j];
            }
        }

        nn::util::Float3* pPos = static_cast<nn::util::Float3*>(mesh.GetVertexBufferCpuAddress(nns::gfx::PrimitiveRenderer::VertexAttribute_Pos));
        nn::util::Float4* pColor = static_cast<nn::util::Float4*>(mesh.GetVertexBufferCpuAddress(nns::gfx::PrimitiveRenderer::VertexAttribute_Color));

        pPos[0] = nn::util::MakeFloat3(
            -1.f + (2.f * (pos.x)) / GetGraphicsSystem().GetScreenWidth(),
            1.f - (2.f * (pos.y)) / GetGraphicsSystem().GetScreenHeight(),
            0);
        pPos[1] = nn::util::MakeFloat3(
            -1.f + (2.f * (pos.x + size.x)) / GetGraphicsSystem().GetScreenWidth(),
            1.f - (2.f * (pos.y)) / GetGraphicsSystem().GetScreenHeight(),
            0);
        pPos[2] = nn::util::MakeFloat3(
            -1.f + (2.f * (pos.x + size.x)) / GetGraphicsSystem().GetScreenWidth(),
            1.f - (2.f * (pos.y + size.y)) / GetGraphicsSystem().GetScreenHeight(),
            0);
        pPos[3] = nn::util::MakeFloat3(
            -1.f + (2.f * (pos.x)) / GetGraphicsSystem().GetScreenWidth(),
            1.f - (2.f * (pos.y + size.y)) / GetGraphicsSystem().GetScreenHeight(),
            0);

        pColor[0].x = static_cast<float>(m_DrawColor[0].GetR()) / 255.f;
        pColor[0].y = static_cast<float>(m_DrawColor[0].GetG()) / 255.f;
        pColor[0].z = static_cast<float>(m_DrawColor[0].GetB()) / 255.f;
        pColor[0].w = static_cast<float>(m_DrawColor[0].GetA()) / 255.f;
        pColor[1] = pColor[2] = pColor[3] = pColor[0];

        pPos[4] = nn::util::MakeFloat3(
            -1.f + (2.f * (pos.x - borderSize)) / GetGraphicsSystem().GetScreenWidth(),
            1.f - (2.f * (pos.y - borderSize)) / GetGraphicsSystem().GetScreenHeight(),
            0);
        pPos[5] = nn::util::MakeFloat3(
            -1.f + (2.f * (pos.x + size.x + borderSize)) / GetGraphicsSystem().GetScreenWidth(),
            1.f - (2.f * (pos.y - borderSize)) / GetGraphicsSystem().GetScreenHeight(),
            0);
        pPos[6] = nn::util::MakeFloat3(
            -1.f + (2.f * (pos.x + size.x + borderSize)) / GetGraphicsSystem().GetScreenWidth(),
            1.f - (2.f * (pos.y + size.y + borderSize)) / GetGraphicsSystem().GetScreenHeight(),
            0);
        pPos[7] = nn::util::MakeFloat3(
            -1.f + (2.f * (pos.x - borderSize)) / GetGraphicsSystem().GetScreenWidth(),
            1.f - (2.f * (pos.y + size.y + borderSize)) / GetGraphicsSystem().GetScreenHeight(),
            0);

        if (m_GradationDirection != GradationDirection::GradationDirection_None)
        {
            pColor[4].x = static_cast<float>(m_DrawColor[1].GetR()) / 255.f;
            pColor[4].y = static_cast<float>(m_DrawColor[1].GetG()) / 255.f;
            pColor[4].z = static_cast<float>(m_DrawColor[1].GetB()) / 255.f;
            pColor[4].w = static_cast<float>(m_DrawColor[1].GetA()) / 255.f;
            pColor[5] = pColor[6] = pColor[7] = pColor[4];
        }
        else
        {
            pColor[4] = pColor[5] = pColor[6] = pColor[7] = pColor[0];
        }

        pRenderer->SetColor(m_DrawColor[0]);
        pRenderer->DrawUserMesh(
            &(GetGraphicsSystem().GetCommandBuffer()),
            nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleList,
            &mesh
        );

    }

    void GraphicsDrawer::Draw2DRoundedRect(const nn::util::Float2& pos, const nn::util::Float2& size, const float round, const int32_t div) NN_NOEXCEPT
    {
        nns::gfx::PrimitiveRenderer::Renderer* pRenderer = &(GetGraphicsSystem().GetPrimitiveRenderer());
        nns::gfx::PrimitiveRenderer::PrimitiveMesh mesh;

        int32_t subDiv = std::max(4, (div / 4) * 4);                                // 最低でも上下左右の頂点が必要なため、頂点数を4の倍数に補正する
        size_t verticesCount = static_cast<size_t>(subDiv) + 8;                     // 頂点数
        int indexCount = subDiv * 3 + 18;                                           // インデックス数
        float circleRadius = (std::min(size.x, size.y) / 2.f) * round;              // 縦幅・横幅の最小/2を最大とする四隅の丸みの半径
        // 左上の角の円形の中心点
        nn::util::Float2 circleCenter = NN_UTIL_FLOAT_2_INITIALIZER(pos.x + circleRadius, pos.y + circleRadius);
        // 四隅の角の円形の中心点
        // それぞれ 0度、90度、180度、270度に当たる頂点を引き伸ばして角丸な矩形を作成します
        nn::util::Float2 circlePosition[4] =
        {
            circleCenter,
            NN_UTIL_FLOAT_2_INITIALIZER(circleCenter.x + size.x - circleRadius * 2.f, circleCenter.y),
            NN_UTIL_FLOAT_2_INITIALIZER(circleCenter.x + size.x - circleRadius * 2.f, circleCenter.y + size.y - circleRadius * 2.f),
            NN_UTIL_FLOAT_2_INITIALIZER(circleCenter.x                              , circleCenter.y + size.y - circleRadius * 2.f),
        };

        size_t circleVertexCount = NN_ARRAY_SIZE(circlePosition);
        size_t pointCount = subDiv / 4;
        if (mesh.Initialize(pRenderer->GetGpuBuffer(), verticesCount, indexCount,
            (nns::gfx::PrimitiveRenderer::VertexFormat)(nns::gfx::PrimitiveRenderer::VertexFormat_Pos | nns::gfx::PrimitiveRenderer::VertexFormat_Color)) == false)
        {
            return;
        }
        // インデックスバッファ
        uint32_t* pIndexData = mesh.GetIndexBufferCpuAddress();
        uint32_t index = NN_ARRAY_SIZE(circlePosition);
        uint32_t prevCenterIndex = 1;

        for (int i = 0; i < (indexCount - 6 * 3); ++i)
        {
            if ((i > 3) && ((i - 4) % 3 == 0))
            {
                pIndexData[i] = index - 1;
            }
            else
            {
                pIndexData[i] = i % 3 == 0 ? (i / (3 * pointCount) + 1) % circleVertexCount : index++;
                if (prevCenterIndex != (i / (3 * pointCount) + 1) % circleVertexCount)
                {
                    prevCenterIndex = (i / (3 * pointCount) + 1) % circleVertexCount;
                    index++;
                }
            }
        }
        // 矩形部
        //
        // ／■＼  <=  ■ の部分に当たるインデックスを設定します
        // ■■■      透明度が設定されている場合に正常に表示されるよう
        // ＼■／      それぞれの矩形が重ならないようにします
        //
        pIndexData[indexCount - 6 * 3 + 0] = pIndexData[indexCount - 6 * 3 - 1];
        pIndexData[indexCount - 6 * 3 + 1] = pIndexData[1];
        pIndexData[indexCount - 6 * 3 + 2] = (pointCount + 1) * 2 - 1 + circleVertexCount;
        pIndexData[indexCount - 6 * 3 + 3] = pIndexData[indexCount - 6 * 3 - 1];
        pIndexData[indexCount - 6 * 3 + 4] = pIndexData[indexCount - 6 * 3 + 2];
        pIndexData[indexCount - 6 * 3 + 5] = pIndexData[indexCount - 6 * 3 + 2] + 1;
        pIndexData[indexCount - 6 * 3 + 6] = 1;
        pIndexData[indexCount - 6 * 3 + 7] = pointCount + circleVertexCount;
        pIndexData[indexCount - 6 * 3 + 8] = pIndexData[indexCount - 6 * 3 + 7] + 1;
        pIndexData[indexCount - 6 * 3 + 9] = pIndexData[indexCount - 6 * 3 + 6];
        pIndexData[indexCount - 6 * 3 + 10] = pIndexData[indexCount - 6 * 3 + 8];
        pIndexData[indexCount - 6 * 3 + 11] = 2;
        pIndexData[indexCount - 6 * 3 + 12] = (pointCount + 1) * 3 + circleVertexCount; // l
        pIndexData[indexCount - 6 * 3 + 13] = 0;
        pIndexData[indexCount - 6 * 3 + 14] = 3;
        pIndexData[indexCount - 6 * 3 + 15] = pIndexData[indexCount - 6 * 3 + 12];
        pIndexData[indexCount - 6 * 3 + 16] = pIndexData[indexCount - 6 * 3 + 14];
        pIndexData[indexCount - 6 * 3 + 17] = pIndexData[indexCount - 6 * 3 + 12] - 1;

        // 頂点を設定
        float    curDeg = 180.f;
        float    addDeg = -(360.f / static_cast<float>(subDiv));
        nn::util::Float3* pPos = static_cast<nn::util::Float3*>(mesh.GetVertexBufferCpuAddress(nns::gfx::PrimitiveRenderer::VertexAttribute_Pos));
        nn::util::Float4* pColor = static_cast<nn::util::Float4*>(mesh.GetVertexBufferCpuAddress(nns::gfx::PrimitiveRenderer::VertexAttribute_Color));

        // 四隅の円形の頂点
        for (size_t i = 0; i < circleVertexCount; ++i)
        {
            pPos[i].x = -1.f + 2.f * circlePosition[i].x / GetGraphicsSystem().GetScreenWidth();
            pPos[i].y = 1.f - 2.f * circlePosition[i].y / GetGraphicsSystem().GetScreenHeight();
            pPos[i].z = 0.f;
        }

        int32_t targetCircleIndex = 1;
        index = circleVertexCount;

        for (size_t i = circleVertexCount; index < verticesCount; ++i)
        {
            pPos[index].x = -1.f + (2.f * (circlePosition[targetCircleIndex].x + nn::util::SinEst(nn::util::DegreeToRadian(curDeg)) * circleRadius)) / GetGraphicsSystem().GetScreenWidth();
            pPos[index].y = 1.f - (2.f * (circlePosition[targetCircleIndex].y + nn::util::CosEst(nn::util::DegreeToRadian(curDeg)) * circleRadius)) / GetGraphicsSystem().GetScreenHeight();
            pPos[index].z = 0.f;
            index++;
            if ((i != circleVertexCount) && ((i - circleVertexCount) % pointCount == 0) && index < verticesCount)
            {
                targetCircleIndex = (targetCircleIndex + 1) % 4;
                pPos[index].x = -1.f + (2.f * (circlePosition[targetCircleIndex].x + nn::util::SinEst(nn::util::DegreeToRadian(curDeg)) * circleRadius)) / GetGraphicsSystem().GetScreenWidth();
                pPos[index].y = 1.f - (2.f * (circlePosition[targetCircleIndex].y + nn::util::CosEst(nn::util::DegreeToRadian(curDeg)) * circleRadius)) / GetGraphicsSystem().GetScreenHeight();
                pPos[index].z = 0.f;
                index++;
            }
            curDeg += addDeg;
        }
        // カラー
        float curPos = 0.f;
        nn::util::Color4u8 primitiveColor;

        for (size_t idxFace = 0; idxFace < verticesCount; ++idxFace)
        {
            switch (m_GradationDirection)
            {
            case nns::hidfw::gfx::GraphicsDrawer::GradationDirection_None:
                curPos = 0.f;
                break;
            case nns::hidfw::gfx::GraphicsDrawer::GradationDirection_Left:
                curPos = 1.f - ((pPos[idxFace].x + 1.f) * (GetGraphicsSystem().GetScreenWidth() / 2.f) - pos.x) / size.x;
                break;
            case nns::hidfw::gfx::GraphicsDrawer::GradationDirection_Up:
                curPos = 1.f - ((1.f - pPos[idxFace].y) * (GetGraphicsSystem().GetScreenHeight() / 2.f) - pos.y) / size.y;
                break;
            case nns::hidfw::gfx::GraphicsDrawer::GradationDirection_Right:
                curPos = (((pPos[idxFace].x + 1.f) * (GetGraphicsSystem().GetScreenWidth() / 2.f)) - pos.x) / size.x;
                break;
            case nns::hidfw::gfx::GraphicsDrawer::GradationDirection_Down:
                curPos = ((1.f - pPos[idxFace].y) * (GetGraphicsSystem().GetScreenHeight() / 2.f) - pos.y) / size.y;
                break;
            default : NN_UNEXPECTED_DEFAULT;
            }
            // カラー
            primitiveColor = nn::util::Color4u8::Lerp(m_DrawColor[0], m_DrawColor[1], curPos);
            for (size_t i = 0; i < NN_ARRAY_SIZE(primitiveColor.v); ++i)
            {
                pColor[idxFace].v[i] = static_cast<float>(primitiveColor.v[i]) / 255.f;
            }
        }
        pRenderer->SetColor(nn::util::Color4u8(
            std::max(m_DrawColor[0].GetR(), m_DrawColor[1].GetR()),
            std::max(m_DrawColor[0].GetG(), m_DrawColor[1].GetG()),
            std::max(m_DrawColor[0].GetB(), m_DrawColor[1].GetB()),
            std::max(m_DrawColor[0].GetA(), m_DrawColor[1].GetA())));
        pRenderer->DrawUserMesh(
            &(GetGraphicsSystem().GetCommandBuffer()),
            nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleList,
            &mesh
        );
    } // NOLINT(impl/function_size)

    void GraphicsDrawer::Draw2DRoundedFrame(const nn::util::Float2& pos, const nn::util::Float2& size, const float round, const int32_t div, const float borderSize) NN_NOEXCEPT
    {
        nns::gfx::PrimitiveRenderer::Renderer* pRenderer = &(GetGraphicsSystem().GetPrimitiveRenderer());
        nns::gfx::PrimitiveRenderer::PrimitiveMesh mesh;

        int32_t subDiv          = std::max(4, (div / 4) * 4);           // 最低でも上下左右の頂点が必要なため、頂点数を4の倍数に補正する
        int     verticesCount   = (subDiv + 4) * 2;                     // 頂点数
        int     indexCount      = verticesCount * 3;                    // インデックス数

        float   circleRadius[2];                                        // [0] 内側 [1] 外側　※ borderSize が負の値の場合順番が逆転する

        circleRadius[0] = (borderSize > 0) ? (std::min(size.x, size.y) / 2.f) * round : (std::min(size.x, size.y) / 2.f) * round + borderSize;
        circleRadius[1] = (borderSize > 0) ? (std::min(size.x, size.y) / 2.f) * round + borderSize : (std::min(size.x, size.y) / 2.f) * round;
                                                                                // 左上の角の円形の中心点
        // 四隅の角の円形の中心点
        // それぞれ 0度、90度、180度、270度に当たる頂点を引き伸ばして角丸な矩形を作成します
        nn::util::Float2 circlePosition[4];
        nn::util::Float2 circleCenter = nn::util::MakeFloat2(pos.x + circleRadius[(borderSize < 0)], pos.y + circleRadius[(borderSize < 0)]);

        circlePosition[0] = circleCenter;
        circlePosition[1] = NN_UTIL_FLOAT_2_INITIALIZER(circleCenter.x + size.x - circleRadius[(borderSize < 0)] * 2.f, circleCenter.y);
        circlePosition[2] = NN_UTIL_FLOAT_2_INITIALIZER(circleCenter.x + size.x - circleRadius[(borderSize < 0)] * 2.f, circleCenter.y + size.y - circleRadius[(borderSize < 0)] * 2.f );
        circlePosition[3] = NN_UTIL_FLOAT_2_INITIALIZER(circleCenter.x, circleCenter.y + size.y - circleRadius[(borderSize < 0)] * 2.f );

        size_t pointCount = subDiv / 4;

        if (mesh.Initialize(pRenderer->GetGpuBuffer(), verticesCount, indexCount,
            (nns::gfx::PrimitiveRenderer::VertexFormat)(nns::gfx::PrimitiveRenderer::VertexFormat_Pos | nns::gfx::PrimitiveRenderer::VertexFormat_Color)) == false)
        {
            return;
        }

        // インデックスバッファ
        // 頂点は内側→外側の順で設定します
        // 頂点は左上(上部直線部の左側部分)から時計回りで配置されます
        uint32_t* pIndexData = mesh.GetIndexBufferCpuAddress();
        int index = static_cast<int>(NN_ARRAY_SIZE(circlePosition));

        for (int i = 0; i < indexCount / 6 - 1; ++i)
        {
            pIndexData[i * 6 + 0] = i;
            pIndexData[i * 6 + 1] = verticesCount / 2 + i + 1;
            pIndexData[i * 6 + 2] = i + 1;
            pIndexData[i * 6 + 3] = verticesCount / 2 + i;
            pIndexData[i * 6 + 4] = verticesCount / 2 + i + 1;
            pIndexData[i * 6 + 5] = i;
        }
        pIndexData[indexCount - 6] = verticesCount / 2 - 1;
        pIndexData[indexCount - 5] = verticesCount / 2;
        pIndexData[indexCount - 4] = 0;
        pIndexData[indexCount - 3] = verticesCount - 1;
        pIndexData[indexCount - 2] = verticesCount / 2;
        pIndexData[indexCount - 1] = verticesCount / 2 - 1;

        // 頂点を設定
        // 左上の頂点から時計回りに設定していきます
        // 内週 : 0 .. (verticesCount / 2) - 1
        // 外周 : verticesCount / 2) .. verticesCount - 1
        float    curDeg = 180.f;
        float    addDeg = -(360.f / static_cast<float>(subDiv));
        nn::util::Float3* pPos = static_cast<nn::util::Float3*>(mesh.GetVertexBufferCpuAddress(nns::gfx::PrimitiveRenderer::VertexAttribute_Pos));
        nn::util::Float4* pColor = static_cast<nn::util::Float4*>(mesh.GetVertexBufferCpuAddress(nns::gfx::PrimitiveRenderer::VertexAttribute_Color));

        int32_t targetCircleIndex = 0;
        index = 0;
        double s, c;        // Sin, Cos 値

        for (int i = 0; index < verticesCount / 2; ++i)
        {
            s = nn::util::SinEst(nn::util::DegreeToRadian(curDeg));
            c = nn::util::CosEst(nn::util::DegreeToRadian(curDeg));

            pPos[index].x = -1.f + (2.f * (circlePosition[targetCircleIndex].x + s * circleRadius[0])) / GetGraphicsSystem().GetScreenWidth();
            pPos[index].y = 1.f - (2.f * (circlePosition[targetCircleIndex].y + c * circleRadius[0])) / GetGraphicsSystem().GetScreenHeight();
            pPos[index].z = 0.f;

            pPos[index + (verticesCount / 2)].x = -1.f + (2.f * (circlePosition[targetCircleIndex].x + s * circleRadius[1])) / GetGraphicsSystem().GetScreenWidth();
            pPos[index + (verticesCount / 2)].y = 1.f - (2.f * (circlePosition[targetCircleIndex].y + c * circleRadius[1])) / GetGraphicsSystem().GetScreenHeight();
            pPos[index + (verticesCount / 2)].z = 0.f;

            index++;
            if ((i % pointCount) == 0 && (index < (verticesCount / 2)))
            {
                targetCircleIndex = (targetCircleIndex + 1) % 4;
                pPos[index].x = -1.f + (2.f * (circlePosition[targetCircleIndex].x + s * circleRadius[0])) / GetGraphicsSystem().GetScreenWidth();
                pPos[index].y = 1.f - (2.f * (circlePosition[targetCircleIndex].y + c * circleRadius[0])) / GetGraphicsSystem().GetScreenHeight();
                pPos[index].z = 0.f;

                pPos[index + (verticesCount / 2)].x = -1.f + (2.f * (circlePosition[targetCircleIndex].x + s * circleRadius[1])) / GetGraphicsSystem().GetScreenWidth();
                pPos[index + (verticesCount / 2)].y = 1.f - (2.f * (circlePosition[targetCircleIndex].y + c * circleRadius[1])) / GetGraphicsSystem().GetScreenHeight();
                pPos[index + (verticesCount / 2)].z = 0.f;
                index++;
            }
            curDeg += addDeg;
        }
        if (m_GradationDirection != GradationDirection::GradationDirection_None)
        {
            for (int i = 0; i < verticesCount; ++i)
            {

                pColor[i].x = (i < verticesCount / 2) ? static_cast<float>(m_DrawColor[((borderSize < 0))].GetR()) / 255.f : static_cast<float>(m_DrawColor[((borderSize > 0))].GetR()) / 255.f;
                pColor[i].y = (i < verticesCount / 2) ? static_cast<float>(m_DrawColor[((borderSize < 0))].GetG()) / 255.f : static_cast<float>(m_DrawColor[((borderSize > 0))].GetG()) / 255.f;
                pColor[i].z = (i < verticesCount / 2) ? static_cast<float>(m_DrawColor[((borderSize < 0))].GetB()) / 255.f : static_cast<float>(m_DrawColor[((borderSize > 0))].GetB()) / 255.f;
                pColor[i].w = (i < verticesCount / 2) ? static_cast<float>(m_DrawColor[((borderSize < 0))].GetA()) / 255.f : static_cast<float>(m_DrawColor[((borderSize > 0))].GetA()) / 255.f;
            }
        }
        else
        {
            for (int i = 0; i < verticesCount; ++i)
            {
                pColor[i].x = static_cast<float>(m_DrawColor[0].GetR()) / 255.f;
                pColor[i].y = static_cast<float>(m_DrawColor[0].GetG()) / 255.f;
                pColor[i].z = static_cast<float>(m_DrawColor[0].GetB()) / 255.f;
                pColor[i].w = static_cast<float>(m_DrawColor[0].GetA()) / 255.f;
            }
        }
        pRenderer->SetColor(nn::util::Color4u8(
            std::max(m_DrawColor[0].GetR(), m_DrawColor[1].GetR()),
            std::max(m_DrawColor[0].GetG(), m_DrawColor[1].GetG()),
            std::max(m_DrawColor[0].GetB(), m_DrawColor[1].GetB()),
            std::max(m_DrawColor[0].GetA(), m_DrawColor[1].GetA())));

        pRenderer->DrawUserMesh(
            &(GetGraphicsSystem().GetCommandBuffer()),
            nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleList,
            &mesh
        );
        pRenderer->SetColor(m_DrawColor[0]);
    }

    void GraphicsDrawer::Draw2DLine(const nn::util::Float2& begin, const nn::util::Float2& end) NN_NOEXCEPT
    {
        Draw2DLine(2, begin, end);
    }

    void GraphicsDrawer::Draw2DLine(int num, const nn::util::Float2& begin, const nn::util::Float2 next, ...) NN_NOEXCEPT
    {
        NN_ASSERT_GREATER_EQUAL(num, 2);

        nn::util::Float2* pointList = new nn::util::Float2[num];

        pointList[0] = begin;
        pointList[1] = next;

        // 第三頂点以降が入力された場合
        if (num > 2)
        {
            va_list vaList;
            va_start(vaList, next);
            for (int i = 2; i < num; ++i)
            {
                pointList[i] = va_arg(vaList, nn::util::Float2);
            }
            va_end(vaList);
        }

        Draw2DLine(num, pointList);

        delete[] pointList;
    }

    void GraphicsDrawer::Draw2DLine(int num, const nn::util::Float2* pointList) NN_NOEXCEPT
    {
        nns::gfx::PrimitiveRenderer::Renderer* pRenderer = &(GetGraphicsSystem().GetPrimitiveRenderer());
        nns::gfx::PrimitiveRenderer::PrimitiveMesh mesh;
        if ((num < 2) || (pointList == nullptr)) { return; }

        // メッシュの初期化
        if (mesh.Initialize(pRenderer->GetGpuBuffer(), num, num,
            (nns::gfx::PrimitiveRenderer::VertexFormat)(nns::gfx::PrimitiveRenderer::VertexFormat_Pos | nns::gfx::PrimitiveRenderer::VertexFormat_Color)) == false)
        {
            return;
        }

        // インデックスの設定
        uint32_t* pIndexData = mesh.GetIndexBufferCpuAddress();
        for (int i = 0; i < num; ++i)
        {
            pIndexData[i] = i;
        }

        // 頂点を設定
        nn::util::Float3* pPos = static_cast<nn::util::Float3*>(mesh.GetVertexBufferCpuAddress(nns::gfx::PrimitiveRenderer::VertexAttribute_Pos));

        // 頂点座標を設定
        for (int i = 0; i < num; ++i)
        {
            pPos[i] = nn::util::MakeFloat3(
                -1.f + (2.f * pointList[i].x) / GetGraphicsSystem().GetScreenWidth(),
                1.f - (2.f * pointList[i].y) / GetGraphicsSystem().GetScreenHeight(),
                0.f);
        }

        // 色情報を設定
        nn::util::Float4* pColor = static_cast<nn::util::Float4*>(mesh.GetVertexBufferCpuAddress(nns::gfx::PrimitiveRenderer::VertexAttribute_Color));

        for (int i = 0; i < num; ++i)
        {
            nn::util::Color4f(nn::util::Color4u8::Lerp(m_DrawColor[0], m_DrawColor[1], (static_cast<float>(i) / static_cast<float>(num)))).Get
            (
                &pColor[i].x, &pColor[i].y, &pColor[i].z, &pColor[i].w
            );
        }

        pRenderer->SetColor(nn::util::Color4u8::White());
        pRenderer->DrawUserMesh(
            &(GetGraphicsSystem().GetCommandBuffer()),
            nn::gfx::PrimitiveTopology::PrimitiveTopology_LineStrip,
            &mesh);
    }

    void GraphicsDrawer::DrawCube(const nn::util::Vector3fType& center, const nn::util::Vector3fType& scale, const nn::util::Quaternion& quaternion) NN_NOEXCEPT
    {
        nns::gfx::PrimitiveRenderer::Renderer* pRenderer = &(GetGraphicsSystem().GetPrimitiveRenderer());

        nn::util::Matrix4x3fType viewMatrix;
        nn::util::Vector3fType camPos = { 0, 0.f, 10.f };
        nn::util::Vector3fType camTarget = { 0.f, 0.f, 0.f };
        nn::util::Vector3fType camUp = { 0.f, 1.f, 0.f };
        nn::util::MatrixLookAtRightHanded(&viewMatrix, camPos, camTarget, camUp);
        pRenderer->SetViewMatrix(&viewMatrix);

        // プロジェクションを初期化
        nn::util::Matrix4x4fType projectionMatrix;
        nn::util::MatrixIdentity(&projectionMatrix);
        const float fovy = nn::util::FloatPi / 3.f;
        const float aspect = 1280.f / 720.f;
        nn::util::MatrixPerspectiveFieldOfViewRightHanded(&projectionMatrix, fovy, aspect, 0.1f, 1000.f);
        pRenderer->SetProjectionMatrix(&projectionMatrix);

        nn::util::Vector3f vecZero;
        nn::util::VectorZero(&vecZero);

        nn::util::Matrix4x3f modelMatrix;
        nn::util::MatrixIdentity(&modelMatrix);

        nn::util::MatrixSetTranslate(&modelMatrix, center);
        nn::util::MatrixSetRotate(&modelMatrix, quaternion);
        nn::util::MatrixSetScaleRotate(&modelMatrix, scale, quaternion);
        //nn::util::MatrixSetScale(&modelMatrix, scale);

        pRenderer->SetModelMatrix(&modelMatrix);
        pRenderer->SetLineWidth(4.f);

        pRenderer->SetColor(m_DrawColor[0]);
        pRenderer->SetDepthStencilState(&(GetGraphicsSystem().GetCommandBuffer()), nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthWriteTest);
        pRenderer->DrawCube(
            &(GetGraphicsSystem().GetCommandBuffer()),
            nns::gfx::PrimitiveRenderer::Surface_Normal,
            nn::util::MakeVector3fType(0.f, 0.f, 0.f),
            nn::util::MakeVector3fType(1.f, 1.f, 1.f));

        pRenderer->SetDepthStencilState(&(GetGraphicsSystem().GetCommandBuffer()), nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest);
        nn::util::MatrixIdentity(&modelMatrix);
        pRenderer->SetModelMatrix(&modelMatrix);

        pRenderer->SetLineWidth(1.f);

        nn::util::MatrixIdentity(&projectionMatrix);
        pRenderer->SetProjectionMatrix(&projectionMatrix);

        nn::util::MatrixIdentity(&viewMatrix);
        pRenderer->SetViewMatrix(&viewMatrix);
    }

    void GraphicsDrawer::Draw2DScreenShot(ScreenShotSlot slot, const nn::util::Float2& pos, const nn::util::Float2& size, const nn::util::Float4& texcoord) NN_NOEXCEPT
    {
        NN_UNUSED(texcoord);
        // Y座標が反転しているため上下を反転して描画
        GetPrimitiveRenderer().SetColor(m_DrawColor[0]);
        GetPrimitiveRenderer().Draw2DRect(&(GetGraphicsSystem().GetCommandBuffer()), pos.x, pos.y + size.y, size.x, -size.y, m_ScreenShotDescriptor[slot], m_SamplerDescriptor);
    }

    void GraphicsDrawer::Draw2DPrevScreen(const nn::util::Float2& pos, const nn::util::Float2& size, const nn::util::Float4& texcoord) NN_NOEXCEPT
    {
        // Y座標が反転しているため上下を反転して描画
        GetPrimitiveRenderer().SetColor(m_DrawColor[0]);
        const auto uvPos = nn::util::MakeFloat2(texcoord.x, texcoord.y);
        const auto uvSize = nn::util::MakeFloat2(texcoord.z - texcoord.x, texcoord.w - texcoord.y);
        Draw2DRect(pos, size, uvPos, uvSize, &m_ScanScreenDescriptor);
        //GetPrimitiveRenderer().Draw2DRect(&(GetGraphicsSystem().GetCommandBuffer()), pos.x, pos.y + size.y, size.x, -size.y, m_ScanScreenDescriptor, m_SamplerDescriptor);
    }

    GraphicsDrawer::GraphicsDrawer() NN_NOEXCEPT
    {
        m_State = State_Initialized;
        m_ClearColor = nn::util::Color4u8::Black();
        for (size_t i = 0; i < NN_ARRAY_SIZE(m_DrawColor); ++i)
        {
            m_DrawColor[i] = nn::util::Color4u8::White();
        }
        m_GradationDirection = GradationDirection::GradationDirection_None;
        m_FrameRate = 60;
        m_isEnableScreenShotJob = false;
        m_ScreenShotScale = NN_UTIL_FLOAT_2_INITIALIZER(1.f, 1.f);
    }

    void GraphicsDrawer::InitializeBuffer() NN_NOEXCEPT
    {
        {
            nn::gfx::Texture::InfoType info;
            info.SetDefault();
            info.SetWidth(GetGraphicsSystem().GetScreenWidth());
            info.SetHeight(GetGraphicsSystem().GetScreenHeight());
            info.SetGpuAccessFlags(nn::gfx::GpuAccess_ColorBuffer);
            info.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
            info.SetImageFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm);
            info.SetMipCount(1);
            info.SetDepth(1);

            GetGraphicsSystem().AllocateTexture(&m_ScanBuffer, &info);
            for (int i = 0; i < ScreenShotSlot_Slot_Num; ++i)
            {
                GetGraphicsSystem().AllocateTexture(&m_ScreenShotBuffer[i], &info);
            }
            GetGraphicsSystem().AllocateTexture(&m_WorkBuffer, &info);
            //info.SetGpuAccessFlags(nn::gfx::GpuAccess_ColorBuffer);
            //GetGraphicsSystem().AllocateTexture(&m_TextBuffer, &info);

            info.SetWidth(8192);
            info.SetHeight(8192);
            info.SetGpuAccessFlags(nn::gfx::GpuAccess_Texture);
            info.SetImageFormat(nn::gfx::ImageFormat_R8_Unorm);
            GetGraphicsSystem().AllocateTexture(&m_TextBuffer, &info);
        }
        {
            nn::gfx::ColorTargetView::InfoType info;
            info.SetDefault();
            info.SetImageDimension(nn::gfx::ImageDimension_2d);
            info.SetImageFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm);
            info.SetTexturePtr(&m_ScanBuffer);
            m_ScanBufferColorTargetView.Initialize(&(GetGraphicsSystem().GetDevice()), info);

            info.SetTexturePtr(&m_WorkBuffer);
            m_WorkBufferColorTargetView.Initialize(&(GetGraphicsSystem().GetDevice()), info);

            info.SetTexturePtr(&m_TextBuffer);
            m_TextBufferColorTargetView.Initialize(&(GetGraphicsSystem().GetDevice()), info);
        }
        {
            nn::gfx::TextureView::InfoType info;
            info.SetDefault();
            info.SetImageDimension(nn::gfx::ImageDimension_2d);
            info.SetImageFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm);
            info.SetTexturePtr(&m_ScanBuffer);

            m_ScanBufferTextureView.Initialize(&(GetGraphicsSystem().GetDevice()), info);

            // フォント用のテクスチャ
            //info.SetTexturePtr(&m_TextBuffer);
            //m_TextBufferTextureView.Initialize(&(GetGraphicsSystem().GetDevice()), info);

            info.SetImageFormat(nn::gfx::ImageFormat_R8_Unorm);
            info.SetTexturePtr(&m_TextBuffer);
            m_FontTextureView.Initialize(&(GetGraphicsSystem().GetDevice()), info);

        }
        GetGraphicsSystem().RegisterTextureViewSlot(&m_ScanScreenDescriptor, m_ScanBufferTextureView);
        //GetGraphicsSystem().RegisterTextureViewSlot(&m_TextureDescriptor, m_TextBufferTextureView);
        GetGraphicsSystem().RegisterTextureViewSlot(&m_FontTextureDescriptor, m_FontTextureView);
        {
            nn::gfx::TextureView::InfoType info;
            info.SetDefault();
            info.SetImageDimension(nn::gfx::ImageDimension_2d);
            info.SetImageFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm);
            for (int i = 0; i < ScreenShotSlot_Slot_Num; ++i)
            {
                info.SetTexturePtr(&m_ScreenShotBuffer[i]);
                m_ScreenShotTextureView[i].Initialize(&(GetGraphicsSystem().GetDevice()), info);
                GetGraphicsSystem().RegisterTextureViewSlot(&m_ScreenShotDescriptor[i], m_ScreenShotTextureView[i]);
            }
        }
        {
            nn::gfx::SamplerInfo info;
            info.SetDefault();
            m_Sampler.Initialize(&(GetGraphicsSystem().GetDevice()), info);
            GetGraphicsSystem().RegisterSamplerSlot(&m_SamplerDescriptor, m_Sampler);
        }
    }

    void GraphicsDrawer::FinalizeBuffer() NN_NOEXCEPT
    {
        m_Sampler.Finalize(&(GetGraphicsSystem().GetDevice()));
        GetGraphicsSystem().UnregisterTextureViewSlot(&m_ScanScreenDescriptor, m_ScanBufferTextureView);
        for (int i = 0; i < ScreenShotSlot_Slot_Num; ++i)
        {
            GetGraphicsSystem().UnregisterTextureViewSlot(&m_ScreenShotDescriptor[i], m_ScreenShotTextureView[i]);
        }
        GetGraphicsSystem().FreeTexture(&m_TextBuffer);
        GetGraphicsSystem().FreeTexture(&m_ScanBuffer);
        GetGraphicsSystem().FreeTexture(&m_WorkBuffer);
        m_ScanBufferColorTargetView.Finalize(&(GetGraphicsSystem().GetDevice()));
        m_WorkBufferColorTargetView.Finalize(&(GetGraphicsSystem().GetDevice()));
        m_TextBufferTextureView.Finalize(&(GetGraphicsSystem().GetDevice()));
        m_ScanBufferTextureView.Finalize(&(GetGraphicsSystem().GetDevice()));
    }
}}}
