﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstring>
#include <cstdlib>

// NintendoSDK のヘッダファイルをインクルードする前に、NN_GFX_UTIL_DEBUGFONT_USE_DEFAULT_LOCALE_CHARSET マクロを
// 定義することで、DebugFontWriter::Print() の入力文字コードを Windows のロケールのデフォルト
// (日本語の場合、CP932)に変更できます。
#define NN_GFX_UTIL_DEBUGFONT_USE_DEFAULT_LOCALE_CHARSET

#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/os.h>
#include <nn/init.h>
#include <nn/vi.h>
#include <nn/gfx.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nn/hws/hws_Message.h>

#include <nn/nn_Log.h>

#include <nn/dd.h>
#include <nn/gc/detail/gc_Types.h>
#include <nn/fs.h>
#include <nn/fs/fs_GameCard.h>
#include <nn/fs/fs_GameCardForInspection.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/fs/fs_MmcPrivate.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fssystem/fs_IStorage.h>

#include <nn/hid.h>
#include <nn/hid/hid_TouchScreen.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_FormatString.h>

#include "GraphicBase.h"

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
#include <nv/nv_MemoryManagement.h>
#endif

const size_t ApplicationHeapSize     = 128 * 1024 * 1024;
const size_t MaxFwVersion            = 63;
const size_t FwSize                  = 30 * 1024;

nn::os::ThreadType g_Thread;
static const int ThreadPriority = 1;
static const size_t ThreadStackSize = 64 * 1024;
NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStack[ ThreadStackSize ];
char g_ThreadName[] = "GcTest";
nn::os::EventType g_ThreadEndEvent;

FontSystem* g_pFontSystem;
GraphicsSystem* g_pGraphicsSystem;

struct TestInfo
{
    bool isTestFinished;
    bool isFileExist;
    bool isSdCardInserted;
    nn::Result result;
    size_t currentIndex;
    nn::gc::RmaInformation rmaInfo;
};

TestInfo g_TestInfo;


char g_CalKey[] = {
        0xd4,0xd0,0x7f,0xb6,0xe0,0xb8,0x63,0x82,0x26,0xfd,0x76,0x88,0x7a,0x86,0x8f,0x03
};

char g_RmaFwBuffer[FwSize];
nn::hid::TouchScreenState<1> g_PrevTouchScreenState;
nn::hid::TouchScreenState<1> g_CurTouchScreenState;

const nn::util::Unorm8x4 Color::White  = { { 255, 255, 255, 255 } };
const nn::util::Unorm8x4 Color::Black  = { {  10,  10,  10, 255 } };
const nn::util::Unorm8x4 Color::Gray   = { { 190, 190, 190, 255 } };
//const nn::util::Unorm8x4 Color::Green  = { { 91, 162,   117, 255 } };
const nn::util::Unorm8x4 Color::Green  = { { 0, 164,   180, 255 } };
const nn::util::Unorm8x4 Color::Orange = { { 255, 153,   0, 255 } };
const nn::util::Unorm8x4 Color::Red    = { { 255,   0,   75, 255 } };
const nn::util::Unorm8x4 Color::Blue   = { { 137, 189, 222, 255 } };

namespace
{
    #if defined(NN_BUILD_TARGET_PLATFORM_NX)
    void* NvAllocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
    {
        NN_UNUSED(userPtr);
        return aligned_alloc(alignment, size);
    }

    void NvFree(void* addr, void* userPtr) NN_NOEXCEPT
    {
        NN_UNUSED(userPtr);
        free(addr);
    }

    void* NvReallocate(void* addr, size_t newSize, void* userPtr) NN_NOEXCEPT
    {
        NN_UNUSED(userPtr);
        return realloc(addr, newSize);
    }
    #endif

    void PrintArray(const char* buffer, const size_t bufferLength) NN_NOEXCEPT
    {
        NN_LOG("    ");
        for(size_t i=0; i<bufferLength; i++)
        {
            if(i!=0 && i % 16 == 0)
            {
                NN_LOG("\n    ");
            }

            NN_LOG("%02X ", buffer[i]);
        }
        NN_LOG("\n");
    }

    bool IsTouchTrigger()
    {
        if(g_PrevTouchScreenState.touches[0].x == 0 && g_PrevTouchScreenState.touches[0].y == 0 )
        {
            return true;
        }
        //    NN_LOG("prev X = %d\n", g_PrevTouchScreenState.touches[0].x);
        //    NN_LOG("prev Y = %d\n", g_PrevTouchScreenState.touches[0].y);
        return false;
    }
}


//------------------------------------------------------------------------------
// 画面描画用スレッド
//------------------------------------------------------------------------------
void SetOneLine(char* pLabel, size_t startY, uint8_t* pValue, size_t valueNum)
{
    const size_t WidthBase  = 50;

    nn::gfx::util::DebugFontTextWriter& textWriter = g_pFontSystem->GetDebugFontTextWriter();
    float startX = 300;
    textWriter.SetTextColor(Color::White);
    textWriter.SetScale(2, 2);
    textWriter.SetCursor(0, startY);

    textWriter.Print(pLabel);
    if(valueNum <= 16)
    {
        for(size_t i = 0; i < valueNum ;i++)
        {
            textWriter.SetCursor( startX + i * WidthBase, startY);
            textWriter.Print("%02X ", *(pValue + i));
        }
    }
    else
    {
        for(size_t i = 0; i < valueNum / 16 ;i++)
        {
            for(size_t j = 0; j < 16 ;j++)
            {
                textWriter.SetCursor( startX + j * WidthBase, startY + i * 40);
                textWriter.Print("%02X ", *(pValue + i * 16 + j));
            }
        }
    }
}

void SetRmaInfo()
{
    const size_t StartPosY  = 160;
    const size_t HeightBase = 40;
    size_t line = 0;
    char buf[64];
    nn::util::Strlcpy(buf, "Pair No : ", sizeof(buf));
    SetOneLine(buf, StartPosY + HeightBase * line++, g_TestInfo.rmaInfo.pairNo, 8);
    nn::util::Strlcpy(buf, "Otp Forbid Flag : ", sizeof(buf));
    SetOneLine(buf, StartPosY + HeightBase * line++, &g_TestInfo.rmaInfo.otpTestForbidFlag, 1);
    nn::util::Strlcpy(buf, "Hw Version : ", sizeof(buf));
    SetOneLine(buf, StartPosY + HeightBase * line++, g_TestInfo.rmaInfo.hardwareVersion, 4);
    nn::util::Strlcpy(buf, "Flag and Version : ", sizeof(buf));
    SetOneLine(buf, StartPosY + HeightBase * line++, g_TestInfo.rmaInfo.flagAndVersion, 8);
    nn::util::Strlcpy(buf, "OTP1 : ", sizeof(buf));
    SetOneLine(buf, StartPosY + HeightBase * line++, g_TestInfo.rmaInfo.otpTestArea1, 2);
    nn::util::Strlcpy(buf, "OTP2 : ", sizeof(buf));
    SetOneLine(buf, StartPosY + HeightBase * line++, g_TestInfo.rmaInfo.otpTestArea2, 16);
    nn::util::Strlcpy(buf, "OTP3 : ", sizeof(buf));
    SetOneLine(buf, StartPosY + HeightBase * line++, g_TestInfo.rmaInfo.otpTestArea3, 32);
}

void SetLine(float startX, float startY, float endX, float endY)
{
    nn::gfx::util::DebugFontTextWriter& textWriter = g_pFontSystem->GetDebugFontTextWriter();
    char buffer = '*';
    textWriter.SetScale(1.2, 1.2);

    float dx = endX - startX;
    float dy = endY - startY;

    if(dx == 0 && dy == 0)
    {
        textWriter.SetCursor(startX, startY);
        textWriter.Print(&buffer);
    }
    else
    {
        // 描画する点の個数を決定します
        float lineLength = 1.0f / nn::util::Rsqrt(dx * dx + dy * dy);
        int pointNum = lineLength / 2.5;

        for(int i = 0; i <= pointNum; i++)
        {
            float x = (startX * (pointNum - i) + endX * i) / pointNum;
            float y = (startY * (pointNum - i) + endY * i) / pointNum;
            textWriter.SetCursor(x, y);
            textWriter.Print(&buffer);
        }
    }
}

bool SetTouchableSquareAndDetectTouch(float x, float y, float width, float height)
{
    nn::gfx::util::DebugFontTextWriter& textWriter = g_pFontSystem->GetDebugFontTextWriter();
    textWriter.SetTextColor(Color::Green);
    SetLine(x, y, x + width, y);
    SetLine(x, y, x, y + height);
    SetLine(x + width, y, x + width, y + height);
    SetLine(x, y + height, x + width, y + height);
    if(g_CurTouchScreenState.touches[0].x > x &&
            g_CurTouchScreenState.touches[0].x < x + width &&
            g_CurTouchScreenState.touches[0].y > y&&
            g_CurTouchScreenState.touches[0].y < y + height)
    {
        return true;
    }
    return false;
}

bool SetButtonAndBuffer(float x, float y, float width, float height, char* buffer)
{
    nn::gfx::util::DebugFontTextWriter& textWriter = g_pFontSystem->GetDebugFontTextWriter();
    auto color = textWriter.GetTextColor();
    bool ret = false;
    ret = SetTouchableSquareAndDetectTouch(x, y, width, height);
    textWriter.SetScale(2.4, 2.4);
    float textWidth  = textWriter.CalculateStringWidth(buffer) - 10;
    float textHeigth = textWriter.CalculateStringHeight(buffer) - 20;
    textWriter.SetTextColor(color);
    textWriter.SetCursor(x + (width - textWidth) / 2, y + (height - textHeigth) / 2);
    textWriter.Print(buffer);
    return ret;
}

bool SetText()
{
    nn::gfx::util::DebugFontTextWriter& textWriter = g_pFontSystem->GetDebugFontTextWriter();
    textWriter.SetTextColor(Color::White);
    textWriter.SetScale(2.4, 2.4);
    textWriter.SetCursor(0, 80);
    textWriter.Print("FwVersion : %d\n", g_TestInfo.currentIndex);
    textWriter.SetCursor(0, 0);
    if(g_TestInfo.isTestFinished)
    {
        textWriter.Print("State : Finished\n");
        SetRmaInfo();
        char buffer[64];
        nn::util::Strlcpy(buffer, "Save", sizeof(buffer));
        if(IsTouchTrigger() && SetButtonAndBuffer(1000, 30, 250, 100, buffer))
        {
            return false;
        }
    }
    else if(!g_TestInfo.isSdCardInserted)
    {
        textWriter.Print("State : Waiting for inserting SdCard .\n");
    }
    else
    {
        textWriter.Print("State : Waiting FW Reboot.\n");
    }
    return true;
}
void RunThreadFunction(void* pThreadNum) NN_NOEXCEPT
        {
    while(1)
    {
        g_PrevTouchScreenState = g_CurTouchScreenState;
        nn::hid::GetTouchScreenState(&g_CurTouchScreenState);
        if(!SetText())
        {
            break;
        }
        g_pGraphicsSystem->BeginDraw();
        g_pFontSystem->Draw();
        g_pGraphicsSystem->EndDraw();
        g_pGraphicsSystem->Synchronize(nn::TimeSpan::FromNanoSeconds(1000 * 1000 * 1000 / FrameRate));
        nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(10));
    }
    nn::os::SignalEvent(&g_ThreadEndEvent);
        }

//------------------------------------------------------------------------------
//  スレッド関連の処理
//------------------------------------------------------------------------------
void InitializeThread()
{
    nn::os::InitializeEvent(&g_ThreadEndEvent, false, nn::os::EventClearMode_AutoClear);
    // 待ちイベントの初期設定
    nn::Result result = nn::os::CreateThread( &g_Thread, RunThreadFunction, &g_Thread, g_ThreadStack, ThreadStackSize, ThreadPriority );
    if ( !result.IsSuccess() )
    {
        NN_ABORT("failed to create a thread\n");
    }
    nn::os::SetThreadNamePointer(&g_Thread, g_ThreadName);
    nn::os::StartThread( &g_Thread );
}

void FinalizeThread()
{
    // スレッドの終了
    nn::os::WaitThread( &g_Thread );
    nn::os::DestroyThread( &g_Thread );
    nn::os::FinalizeEvent(&g_ThreadEndEvent);
}

//------------------------------------------------------------------------------
//  SD カードから FW を読み込み
//------------------------------------------------------------------------------
void GetFilePath(char* outBuffer, size_t outBufferSize, size_t index)
{
    nn::util::SNPrintf(outBuffer, outBufferSize, "sdcard:/rma%zu.bin", index);
    if(index > MaxFwVersion)
    {
        NN_ABORT("FwVersion is too large. Max Fw Version is %zu\n", MaxFwVersion);
    }
    NN_LOG("   FileName : %s\n",outBuffer);
}


nn::Result ReadFwFromSd(char* outBuffer, size_t outBufferLength, size_t index)
{
    nn::fs::FileHandle fileHandle;
    char filePath[64];
    GetFilePath(filePath, sizeof(filePath), index);
    NN_RESULT_DO(nn::fs::OpenFile(&fileHandle, filePath, nn::fs::OpenMode_Read));
    int64_t fileSize = 0;
    NN_RESULT_DO(nn::fs::GetFileSize(&fileSize, fileHandle));
    std::memset(outBuffer, 0x77, outBufferLength);
    NN_RESULT_DO(nn::fs::ReadFile(fileHandle, 0, outBuffer, outBufferLength));
    nn::fs::CloseFile(fileHandle);
    NN_RESULT_SUCCESS;
}

//------------------------------------------------------------------------------
//  SD カードに結果を保存
//------------------------------------------------------------------------------
nn::Result WriteResultToSd()
{
    nn::fs::FileHandle file;
    char filePath[64];
    nn::util::Strlcpy(filePath, "sdcard:/GcAsicInfo.bin", sizeof(filePath));
    //nn::fs::DeleteFile(filePath);
    NN_RESULT_DO(nn::fs::CreateFile(filePath, 0));
    NN_RESULT_DO(nn::fs::OpenFile(&file, filePath, nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend));
    int64_t fileSize;
    NN_RESULT_DO(GetFileSize(&fileSize, file));
    NN_LOG("Write\n");
    auto option = nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag::WriteOptionFlag_Flush);
    NN_RESULT_DO(WriteFile(file, fileSize, reinterpret_cast<char*>(&g_TestInfo.rmaInfo), sizeof(g_TestInfo.rmaInfo), option));
    CloseFile(file);
    NN_RESULT_SUCCESS;
}


void InitializeTestInfo()
{
    g_TestInfo.isTestFinished   = false;
    g_TestInfo.isSdCardInserted = true;
    g_TestInfo.currentIndex = 0;
    g_TestInfo.result = nn::ResultSuccess();
    memset(&(g_TestInfo.rmaInfo), 0, sizeof(g_TestInfo.rmaInfo));
}

//------------------------------------------------------------------------------
//  メイン 関数
//------------------------------------------------------------------------------
extern "C" void nnMain()
{
    // ABORT 殺す
    nn::fs::SetEnabledAutoAbort(false);
#if defined(NN_BUILD_TARGET_PLATFORM_NX)
    // グラフィックスシステムのためのメモリ周りの初期化を行います。
    {
        const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
        nv::SetGraphicsAllocator(NvAllocate, NvFree,NvReallocate,NULL);
        nv::InitializeGraphics(malloc(GraphicsSystemMemorySize), GraphicsSystemMemorySize);
    }
#endif

    InitializeTestInfo();
    nn::hid::InitializeTouchScreen();

    ApplicationHeap applicationHeap;
    applicationHeap.Initialize(ApplicationHeapSize);

    g_pGraphicsSystem = new GraphicsSystem();
    g_pGraphicsSystem->SetApplicationHeap(&applicationHeap);
    g_pGraphicsSystem->Initialize();

    g_pFontSystem = new FontSystem();
    g_pFontSystem->SetApplicationHeap(&applicationHeap);
    g_pFontSystem->SetGraphicsSystem(g_pGraphicsSystem);
    g_pFontSystem->Initialize();

    InitializeThread();
    NN_LOG("Setting End.Gc Test Start.\n");

    NN_LOG("---------LoopStart----------\n");

    auto result = nn::fs::MountSdCardForDebug("sdcard");
    if(result.IsFailure())
    {
        if(!nn::fs::IsSdCardInserted())
        {
            while(!nn::fs::IsSdCardInserted())
            {
                g_TestInfo.isSdCardInserted = false;
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));
            }
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountSdCardForDebug("sdcard"));
            g_TestInfo.isSdCardInserted = true;
        }
        else
        {
            NN_ABORT("UNKNOWN SD CARD ERROR\n");
        }
    }

    const size_t MaxVersionValue = 63;
    for(g_TestInfo.currentIndex = 0; g_TestInfo.currentIndex < MaxVersionValue; g_TestInfo.currentIndex++)
    {
        NN_LOG("LOOP %zu\n", g_TestInfo.currentIndex);
        NN_LOG("  --ReadFwFromSd--\n");
        g_TestInfo.result = ReadFwFromSd(g_RmaFwBuffer, sizeof(g_RmaFwBuffer), g_TestInfo.currentIndex);
        if(g_TestInfo.result.IsFailure())
        {
            NN_LOG("ReadFwFromSd is failure(Module:%d, Description:%d)\n", g_TestInfo.result.GetModule(), g_TestInfo.result.GetDescription());
            break;
        }
        NN_LOG("   ReadFw Dump(first 32 Byte)\n");
        PrintArray(g_RmaFwBuffer, 32);
        NN_LOG("  --SendFw AND GetGameCardAsicInfo--\n");
        if(nn::fs::GetGameCardAsicInfo(&g_TestInfo.rmaInfo, g_RmaFwBuffer, FwSize).IsSuccess())
        {
            g_TestInfo.isTestFinished = true;
            NN_LOG("---------LoopEnd----------\n");
            NN_LOG("\n\nSuccess.\n");
            break;
        }
        NN_LOG("  Failure.Next\n");
    }

    nn::os::WaitEvent(&g_ThreadEndEvent);

    WriteResultToSd();
    nn::fs::Unmount("sdcard");
    FinalizeThread();

    g_pFontSystem->Finalize();
    delete g_pFontSystem;

    g_pGraphicsSystem->Finalize();
    delete g_pGraphicsSystem;

    applicationHeap.Finalize();

    //    nn::os::FreeMemoryBlock(g_MallocBufferAddress, MemoryHeapSizeForMalloc);
    //    nn::os::FreeMemoryBlock(g_DataBufferAddress, DataBufferSize);
    //    nn::os::FreeMemoryBlock(g_WorkBufferAddress, WorkBufferSize);
}
