﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/rid.h>
#include "../DevQuestMenu_Log.h"
#include "DevQuestMenu_MenuUpdateModeScene.h"

namespace
{
    //! ダウンロード状況のラベル
    static const std::map<nn::rid::MenuUpdateState, std::string> UpdateStateLabel =
    {
        { nn::rid::MenuUpdateState::DoNothing, "DoNothing" },
        { nn::rid::MenuUpdateState::Downloading, "Downloading" },
        { nn::rid::MenuUpdateState::Completed, "Completed" },
        { nn::rid::MenuUpdateState::Failed, "Failed" },
        { nn::rid::MenuUpdateState::NeedNoUpdate, "NeedNoUpdate" },
    };
}

namespace nn { namespace devquestmenu {

    /**
     * @brief       試遊台メニュー更新モード起動時処理を行います
     *
     * @detail      試遊台メニュー更新を実行するスレッドを生成・実行します。
     */
    MenuUpdateModeScene::MenuUpdateModeScene(RootSurfaceContext* pRootSurface) NN_NOEXCEPT
        : ModeSceneCommon("Menu Update Mode", pRootSurface)
    {
        InitializeView();

        //! 試遊台メニュー更新スレッドの生成
        static NN_OS_ALIGNAS_THREAD_STACK char s_Stack[16 * 1024];
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&m_UpdateExecutionThread, [](void * p)
        {
            auto self = reinterpret_cast<MenuUpdateModeScene*>(p);
            self->m_MenuUpdateResult = self->m_MenuUpdater.Execute();
            self->m_IsMenuUpdateDone = true;
        }
        , this, s_Stack, sizeof(s_Stack), os::DefaultThreadPriority));

        //! 試遊台メニュー更新開始
        nn::os::StartThread(&m_UpdateExecutionThread);
    };

    /**
     * @brief       試遊台メニュー更新モード終了時処理を行います。
     *
     */
    MenuUpdateModeScene::~MenuUpdateModeScene() NN_NOEXCEPT
    {
        //! 実行スレッドの破棄
        nn::os::WaitThread(&m_UpdateExecutionThread);
        nn::os::DestroyThread(&m_UpdateExecutionThread);
    }

    /**
     * @brief       試遊台メニュー更新をポーリングします
     *
     */
    void MenuUpdateModeScene::OnLoop() NN_NOEXCEPT
    {
        rid::MenuUpdateState progress = m_MenuUpdater.GetState();
        m_MenuUpdateStatus = MakeLabel(UpdateStateLabel.at(progress).c_str(), m_StatusLabelSpec);
        QUESTMENU_LOG("%s\n", UpdateStateLabel.at(progress).c_str());
    };

    /**
     * @brief       アプリケーション更新モードに遷移できるかを判定します
     *
     * @param[in]   displayModeTime     更新開始時間の設定
     *
     */
    ModeType MenuUpdateModeScene::DetermineNextMode(const DisplayModeTime& displayModeTime) NN_NOEXCEPT
    {
        NN_UNUSED(displayModeTime);

        if (!m_IsMenuUpdateDone)    //! 試遊台メニュー更新が完了していない
        {
            return ModeType_NonChange;
        }
        else
        {
            if (m_MenuUpdateResult.IsFailure()) //! 試遊台メニューアップデートに失敗した
            {
                QUESTMENU_LOG("Failed as result 0x%08x\n", m_MenuUpdateResult.GetInnerValueForDebug());
                return ModeType_SleepMode;
            }
            else
            {
                rid::MenuUpdateState progress = m_MenuUpdater.GetState();
                if (progress == rid::MenuUpdateState::NeedNoUpdate)
                {
                    m_pRootSurface->SetMenuUpdateRequired(false); //! 要メニュー更新フラグを降ろす
                }
                else if (progress == rid::MenuUpdateState::Completed)
                {
                    m_pRootSurface->SetMenuUpdateRequired(false); //! 要メニュー更新フラグを降ろす
                    m_pRootSurface->SaveSettingsAndReboot(); //! 本体再起動
                }

                return ModeType_ApplicationUpdateMode;
            }
        }
    };

    ModeType MenuUpdateModeScene::GetSuccessNextMode() NN_NOEXCEPT
    {
        if (!m_IsMenuUpdateDone)    //! 試遊台メニュー更新が完了していない
        {
            m_MenuUpdateStatus = MakeLabel("Now Menu Updating. Please Wait!", m_StatusLabelSpec);
            return ModeType_NonChange;
        }
        else
        {
            return ModeType_ApplicationUpdateMode;
        }
    };

    void MenuUpdateModeScene::InitializeView() NN_NOEXCEPT
    {
        //! 更新状態ラベル
        m_MenuUpdateStatus = MakeLabel("Menu Update Start", m_StatusLabelSpec);
    };
}}
