﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <iostream>
#include <iomanip>
#include <sstream>
#include <nn/rid.h>
#include <nn/account/account_Types.h>
#include <nn/fs/fs_DeviceSaveData.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/rid/rid_Result.h>
#include <nn/util/util_ScopeExit.h>
#include "DevQuestMenu_DisplayModeScene.h"
#include "../DevQuestMenu_Log.h"
#include "../DevQuestMenu_RapidJsonParser.h"
#include "../DevQuestMenu_RidBuffer.h"
#include "../DevQuestMenu_RidFileIo.h"
#include "../DevQuestMenu_RidSaveDataIo.h"

namespace
{
    //! 追加コンテンツインデックスを列挙するためのバッファ
    const int MaxListableCount = 256;
    nn::aoc::AddOnContentIndex g_AocListBuffer[MaxListableCount];
    nn::ncm::ApplicationId g_ApplicationListBuffer[MaxListableCount];

    /**
     * @brief       本体にインストールされている Aoc のコンテントメタ情報 (.json) を読み込みます
     *
     * @param[out]  jsonBuffer    json の中身
     * @param[in]   index         本体内での index 値
     *
     * @details     aoc 内に .json が存在しない場合、aoc の中身をログとして列挙するだけで処理を終了します。
     */
    void ReadAssetMetaJson(nn::Bit8* jsonBuffer, int bufferSize, const int index) NN_NOEXCEPT
    {
        //! 追加コンテンツのマウント
        std::size_t cacheSize = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountAddOnContentCacheSize(&cacheSize, index));

        std::unique_ptr<char[]> mountCacheBuffer(new(std::nothrow) char[cacheSize]);
        NN_ASSERT_NOT_NULL(mountCacheBuffer);

        const char mountName[] = "aoc";
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountAddOnContent(mountName, index, mountCacheBuffer.get(), cacheSize));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(mountName);
        };

        //! 先にQuestMenu と同じ仕様 Asset を探す
        bool hasJsonInChildDirectory = false;
        {
            //! マウントされた追加コンテンツのディレクトリへのアクセス
            nn::fs::DirectoryHandle directoryHandle;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenDirectory(&directoryHandle, "aoc:/", nn::fs::OpenDirectoryMode::OpenDirectoryMode_Directory));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseDirectory(directoryHandle);
            };

            //! ディレクトリの子エントリの個数を取得する
            int64_t directoryReadCount;
            nn::fs::DirectoryEntry directoryEntry;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetDirectoryEntryCount(&directoryReadCount, directoryHandle));

            for (int i = 0; i < directoryReadCount; i++)
            {
                int64_t readDirectoryEntryValue;
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadDirectory(&readDirectoryEntryValue, &directoryEntry, directoryHandle, 1));

                std::string childDirectoryPath = "aoc:/" + std::string(directoryEntry.name);
                nn::fs::DirectoryHandle fileHandle;
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenDirectory(&fileHandle, childDirectoryPath.c_str(), nn::fs::OpenDirectoryMode::OpenDirectoryMode_File));
                NN_UTIL_SCOPE_EXIT
                {
                    nn::fs::CloseDirectory(fileHandle);
                };

                int64_t fileReadCount;
                nn::fs::DirectoryEntry fileEntry;
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetDirectoryEntryCount(&fileReadCount, fileHandle));

                for (int j = 0; j < fileReadCount; j++)
                {
                    int64_t readFileEntryValue;
                    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadDirectory(&readFileEntryValue, &fileEntry, fileHandle, 1));

                    QUESTMENU_LOG("EntryName -> %s\n", fileEntry.name);

                    std::string entryPath = fileEntry.name;
                    if (entryPath.rfind(".json") != std::string::npos)
                    {
                        entryPath = childDirectoryPath + "/" +entryPath;
                        nn::devquestmenu::RidFileIo::ReadFile(jsonBuffer, bufferSize, entryPath.c_str());
                        hasJsonInChildDirectory = true;
                    }
                }
            }
        }

        //! QuestMenu と同じ仕様の Asset がなかった場合 DevQuestMenu 用の Asset を探す
        if (!hasJsonInChildDirectory)
        {
            //! マウントされた追加コンテンツのディレクトリへのアクセス
            nn::fs::DirectoryHandle directoryHandle;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenDirectory(&directoryHandle, "aoc:/", nn::fs::OpenDirectoryMode::OpenDirectoryMode_All));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseDirectory(directoryHandle);
            };

            //! ディレクトリの子エントリの個数を取得する
            int64_t readCount;
            nn::fs::DirectoryEntry entry;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetDirectoryEntryCount(&readCount, directoryHandle));

            for (int i = 0; i < readCount; i++)
            {
                int64_t readDirectoryEntryValue;
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadDirectory(&readDirectoryEntryValue, &entry, directoryHandle, 1));
                QUESTMENU_LOG("EntryName -> %s\n", entry.name);

                std::string entryPath = entry.name;
                if (entryPath.rfind(".json") != std::string::npos)
                {
                    entryPath = "aoc:/" + entryPath;
                    nn::devquestmenu::RidFileIo::ReadFile(jsonBuffer, bufferSize, entryPath.c_str());
                }
            }
        }
    }
}

namespace nn { namespace devquestmenu {

    DisplayModeScene::DisplayModeScene(RootSurfaceContext* pRootSurface) NN_NOEXCEPT
        : ModeSceneCommon("Display Mode", pRootSurface)
        , m_InstalledAocCount(0)
        , m_IsLaunchDelayed(false)
        , m_IsSleepRequiredForPowerSupply(false)
    {
        bool isUpdateDone = false;
        // 試遊モードの開始時に必要な処理を行う
        // 時間がかかる可能性もあるので、コンストラクタで行わないほうがよいがとりあえず
        Result startUpResult = rid::StartUp();
        if (rid::ResultAccountNotFound::Includes(startUpResult))
        {
            QUESTMENU_LOG("User Account is not found. Please create user account.\n");
        }

        // SD カードのチェックを行う
        Result checkSdCardResult = rid::CheckSdCard();
        if (rid::ResultSdCardNotInserted::Includes(checkSdCardResult))
        {
            QUESTMENU_LOG("SD card is not inserted.\n");
        }
        else if(rid::ResultSdCardNotMounted::Includes(checkSdCardResult))
        {
            QUESTMENU_LOG("SD card was inserted after starting. Please reboot device.\n");
        }
        else if (rid::ResultSdCardNeedsSystemUpdate::Includes(checkSdCardResult))
        {
            QUESTMENU_LOG("System update is needed to use SD card. Please update system.\n");
        }
        else if (rid::ResultSdCardNoOwnership::Includes(checkSdCardResult) || rid::ResultSdCardDatabaseCorrupted::Includes(checkSdCardResult) || rid::ResultSdCardFileSystemCorrupted::Includes(checkSdCardResult))
        {
            QUESTMENU_LOG("A fault of SD card is found. Please format SD card and try again.\n");
        }

        int applicationCount;
        isUpdateDone &= UpdateAocListSaveData(g_AocListBuffer, &m_InstalledAocCount);
        isUpdateDone &= UpdateApplicationListSaveData(g_ApplicationListBuffer, &applicationCount);

        if (isUpdateDone)
        {
            //! TODO: UpdateAssetDatabase() を実装する
        }

        //! TODO: 最終的には AssetDatabase を入力する
        InitializeAssetMetaList(g_AocListBuffer, m_InstalledAocCount);

        //! 電源監視の開始
        m_pPowerSupplyChecker.reset(new PowerSupplyChecker(&m_IsSleepRequiredForPowerSupply));
    };

    DisplayModeScene::~DisplayModeScene() NN_NOEXCEPT
    {
        //! 電源供給チェッカーのみ明示的に削除する
        m_pPowerSupplyChecker.reset();
    };

    void DisplayModeScene::OnLoop() NN_NOEXCEPT
    {
    };

    /**
     * @brief       本体更新モードに遷移できるかを判定します
     *
     * @param[in]   displayModeTime     更新開始時間の設定
     *
     * @detail      バッテリーが残量が 25% 以下かつ給電されていないの場合、スリープさせます
     *              displayModeTime で定義された更新時間に入ったら、本体更新モードに遷移します
     */
    ModeType DisplayModeScene::DetermineNextMode(const DisplayModeTime& displayModeTime) NN_NOEXCEPT
    {
        if (m_IsSleepRequiredForPowerSupply) //! 電源供給が足りていない
        {
            return ModeType_SleepMode;
        }
        else
        {
            if (!displayModeTime.IsDuringDisplayModeTime()) //! 試遊モード時間外となったら本体再起動
            {
                m_pRootSurface->SaveSettingsAndReboot();
            }
            return ModeType_NonChange;
        }
    };

    ModeType DisplayModeScene::GetSuccessNextMode() NN_NOEXCEPT
    {
        return ModeType_SystemUpdateMode;
    };

    bool DisplayModeScene::UpdateAocListSaveData(nn::aoc::AddOnContentIndex* pOutAocBuffer, int* pOutAocCount) NN_NOEXCEPT
    {
        //! 本体内の Aoc リストの取得
        *pOutAocCount = nn::aoc::ListAddOnContent(pOutAocBuffer, 0, MaxListableCount);
        std::vector<std::string> deviceAocList;
        for (int i = 0; i < *pOutAocCount; i++)
        {
            auto index = pOutAocBuffer[i];
            deviceAocList.push_back(std::to_string(index));
        }

        return UpdateSaveData("aoc", deviceAocList, *pOutAocCount, "ridSaveDataAoc.json");
    }

    bool DisplayModeScene::UpdateApplicationListSaveData(nn::ncm::ApplicationId* pOutApplicationBuffer, int* pOutApplicationCount) NN_NOEXCEPT
    {
        //! 本体内の Application リストの取得
        //! TODO: rid 用の Application Id 取得 API が提供されたら移行する
        *pOutApplicationCount = nn::rid::ListApplication(pOutApplicationBuffer, MaxListableCount);
        const int idLength = 16; //! Application Id は 16 進数 16 桁
        std::vector<std::string> applicationList;
        for (int i = 0; i < *pOutApplicationCount; i++)
        {
            char id[idLength + 1];
#if defined( NN_BUILD_CONFIG_ADDRESS_SUPPORTS_32 )
            nn::util::SNPrintf(id, idLength + 1, "%016llx", pOutApplicationBuffer[i].value);
#else
            nn::util::SNPrintf(id, idLength + 1, "%016lx", pOutApplicationBuffer[i].value);
#endif
            applicationList.push_back(std::string(id));
        }

        return UpdateSaveData("application", applicationList, *pOutApplicationCount, "ridSaveDataApplication.json");
    }

    bool DisplayModeScene::UpdateSaveData(const char* key, const std::vector<std::string>& idList, int count, const char* savedataName) NN_NOEXCEPT
    {
        //! idList の json を作成
        std::string idListJson = "{" + RapidJsonParser::CreateJsonArray(key, idList, count) + "}";
        QUESTMENU_LOG("Device Json -> %s\n", idListJson.c_str());

        //! Save Data List と Device List を比較する
        if (RidSaveDataIo::Read(RidBuffer::s_Buffer, RidBuffer::BufferSize, savedataName))
        {
            QUESTMENU_LOG("Save Data Json -> %s\n", (const char*)RidBuffer::s_Buffer);
            if (std::strcmp(idListJson.c_str(), (const char*)RidBuffer::s_Buffer) == 0)
            {
                return false;
            }
        }
        else
        {
            QUESTMENU_LOG("%s not found\n", savedataName);
        }

        memset(RidBuffer::s_Buffer, 0, RidBuffer::BufferSize); //! バッファクリア
        memcpy(RidBuffer::s_Buffer, (nn::Bit8*)idListJson.c_str(), idListJson.length()); //! バッファコピー

        //! セーブデータを更新する
        RidSaveDataIo::Write(RidBuffer::s_Buffer, RidBuffer::BufferSize, savedataName);
        return true;
    }

    /**
     * @brief       本体にインストールされている Aoc 情報を初期化します
     *
     */
    void DisplayModeScene::InitializeAssetMetaList(const nn::aoc::AddOnContentIndex* pAocList, int aocCount) NN_NOEXCEPT
    {
        m_AssetMetaContainer.clear();
        for (int i = 0; i < aocCount; i++)
        {
            auto index = pAocList[i];
            QUESTMENU_LOG("  Index[%d]: %d\n", i, index);
            ReadAssetMetaJson(RidBuffer::s_Buffer, RidBuffer::BufferSize, pAocList[i]);
            std::unique_ptr<AssetMeta> assetMeta(new AssetMeta((const char*)RidBuffer::s_Buffer));
            m_AssetMetaContainer.push_back(std::move(assetMeta));
        }
    };

    /**
     * @brief       アプリケーションジャンプを行うボタンをリスト状に列挙します。
     *
     */
    void DisplayModeScene::RegisterDebugButton(glv::ViewContainer* pContainer) NN_NOEXCEPT
    {
        //! アプリジャンプ遅延のデバッグボタン
        glv::Rect delayButtonRect = { m_GridWidth * 3, m_GridHeight * 11, m_GridWidth * 2, m_GridHeight * 3};
        auto* pIsLaunchDelayed = &m_IsLaunchDelayed;
        auto* pDelayLaunchButton = &m_DelayLaunchButton;
        m_DelayLaunchButton = MakeLabelButton(pContainer, "Delay Launch\nfalse", [pIsLaunchDelayed, pDelayLaunchButton]
        {
            *pIsLaunchDelayed ^= true;
            std::string flag = *pIsLaunchDelayed ? "true" : "false";
            std::string caption = "Delay Launch\n" + flag;
            pDelayLaunchButton->get()->ChangeCaption(caption.c_str());
        }
        , delayButtonRect);

        //! メニューをアクセス例外でクラッシュさせる（メニューが死に SA のダイアログが出た後、再起動される）
        glv::Rect accessExceptionButtonRect = { m_GridWidth * 3, m_GridHeight * 8, m_GridWidth * 2, m_GridHeight * 3 };
        m_AccessExceptionButton = MakeLabelButton(pContainer, "Menu\nAccess\nException", []
        {
            //! pc = 0xffc, r0 = 0xdeadbeef でメニューがアクセス例外で落ちる
            void (*pFunc) (int n) = reinterpret_cast<void(*) (int n)>(0xffc);
            pFunc(0xdeadbeef);
        }
        , accessExceptionButtonRect);

        //! メニューをアボートでクラッシュさせる（FATAL 画面が出る。再起動はしない）
        glv::Rect menuAbortButtonRect = { m_GridWidth * 3, m_GridHeight * 5, m_GridWidth * 2, m_GridHeight * 3 };
        m_MenuAbortButton = MakeLabelButton(pContainer, "Menu\nAbort", []
        {
            //! TODO: rid に Quest Menu を強制アボートさせた場合の Result を設定するのが理想
            NN_ABORT();
        }
        , menuAbortButtonRect);

        //! AssetMeta のタイトル情報をアプリケーションジャンプ付きボタンで列挙する
        glv::Rect aocButtonRect = m_AocButtonRect;
        auto pRootSurface = m_pRootSurface;
        for (unsigned int i = 0; i < m_InstalledAocCount; i++)
        {
            auto* p = m_AssetMetaContainer[i].get();
            std::unique_ptr<LabelButton> pButton = MakeLabelButton(pContainer, m_AssetMetaContainer[i].get()->m_Title.usEnglish, [pRootSurface, p, pIsLaunchDelayed]
            {
                pRootSurface->SaveSettings();
                p->LaunchApplication(*pIsLaunchDelayed); //! アプリケーションジャンプ
            }
            , aocButtonRect);

            m_LabelButtonContainer.push_back(std::move(pButton));
            aocButtonRect.top(aocButtonRect.t + aocButtonRect.h);
        }

        //! ダミーアイコン
        std::unique_ptr<TextureView> dummyIcon(new TextureView(m_TextureRect));
        *this << dummyIcon.get();
        m_SelectedAocIcon = std::move(dummyIcon);
    };
}}
