﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once
#include "common.h"
#include <glv_resources.h>
#include <nn/rid.h>
#include "DevQuestMenu_LabelButton.h"
#include "DevQuestMenu_ModeType.h"
#include "ModeScene/DevQuestMenu_ModeSceneCommon.h"

namespace nn { namespace devquestmenu {

    /**
     * @brief       DevQuestMenu のエントリポイントです
     */
    void DevQuestMenuMain() NN_NOEXCEPT;

    struct TimeScheduleUnit
    {
        std::unique_ptr<glv::Label> captureLabel;
        std::unique_ptr<glv::Label> hourLabel;
        std::unique_ptr<glv::Label> minuteLabel;
        std::unique_ptr<LabelButton> hourUp;
        std::unique_ptr<LabelButton> minuteUp;
        std::unique_ptr<LabelButton> hourDown;
        std::unique_ptr<LabelButton> minuteDown;
    };

    class ModeSceneCommon; //! RootSurface, ModeSceneCommon 相互参照用の前方宣言

    /**
     * @brief       DevQuestMenu 用の RootSurface です
     */
    class RootSurfaceContext : public glv::Window, public glv::GLV, public glv::ApplicationLoopCallback
    {
    public:

        //! デバッグタブは各シーンからもボタンを登録できるように public に置きます
        std::unique_ptr<glv::ViewContainer> m_DebugTab;

        /**
         * @brief       コンストラクタです。
         *
         * @param[in]   width       幅
         * @param[in]   height      高
         */
        RootSurfaceContext(float width, float height) NN_NOEXCEPT;

        /**
         * @brief       デストラクタです。
         */
        virtual ~RootSurfaceContext() NN_NOEXCEPT NN_OVERRIDE;

        /**
         * @brief   試遊台メニューの設定を保存します
         *
         */
        void SaveSettings() NN_NOEXCEPT;

        /**
         * @brief   試遊台メニューの設定を保存して本体再起動します
         *
         */
        void SaveSettingsAndReboot() NN_NOEXCEPT;

        /**
         * @brief       ランタイムエンジンにアタッチされた際に呼ばれます。
         *
         * @param[in]   context     コンテキスト
         *
         * @see         glv::ApplicationLoopCallback::OnLoopAttached()
         */
        virtual void OnLoopAttached(glv::ApplicationLoopContext& context) NN_NOEXCEPT NN_OVERRIDE;

        /**
         * @brief       ランタイムエンジンからデタッチされた際に呼ばれます。
         *
         * @param[in]   context     コンテキスト
         *
         * @see         glv::ApplicationLoopCallback::OnLoopDetached()
         */
        virtual void OnLoopDetached(glv::ApplicationLoopContext& context) NN_NOEXCEPT NN_OVERRIDE;

        /**
         * @brief       glv シーンレンダラ前に呼ばれます。
         *
         * @param[in]   context     コンテキスト
         * @param[in]   events      イベント
         *
         * @return      現在は RequiredRestoration::RequireRestrationNothing を返すようにしてください。
         */
        virtual const glv::RequiredRestoration OnLoopBeforeSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT NN_OVERRIDE;

        /**
         * @brief       glv シーンレンダラ後に呼ばれます。
         *
         * @param[in]   context     コンテキスト
         * @param[in]   events      イベント
         *
         * @return      現在は RequiredRestoration::RequireRestrationNothing を返すようにしてください。
         */
        virtual const glv::RequiredRestoration OnLoopAfterSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT NN_OVERRIDE;

        virtual bool onEvent(glv::Event::t events, glv::GLV& context) NN_NOEXCEPT NN_OVERRIDE;

        virtual void onDraw(glv::GLV& context) NN_NOEXCEPT NN_OVERRIDE;

        //! getter
        float GetDisplayWidth() const NN_NOEXCEPT { return m_DisplayWidth; };
        float GetDisplayHeight() const NN_NOEXCEPT { return m_DisplayHeight; };
        float GetDisplayGridWidth() const NN_NOEXCEPT { return m_DisplayGridWidth; };
        float GetDisplayGridHeight() const NN_NOEXCEPT { return m_DisplayGridHeight; };
        DisplayModeTime GetDisplayModeTime() const NN_NOEXCEPT { return m_DisplayModeTime; };
        bool IsMenuUpdateRequired() const NN_NOEXCEPT { return m_IsMenuUpdateRequired; };
        bool IsApplicationUpdating() const NN_NOEXCEPT { return m_IsApplicationUpdating; };
        bool IsMenuUpdateSkipped() const NN_NOEXCEPT { return m_IsMenuUpdateSkipped; };
        LabelButton* GetMenuUpdateSkipButton() const NN_NOEXCEPT { return m_MenuUpdateSkipButton.get(); };

        //! setter
        void SetMenuUpdateRequired(bool isMenuUpdateRequired) NN_NOEXCEPT { m_IsMenuUpdateRequired = isMenuUpdateRequired; };
        void SetApplicationUpdating(bool isApplicationUpdating) NN_NOEXCEPT { m_IsApplicationUpdating = isApplicationUpdating; };
        void SetMenuUpdateSkipped(bool isMenuUpdateSkipped) NN_NOEXCEPT { m_IsMenuUpdateSkipped = isMenuUpdateSkipped; };

    private:

        /**
         * @brief       デバッグメニューを配置します
         *
         */
        void InitializeDebugMenu(float gridWidth, float gridHeight) NN_NOEXCEPT;

        void InitializeMenuSettings() NN_NOEXCEPT;

        /**
         * @brief       時間設定用の UI を作成します
         *
         */
        std::unique_ptr<TimeScheduleUnit> MakeTimeScheduleUnit(glv::ViewContainer* pContainer, nn::time::CalendarTime* pTime, const char* capture, const glv::Rect& r) NN_NOEXCEPT;

        /**
         * @brief       ラベルを作成します
         *
         * @param[in]   label       表示
         * @param[in]   spec        位置
         */
        std::unique_ptr<glv::Label> MakeLabel(const std::string& label, const glv::Label::Spec& spec) NN_NOEXCEPT;

        /**
         * @brief       ラベルを作成します
         *
         * @param[in]   label       表示
         * @param[in]   spec        位置
         */
        std::unique_ptr<glv::Label> MakeLabel(glv::ViewContainer* pContainer, const std::string& label, const glv::Label::Spec& spec) NN_NOEXCEPT;

        /**
         * @brief       ボタンを作成します
         *
         * @param[in]   label       表示
         * @param[in]   callback    実行される関数
         * @param[in]   r           位置・大きさ
         */
        std::unique_ptr<LabelButton> MakeLabelButton(const std::string& label, std::function< void() > callback, const glv::Rect& r) NN_NOEXCEPT;

        /**
         * @brief       ボタンを作成します
         *
         * @param[in]   label       表示
         * @param[in]   callback    実行される関数
         * @param[in]   r           位置・大きさ
         */
        std::unique_ptr<LabelButton> MakeLabelButton(glv::ViewContainer* pContainer, const std::string& label, std::function< void() > callback, const glv::Rect& r) NN_NOEXCEPT;

        /**
         * @brief       現在時刻を更新します
         *
         */
        void UpdateCurrentTime() NN_NOEXCEPT;

        /**
         * @brief       ループ毎の後処理を行います。
         *
         * @param[in]   context     コンテキスト
         * @param[in]   events      イベント
         */
        void HandleLoopPost(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT;

        /**
         * @brief       HID イベントを処理します。
         *
         * @param[in]   context     コンテキスト
         * @param[in]   events      イベント
         */
        void HandleHidEvent(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT;

        /**
         * @brief       対象の動作モードのシーンを作成します
         *
         * @param[in]   width       画面の横幅
         * @param[in]   height      画面の縦幅
         */
        template<typename T>
        std::unique_ptr<ModeSceneCommon> CreateScene() NN_NOEXCEPT;

        /**
         * @brief       対象の動作モードに遷移させます
         *
         * @param[in]   targetMode  遷移先の動作モード
         */
        void ChangeMode(ModeType targetMode) NN_NOEXCEPT;

        //! 内部処理に関わる変数
        std::unique_ptr<ModeSceneCommon> m_pModeScene;              //! 動作モード
        nn::os::Tick m_ElapsedTick;                                 //! 経過時間計測用の Tick
        float m_DisplayWidth;                                       //! ディスプレイの横幅
        float m_DisplayHeight;                                      //! ディスプレイの縦幅
        float m_DisplayGridWidth;                                   //! width の 16 分割単位
        float m_DisplayGridHeight;                                  //! height の 18 分割単位
        DisplayModeTime m_DisplayModeTime;                      //! 更新モード時間の管理クラス
        bool m_IsMenuUpdateRequired;                                //! 要試遊台メニュー更新フラグ
        bool m_IsApplicationUpdating;                               //! アプリケーション更新中フラグ

        //! Menu 更新を行わないフラグ
        bool m_IsMenuUpdateSkipped;

        //! 描画オブジェクトの位置
        glv::Rect m_DebugButtonRect;
        glv::Rect m_MenuUpdateSkipButtonRect;
        glv::Label::Spec m_HeaderSpec;
        glv::Label::Spec m_CurrentTimeSpec;

        //! 描画オブジェクト
        std::unique_ptr<glv::Label> m_HeaderLabel;              //! Header
        std::unique_ptr<glv::Label> m_CurrentTimeLabel;         //! 現在時刻ラベル
        std::unique_ptr<glv::Label> m_StartTimeLabel;           //! 更新開始時刻ラベル
        std::unique_ptr<glv::Label> m_EndTimeLabel;             //! 更新終了時刻ラベル
        std::unique_ptr<LabelButton> m_TransitionButton;        //! モード遷移強制ボタン
        std::unique_ptr<LabelButton> m_MenuUpdateSkipButton; //! メニュー更新スキップボタン
        std::unique_ptr<TimeScheduleUnit> m_StartTimeUnit;
        std::unique_ptr<TimeScheduleUnit> m_EndTimeUnit;
    };
}}
