﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/os.h>
#include <nn/psm/psm.h>
#include <nn/psm/psm_System.h>
#include <nn/psm/psm_SystemProcessApi.h>
#include "DevQuestMenu_PowerSupplyChecker.h"

namespace
{
    const double EnoughPowerSupply = 25.f;            //! システム維持に必要なバッテリ残量
}

namespace nn {namespace devquestmenu {

    PowerSupplyChecker::PowerSupplyChecker(bool* pIsSleepRequired) NN_NOEXCEPT
    {
        nn::os::InitializeEvent( &m_EndEvent, false, nn::os::EventClearMode_ManualClear);
        m_pIsSleepRequired = pIsSleepRequired;

        //! 電源監視スレッドの生成
        static NN_OS_ALIGNAS_THREAD_STACK char s_Stack[16 * 1024];
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&m_PowerSupplyCheckThreadFunction, [](void * p)
        {
            auto self = reinterpret_cast<PowerSupplyChecker*>(p);

            //! 給電状態変化イベントの設定
            nn::os::SystemEventType chargerTypeChangeEvent;
            nn::psm::GetChargerTypeChangeEvent(&chargerTypeChangeEvent);

            //! 終了イベントが signal されるまで電源監視を続ける
            while (!nn::os::TryWaitEvent( &self->m_EndEvent ))
            {
                //! 給電状態の取得
                nn::psm::ChargerType chargerType = nn::psm::GetChargerType();

                if (chargerType == nn::psm::ChargerType::ChargerType_EnoughPower)
                {
                    *(self->m_pIsSleepRequired) = false;

                    // 給電が切れるか、監視終了イベントがシグナルされるまで待機する
                    const auto index = nn::os::WaitAny(&chargerTypeChangeEvent, &self->m_EndEvent);
                    if (index == 0)
                    {
                        //! 給電状態変化のイベントのみクリア
                        nn::os::ClearSystemEvent(&chargerTypeChangeEvent);
                    }
                }
                else
                {
                    //! 非給電状態の時はバッテリ残量をポーリングする
                    *(self->m_pIsSleepRequired) = nn::psm::GetRawBatteryChargePercentage() <= EnoughPowerSupply;
                    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
                }
            }
        }
        , this, s_Stack, sizeof(s_Stack), os::DefaultThreadPriority));

        //! 電源監視開始
        nn::os::StartThread(&m_PowerSupplyCheckThreadFunction);
    };

    /**
     * @brief       電源供給監視の終了処理です
     *
     */
    PowerSupplyChecker::~PowerSupplyChecker() NN_NOEXCEPT
    {
        nn::os::SignalEvent(&m_EndEvent);     //! 電源監視スレッド内の繰り返し処理を終了させる。

        nn::os::WaitThread(&m_PowerSupplyCheckThreadFunction);
        nn::os::DestroyThread(&m_PowerSupplyCheckThreadFunction);
        nn::os::FinalizeEvent(&m_EndEvent);
    }
}}
