﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/util/util_StringUtil.h>
#include "DevQuestMenu_DebugSettings.h"
#include "DevQuestMenu_Log.h"
#include "DevQuestMenu_RapidJsonParser.h"
#include "DevQuestMenu_RidBuffer.h"
#include "DevQuestMenu_RidFileIo.h"

namespace nn { namespace devquestmenu {

    namespace
    {
        // .json 指定の ApplicationAsset を作成する
        void CreateApplicationAsset(rid::ApplicationAsset outAssetList[], int assetListCount, const nne::rapidjson::Document& document) NN_NOEXCEPT
        {
            auto toAsset = [](rid::ApplicationAsset* pAsset, const nne::rapidjson::Value& value)->void
            {
                std::string usage = value["usage"].GetString();

                if (usage == "index")
                {
                    pAsset->index = std::stoi(value["id"].GetString());
                }
                else if (usage == "demoProgramID")
                {
                    if (util::Strncmp(value["id"].GetString(), "", 1) == 0)
                    {
                        pAsset->demoApplicationId = rid::InvalidApplicationId;
                    }
                    else
                    {
                        pAsset->demoApplicationId = { std::stoull(value["id"].GetString(), nullptr, 16) };
                    }
                }
                else if (usage == "menuProgramID")
                {
                    pAsset->aocApplicationId = { std::stoull(value["id"].GetString(), nullptr, 16) };
                }
            };

            //! "content" 配列の参照
            const nne::rapidjson::Value& contentValue = document["content"];
            for (int i = 0; i < assetListCount; i++)
            {
                //! "content" 内オブジェクトの参照
                for (nne::rapidjson::Value::ConstMemberIterator contentIterator = contentValue[i].MemberBegin();
                    contentIterator != contentValue[i].MemberEnd(); contentIterator++)
                {
                    if (contentIterator->name == "id")
                    {
                        //! "id" オブジェクトの参照
                        const nne::rapidjson::Value& idValue = contentIterator->value;
                        for (nne::rapidjson::Value::ConstMemberIterator idIterator = idValue.MemberBegin();
                            idIterator != idValue.MemberEnd(); idIterator++)
                        {
                            if (idIterator->name == "idForDeviceMultiple")
                            {
                                //! "idForDeviceMultiple" 配列の参照
                                const int idCountOfApplicationAsset = 3;
                                const nne::rapidjson::Value& idForDeviceMultipleValue = idIterator->value;
                                for (int j = 0; j < idCountOfApplicationAsset; j++)
                                {
                                    toAsset(&(outAssetList[i]), idForDeviceMultipleValue[j]);
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    bool ReadDisplayModeSettingsForDebug(time::CalendarTime* outStartTime, time::CalendarTime* outEndTime) NN_NOEXCEPT
    {
        Result result = fs::MountSdCardForDebug("sd");
        if (result.IsSuccess())
        {
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::Unmount("sd");
            };

            if (RidFileIo::ReadFile(RidBuffer::s_Buffer, RidBuffer::BufferSize, "sd:/NintendoSDK/RetailInteractiveDisplay/DisplayModeSettings.json"))
            {
                nne::rapidjson::Document document;
                document.Parse(reinterpret_cast<const char*>(RidBuffer::s_Buffer));

                //! json の Parse に失敗した場合はアボートさせる
                if (document.HasParseError())
                {
                    NN_ABORT("PARSE ERROR CODE %d: DisplayModeSettings.json is not correct Format.\n", document.GetParseError());
                }

                int startHour = 0;
                if (!RapidJsonParser::GetIntValue(&startHour, "display_mode_start_hour", document))
                {
                    return false;
                }

                int startMinute = 0;
                if (!RapidJsonParser::GetIntValue(&startMinute, "display_mode_start_minute", document))
                {
                    return false;
                }

                int endHour = 0;
                if (!RapidJsonParser::GetIntValue(&endHour, "display_mode_end_hour", document))
                {
                    return false;
                }

                int endMinute = 0;
                if (!RapidJsonParser::GetIntValue(&endMinute, "display_mode_end_minute", document))
                {
                    return false;
                }

                outStartTime->hour = static_cast<int8_t>(startHour);
                outStartTime->minute = static_cast<int8_t>(startMinute);
                outEndTime->hour = static_cast<int8_t>(endHour);
                outEndTime->minute = static_cast<int8_t>(endMinute);

                return true;
            }
        }

        return false;
    }

    bool ReadPlayableTimerSettingsForDebug(bool* outValue) NN_NOEXCEPT
    {
        Result result = fs::MountSdCardForDebug("sd");
        if (result.IsSuccess())
        {
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::Unmount("sd");
            };

            if (RidFileIo::ReadFile(RidBuffer::s_Buffer, RidBuffer::BufferSize, "sd:/NintendoSDK/RetailInteractiveDisplay/TimerSettings.json"))
            {
                nne::rapidjson::Document document;
                document.Parse(reinterpret_cast<const char*>(RidBuffer::s_Buffer));

                //! json の Parse に失敗した場合はアボートさせる
                if (document.HasParseError())
                {
                    NN_ABORT("PARSE ERROR CODE %d: DisplayModeSettings.json is not correct Format.\n", document.GetParseError());
                }

                if (document.HasMember("playable_timer"))
                {
                    const nne::rapidjson::Value& playableTimer = document["playable_timer"];
                    if (RapidJsonParser::GetBoolValue(outValue, "is_enabled", playableTimer))
                    {
                        return true;
                    }
                }
            }
        }

        return false;
    }

    bool ReadIdleDetectionTimerSettingsForDebug(bool* outValue, uint32_t* outTime) NN_NOEXCEPT
    {
        Result result = fs::MountSdCardForDebug("sd");
        if (result.IsSuccess())
        {
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::Unmount("sd");
            };

            if (RidFileIo::ReadFile(RidBuffer::s_Buffer, RidBuffer::BufferSize, "sd:/NintendoSDK/RetailInteractiveDisplay/TimerSettings.json"))
            {
                nne::rapidjson::Document document;
                document.Parse(reinterpret_cast<const char*>(RidBuffer::s_Buffer));

                //! json の Parse に失敗した場合はアボートさせる
                if (document.HasParseError())
                {
                    NN_ABORT("PARSE ERROR CODE %d: DisplayModeSettings.json is not correct Format.\n", document.GetParseError());
                }

                if (document.HasMember("idle_detection_timer"))
                {
                    bool isEnabled;
                    const nne::rapidjson::Value& idleDetectionTimer = document["idle_detection_timer"];
                    if (!RapidJsonParser::GetBoolValue(&isEnabled, "is_enabled", idleDetectionTimer))
                    {
                        return false;
                    }
                    // 無操作時間タイマー設定が有効な場合、時間の設定を読む
                    if (isEnabled)
                    {
                        // 有効で時間の記述がないのは認めない
                        if (!RapidJsonParser::GetUintValue(outTime, "time", idleDetectionTimer))
                        {
                            return false;
                        }

                        *outValue = true;
                        return true;
                    }
                    // 無操作時間タイマー設定が無効な場合
                    else
                    {
                        *outValue = false;
                        *outTime = 0;
                        return true;
                    }
                }
            }
        }

        return false;
    }

    bool ReadUpdateApplicationListAndCreateAssetListForDebug(int* outAssetListCount, rid::ApplicationAsset outAssetList[], int assetListLength) NN_NOEXCEPT
    {
        nn::Result result = nn::fs::MountSdCardForDebug("sd");
        if (result.IsSuccess())
        {
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::Unmount("sd");
            };

            if (RidFileIo::ReadFile(RidBuffer::s_Buffer, RidBuffer::BufferSize, "sd:/NintendoSDK/RetailInteractiveDisplay/ContentList.json"))
            {
                //! SD カード内に ContentList.json が存在する場合、.json 内で指定されたアプリケーション更新を行う
                nne::rapidjson::Document document;
                document.Parse(reinterpret_cast<const char*>(RidBuffer::s_Buffer));

                //! json の Parse に失敗した場合はアボートさせる
                if (document.HasParseError())
                {
                    NN_ABORT("PARSE ERROR CODE %d: ContentList.json is not correct Format.\n", document.GetParseError());
                }

                std::string jsonText;
                RapidJsonParser::CreatePrettyJsonText(&jsonText, document);
                QUESTMENU_LOG("ContentList.json:\n%s\n", jsonText.c_str());

                int assetListCount = 0;
                RapidJsonParser::GetStringArrayLength(&assetListCount, "content", document);
                if (assetListCount > assetListLength)
                {
                    assetListCount = assetListLength;
                }
                QUESTMENU_LOG("assetListCount: %d\n", assetListCount);

                CreateApplicationAsset(outAssetList, assetListCount, document);
                for (int i = 0; i < assetListCount; i++)
                {
                    QUESTMENU_LOG("index: %d\n", outAssetList[i].index);
                    QUESTMENU_LOG("demoApplicationId: %016llx\n", outAssetList[i].demoApplicationId);
                    QUESTMENU_LOG("menuApplicationId: %016llx\n", outAssetList[i].aocApplicationId);
                }

                *outAssetListCount = assetListCount;
                return true;
            }
        }

        return false;
    }
}}
