﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ui_MenuItemDropDownList.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include "../framework/Framework.h"
#include "util/ui_CalculateItemSize.h"
#include "util/ui_CalculateTextSize.h"
#include "util/ui_CalculateAlignedPosition.h"
#include "util/ui_MenuItemGeometry.h"
#include "util/ui_ProcessLoadStoreValue.h"
#include "ui_MenuItemDropDownListEntry.h"
#include "ui_MenuItemScrollBox.h"

namespace ui{

    MenuItemDropDownList::MenuItemDropDownList(const MenuItemDropDownListStyle& style) NN_NOEXCEPT
        : m_Style(style)
    {
        m_pPanelContainer = std::make_shared<panel::PanelContainer>();
        m_pPanelText = std::make_shared<panel::PanelText>();
        m_pPanelContainer->AddChild(m_pPanelText);

        m_pListTable = std::make_shared<ui::MenuItemColumn>(ui::MenuItemTableStyle());
        m_pListEntryGroup = std::make_shared<ui::RadioButtonGroup>();
    }

    std::shared_ptr<MenuItemDropDownList> MenuItemDropDownList::AddDropDownItem(
        const std::string& name,
        const std::function<bool ()>& getter,
        const std::function<void (bool)>& setter,
        const std::function<void (bool, const std::shared_ptr<IMenuPage>&)>& valueChangedCallback
    ) NN_NOEXCEPT
    {
        auto p = std::make_shared<ui::MenuItemDropDownListEntry>(m_Style, this->shared_from_this(), m_pListEntryGroup, static_cast<int>(m_EntryList.size()), name);
        p->SetValueAccessor(getter, setter);
        p->SetValueChangedCallbackFunction(valueChangedCallback);

        EntryInfo info = {};
        info.m_pEntry = p;

        m_pListTable->AddCell(p);
        m_EntryList.push_back(info);

        return this->shared_from_this();
    }

    void MenuItemDropDownList::UpdateLayoutRecursively(const MenuLayoutUpdateContext& context) NN_NOEXCEPT
    {
        NN_UNUSED(context);
        auto margin = m_Style.GetMargin();

        // レイアウトサイズの最大値を計算
        Size layoutSizeMax = {};
        for(auto& e : m_EntryList)
        {
            std::string text = e.m_pEntry->GetText();
            Size textSize = util::CalculateTextSize::Calculate(text.c_str(), m_Style.GetTextStyle(IsEnabled(), IsFocused(), false).GetFontSize(), framework::TextWriterUsage_Text);
            Size layoutSize = {
                textSize.width + margin.left + margin.right,
                textSize.height + margin.top + margin.bottom
            };

            layoutSizeMax.width  = std::max(layoutSizeMax.width , layoutSize.width );
            layoutSizeMax.height = std::max(layoutSizeMax.height, layoutSize.height);
        }

        auto outerSize = util::CalculateItemSize::Calculate([&](){ return layoutSizeMax; }, m_Style.GetSizeStyle());

        // 最大値に合せて位置決め
        for(auto& e : m_EntryList)
        {
            auto layoutPosition = util::CalculateAlignedPosition::Calculate(layoutSizeMax, outerSize, m_Style.GetAlignment());
            e.m_TextPosition = {layoutPosition.x + margin.left, layoutPosition.y + margin.top};
        }

        SetLayoutSize(outerSize);
    }

    bool MenuItemDropDownList::IsFocusAcceptable() const NN_NOEXCEPT
    {
        return IsEnabled();
    }

    bool MenuItemDropDownList::IsTouchAcceptable() const NN_NOEXCEPT
    {
        return IsEnabled();
    }

    MenuButtonResult MenuItemDropDownList::HandlePositiveButtonPressed(const MenuButtonHandleContext& context) NN_NOEXCEPT
    {
        context.GetPage()->ChangeFocus(this->shared_from_this());
        PushSelectionLayerImpl(context.GetPage());
        return MenuButtonResult::GetHandled();
    }

    MenuButtonResult MenuItemDropDownList::HandleTouched(const MenuButtonHandleContext& context, const nn::util::Vector3f& position) NN_NOEXCEPT
    {
        NN_UNUSED(position);
        context.GetPage()->ChangeFocus(this->shared_from_this());
        PushSelectionLayerImpl(context.GetPage());
        return MenuButtonResult::GetHandled();
    }

    void MenuItemDropDownList::LoadValue(const std::shared_ptr<IMenuPage>& pPage) NN_NOEXCEPT
    {
        for(auto& e : m_EntryList)
        {
            util::ProcessLoadStoreValue::Load(e.m_pEntry, pPage);
        }
    }

    void MenuItemDropDownList::StoreValue() const NN_NOEXCEPT
    {
        for(auto& e : m_EntryList)
        {
            util::ProcessLoadStoreValue::Store(e.m_pEntry);
        }
    }

    void MenuItemDropDownList::PushSelectionLayerImpl(const std::shared_ptr<IMenuPage>& pPage) NN_NOEXCEPT
    {
        if(m_pListTable->GetParentInfo().pParent.lock())
        {
            NN_DEVOVL_MENU_LOG_INFO("dropdown already poped up\n");
            return;
        }

        const auto& popStyle = m_Style.GetDropDownPopupStyle();
        // 表示サイズを計算
        m_pListTable->UpdateLayoutRecursively(MenuLayoutUpdateContext(nullptr));
        auto tableSize = m_pListTable->GetLayoutSize();
        //NN_SDK_LOG("dropdown-popup: tableSize(%dx%d)\n", tableSize.width, tableSize.height);

        // 表示位置を計算
        auto myPos = util::MenuItemGeometry::GetGlobalPosition(this->shared_from_this());
        //NN_SDK_LOG("dropdown-popup: myPos(%d,%d)\n", myPos.x, myPos.y);

        // スクロールバーの幅を計算
        int hbarWidth = (popStyle.GetHorizontalScrollbarStyle().GetMode() == ScrollbarMode_Always) ? popStyle.GetHorizontalScrollbarStyle().GetWidth() : 0;
        int vbarWidth = (popStyle.GetVerticalScrollbarStyle().GetMode() == ScrollbarMode_Always) ? popStyle.GetVerticalScrollbarStyle().GetWidth() : 0;

        Position boxPos = {myPos.x, myPos.y + this->GetLayoutSize().height};
        Size boxSize = {tableSize.width + vbarWidth, tableSize.height + hbarWidth};
        //NN_SDK_LOG("dropdown-popup: box(%d,%d;%dx%d)\n", boxPos.x, boxPos.y, boxSize.width, boxSize.height);

        // ポップアップの表示範囲内に収める
        {
            auto regionLimit = m_Style.GetDropDownPopupStyle().GetRegion();

            boxSize.width  = std::min(boxSize.width , regionLimit.size.width);
            boxSize.height = std::min(boxSize.height, regionLimit.size.height);

            // 左端
            boxPos.x = std::max(boxPos.x, regionLimit.position.x);
            boxPos.y = std::max(boxPos.y, regionLimit.position.y);

            // 右端: boxPos + boxSize < reigionLimit.pos + regionLimit.size
            boxPos.x = std::min(boxPos.x, regionLimit.position.x + regionLimit.size.width  - boxSize.width);
            boxPos.y = std::min(boxPos.y, regionLimit.position.y + regionLimit.size.height - boxSize.height);
        }
        //NN_SDK_LOG("dropdown-popup: box(fixed) (%d,%d;%dx%d)\n", boxPos.x, boxPos.y, boxSize.width, boxSize.height);

        auto pBox = std::make_shared<MenuItemScrollBox>(MenuItemScrollBoxStyle()
            .SetSizeStyle(ui::SizeStyle()
                .SetFixedSize(boxSize)
            )
            .SetHorizontalScrollbarStyle(popStyle.GetHorizontalScrollbarStyle())
            .SetVerticalScrollbarStyle(popStyle.GetVerticalScrollbarStyle())
            .SetArrowButtonScrollDistance(popStyle.GetArrowButtonScrollDistance())
            .SetAnalogStickScrollDistanceMax(popStyle.GetAnalogStickScrollDistanceMax())
        );
        pBox->SetInnerItem(m_pListTable);

        // pBox は使い捨て。 MenuLayer が除外される際に破棄される。
        pPage->PushMenuLayerImpl(pBox, boxPos);
    }

    void MenuItemDropDownList::NotifyEntrySelectionStateChanged(int index, bool isSelected) NN_NOEXCEPT
    {
        if(isSelected)
        {
            m_SelectedEntryIndex = index;
        }
        else
        {
            if(m_SelectedEntryIndex == index)
            {
                m_SelectedEntryIndex = -1;
            }
        }
    }

    std::shared_ptr<panel::IPanel> MenuItemDropDownList::GetPanel() NN_NOEXCEPT
    {
        return m_pPanelContainer;
    }

    void MenuItemDropDownList::UpdatePanel() NN_NOEXCEPT
    {
        auto textStyle = m_Style.GetTextStyle(IsEnabled(), IsFocused(), false);
        std::string text = {};
        Position textPosition = {};
        if(m_SelectedEntryIndex >= 0 && m_SelectedEntryIndex < m_EntryList.size())
        {
            text = m_EntryList[m_SelectedEntryIndex].m_pEntry->GetText();
            textPosition = m_EntryList[m_SelectedEntryIndex].m_TextPosition;
        }

        m_pPanelContainer->SetVisibility(panel::PanelVisibility::Transparent);

        auto pos = this->GetLayoutPosition();
        auto size = this->GetLayoutSize();
        m_pPanelContainer->SetPosition(pos.x, pos.y);
        m_pPanelContainer->SetSize(size.width, size.height);

        m_pPanelText->SetVisibility(panel::PanelVisibility::Visible);
        m_pPanelText->SetSize(size.width, size.height);
        m_pPanelText->SetColor(m_Style.GetBackgroundColor(IsEnabled(), IsFocused(), false));
        m_pPanelText->SetText(text);
        m_pPanelText->SetTextPosition(textPosition.x, textPosition.y);
        m_pPanelText->SetTextColor(textStyle.GetFontColor());
        m_pPanelText->SetTextSize(textStyle.GetFontSize());

    }

}
