﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ui_MenuItemCheckBox.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include "../framework/Framework.h"
#include "util/ui_CalculateItemSize.h"
#include "util/ui_CalculateTextSize.h"
#include "util/ui_CalculateAlignedPosition.h"

#include "../Config.h"
#define UI_LOG_DEV_CHECKBOX(...) //NN_DEVOVL_UI_LOG_ROOT("[dev:cb]" __VA_ARGS__)
#define UI_DEV_CHECKBOX_TEXT() (m_pPanelText->GetText().c_str())
#define UI_DEV_CHECKBOX_STATENAME(v) (v == State_Checked ? "/" : (v == State_NotChecked ? "_" : "?"))

namespace ui{

    namespace {

        const char* GetIconText() NN_NOEXCEPT
        {
            return "\uE14B"; // チェックマーク
        }

    }

    MenuItemCheckBox::MenuItemCheckBox(const MenuItemCheckBoxStyle& style, const std::string& text) NN_NOEXCEPT
    {
        UI_LOG_DEV_CHECKBOX("create '%s'\n", text.c_str());
        m_pContainer = std::make_shared<panel::PanelContainer>();
        m_pCheckMarkContainer = std::make_shared<panel::PanelContainer>();
        m_pPanelCheckMark = std::make_shared<panel::PanelText>();
        m_pPanelText = std::make_shared<panel::PanelText>();

        m_pPanelCheckMark->SetText(GetIconText());
        m_pPanelText->SetText(text);

        m_pCheckMarkContainer->AddChild(m_pPanelCheckMark);
        m_pContainer->AddChild(m_pCheckMarkContainer);
        m_pContainer->AddChild(m_pPanelText);
        m_Style = style;
    }

    std::shared_ptr<MenuItemCheckBox> MenuItemCheckBox::SetValueAccessor(const std::function<bool ()>& getter, const std::function<void (bool)>& setter) NN_NOEXCEPT
    {
        UI_LOG_DEV_CHECKBOX("set accessor '%s'\n", UI_DEV_CHECKBOX_TEXT());
        m_ValueLoadFunction = getter;
        m_ValueStoreFunction = setter;
        return this->shared_from_this();
    }

    std::shared_ptr<MenuItemCheckBox> MenuItemCheckBox::SetValueChangedCallbackFunction(const std::function<void (bool, const std::shared_ptr<IMenuPage>&)>& f) NN_NOEXCEPT
    {
        UI_LOG_DEV_CHECKBOX("set value-callback '%s'\n", UI_DEV_CHECKBOX_TEXT());
        m_ValueChangedCallbackFunction = f;
        return this->shared_from_this();
    }

    bool MenuItemCheckBox::IsSelected() const NN_NOEXCEPT
    {
        // Unspecified は NotChecked 扱い
        return m_CheckState == State_Checked;
    }

    void MenuItemCheckBox::SetValueImpl(bool isSelected, const std::shared_ptr<IMenuPage>& pPage) NN_NOEXCEPT
    {
        UI_LOG_DEV_CHECKBOX("  set value (%s <- %s) '%s'\n",
            isSelected ? "/" : "_",
            UI_DEV_CHECKBOX_STATENAME(m_CheckState),
            UI_DEV_CHECKBOX_TEXT()
        );

        auto prev = m_CheckState;
        m_CheckState = isSelected ? State_Checked : State_NotChecked;

        if(prev != m_CheckState && m_ValueChangedCallbackFunction)
        {
            m_ValueChangedCallbackFunction(isSelected, pPage);
        }
    }

    auto MenuItemCheckBox::CalculateMarkLayoutImpl() const NN_NOEXCEPT
        -> MarkLayout
    {
        const auto markStyle = m_Style.GetCheckMarkStyle(IsEnabled(), IsFocused(), IsSelected());

        const Size boxSize = { markStyle.GetWidth(), markStyle.GetHeight() };

        const Size panelSize = {
            boxSize.width - 2 * markStyle.GetBorderWidth(),
            boxSize.height - 2 * markStyle.GetBorderWidth()
        };

        const Size markTextSize = util::CalculateTextSize::Calculate(
            m_pPanelCheckMark->GetText().c_str(),
            markStyle.GetCheckMarkSize(),
            framework::TextWriterUsage_Icon
        );

        const Position panelPosition = { markStyle.GetBorderWidth(), markStyle.GetBorderWidth() };

        const Position iconPosition = util::CalculateAlignedPosition::Calculate(markTextSize, panelSize, ui::ItemAlignment_MiddleCenter);

        MarkLayout layout = {};
        layout.m_ContainerSize    = boxSize;
        layout.m_PanelSize        = panelSize;
        layout.m_PanelPosition    = panelPosition;
        layout.m_IconPlotPosition = iconPosition;
        return layout;
    }

    auto MenuItemCheckBox::CalculateTextLayoutImpl() const NN_NOEXCEPT
        -> TextLayout
    {
        const auto textStyle = m_Style.GetTextStyle(IsEnabled(), IsFocused(), IsSelected());

        const Size textSize = util::CalculateTextSize::Calculate(
            m_pPanelText->GetText().c_str(),
            textStyle.GetFontSize(),
            framework::TextWriterUsage_Text
        );

        TextLayout layout = {};
        layout.m_TextSize = textSize;
        return layout;
    }

    void MenuItemCheckBox::UpdateLayoutRecursively(const MenuLayoutUpdateContext& context) NN_NOEXCEPT
    {
        UI_LOG_DEV_CHECKBOX("update layout '%s' ...\n", UI_DEV_CHECKBOX_TEXT());
        NN_UNUSED(context);
        const auto margin = m_Style.GetMargin();
        const auto alignment = m_Style.GetAlignment();

        auto markLayout = CalculateMarkLayoutImpl();
        auto textLayout = CalculateTextLayoutImpl();

        const int spacing = markLayout.m_ContainerSize.width / 8; // TORIAEZU

        auto& markSize = markLayout.m_ContainerSize;
        auto& textSize = textLayout.m_TextSize;

        UI_LOG_DEV_CHECKBOX("  mark-size:(%dx%d)\n", markSize.width, markSize.height);
        UI_LOG_DEV_CHECKBOX("  text-size:(%dx%d)\n", textSize.width, textSize.height);

        const Size innerSize = { markSize.width + spacing + textSize.width, std::max(markSize.height, textSize.height) };
        UI_LOG_DEV_CHECKBOX("  inner-size    :(%dx%d)\n", innerSize.width, innerSize.height);

        const Position markPanelPosition = util::CalculateAlignedPosition::Calculate(markSize, {markSize.width, innerSize.height}, alignment);

        const Position textPanelPosition = Position{markSize.width + spacing, 0}; // 高さはプロット位置で調整するので一番上に貼り付けておく
        const Size     textPanelSize = {textSize.width, innerSize.height};
        const Position textPlotPosition = util::CalculateAlignedPosition::Calculate(textSize, textPanelSize, alignment);

        const Size layoutSize = {
            innerSize.width  + margin.left + margin.right,
            innerSize.height + margin.top + margin.bottom
        };
        const Position innerOrigin = { margin.left, margin.top };

        const auto outerSize = util::CalculateItemSize::Calculate([&](){ return layoutSize; }, m_Style.GetSizeStyle());
        const auto layoutPosition = util::CalculateAlignedPosition::Calculate(layoutSize, outerSize, m_Style.GetAlignment());

        SetLayoutSize(outerSize);
        m_Layout.m_MarkLayout = markLayout;
        m_Layout.m_MarkPanelPosition = layoutPosition + innerOrigin + markPanelPosition;
        m_Layout.m_TextPanelPosition = layoutPosition + innerOrigin + textPanelPosition;
        m_Layout.m_TextPanelSize     = textPanelSize;
        m_Layout.m_TextPlotPosition  = textPlotPosition;

        //UI_LOG_DEV_CHECKBOX("  outer-size         :(%dx%d)\n", outerSize.width, outerSize.height);
        //UI_LOG_DEV_CHECKBOX("  icon-panel-size    :(%dx%d)\n", m_Layout.m_IconPanelSize.width, m_Layout.m_IconPanelSize.height);
        //UI_LOG_DEV_CHECKBOX("  text-panel-size    :(%dx%d)\n", m_Layout.m_TextPanelSize.width, m_Layout.m_TextPanelSize.height);
        //UI_LOG_DEV_CHECKBOX("  icon-panel-position:(%d,%d)\n", m_Layout.m_IconPanelPosition.x, m_Layout.m_IconPanelPosition.y);
        //UI_LOG_DEV_CHECKBOX("  text-panel-position:(%d,%d)\n", m_Layout.m_TextPanelPosition.x, m_Layout.m_TextPanelPosition.y);
        //UI_LOG_DEV_CHECKBOX("  icon-plot-position :(%d,%d)\n", iconPlotPosition.x, iconPlotPosition.y);
        //UI_LOG_DEV_CHECKBOX("  text-plot-position :(%d,%d)\n", textPlotPosition.x, textPlotPosition.y);
        UI_LOG_DEV_CHECKBOX("update layout '%s' ... done\n", UI_DEV_CHECKBOX_TEXT());
    }


    bool MenuItemCheckBox::IsFocusAcceptable() const NN_NOEXCEPT
    {
        return true;
    }

    bool MenuItemCheckBox::IsTouchAcceptable() const NN_NOEXCEPT
    {
        return true;
    }

    MenuButtonResult MenuItemCheckBox::HandlePositiveButtonPressed(const MenuButtonHandleContext& context) NN_NOEXCEPT
    {
        UI_LOG_DEV_CHECKBOX("handle positive-button '%s'\n", UI_DEV_CHECKBOX_TEXT());
        if(context.GetCurrentFocusedItem() == this->shared_from_this())
        {
            SetValueImpl(!IsSelected(), context.GetPage());
        }
        else
        {
            context.GetPage()->ChangeFocus(this->shared_from_this());
        }
        return MenuButtonResult::GetHandled();
    }

    MenuButtonResult MenuItemCheckBox::HandleTouched(const MenuButtonHandleContext& context, const nn::util::Vector3f&) NN_NOEXCEPT
    {
        UI_LOG_DEV_CHECKBOX("handle touched '%s'\n", UI_DEV_CHECKBOX_TEXT());
        if(context.GetCurrentFocusedItem() == this->shared_from_this())
        {
            SetValueImpl(!IsSelected(), context.GetPage());
        }
        else
        {
            context.GetPage()->ChangeFocus(this->shared_from_this());
        }
        return MenuButtonResult::GetHandled();
    }

    void MenuItemCheckBox::LoadValue(const std::shared_ptr<IMenuPage>& pPage) NN_NOEXCEPT
    {
        UI_LOG_DEV_CHECKBOX("load '%s'\n", UI_DEV_CHECKBOX_TEXT());
        if(m_ValueLoadFunction)
        {
            SetValueImpl(m_ValueLoadFunction(), pPage);
        }
    }

    void MenuItemCheckBox::StoreValue() const NN_NOEXCEPT
    {
        UI_LOG_DEV_CHECKBOX("store '%s'\n", UI_DEV_CHECKBOX_TEXT());
        if(m_ValueStoreFunction)
        {
            m_ValueStoreFunction(IsSelected());
        }
    }

    std::shared_ptr<panel::IPanel> MenuItemCheckBox::GetPanel() NN_NOEXCEPT
    {
        return m_pContainer;
    }

    void MenuItemCheckBox::UpdatePanel() NN_NOEXCEPT
    {
        auto pos = this->GetLayoutPosition();
        auto size = this->GetLayoutSize();
        auto bgcolor = m_Style.GetBackgroundColor(IsEnabled(), IsFocused(), IsSelected());
        m_pContainer->SetVisibility(panel::PanelVisibility::Visible);
        m_pContainer->SetColor(bgcolor);
        m_pContainer->SetPosition(pos.x, pos.y);
        m_pContainer->SetSize(size.width, size.height);

        // CheckMark
        {
            auto style = m_Style.GetCheckMarkStyle(IsEnabled(), IsFocused(), IsSelected());
            auto color = style.GetCheckMarkColor();
            auto fontSize = style.GetCheckMarkSize();
            auto& markLayout = m_Layout.m_MarkLayout;

            m_pPanelCheckMark->SetVisibility(panel::PanelVisibility::Visible);
            m_pPanelCheckMark->SetColor(style.GetBackgroundColor());
            m_pPanelCheckMark->SetPosition(markLayout.m_PanelPosition.x, markLayout.m_PanelPosition.y);
            m_pPanelCheckMark->SetSize(markLayout.m_PanelSize.width, markLayout.m_PanelSize.height);
            m_pPanelCheckMark->SetTextColor(color);
            m_pPanelCheckMark->SetTextPosition(markLayout.m_IconPlotPosition.x, markLayout.m_IconPlotPosition.y);
            m_pPanelCheckMark->SetTextSize(fontSize);

            m_pCheckMarkContainer->SetVisibility(panel::PanelVisibility::Visible);
            m_pCheckMarkContainer->SetColor(style.GetBorderColor());
            m_pCheckMarkContainer->SetPosition(m_Layout.m_MarkPanelPosition.x, m_Layout.m_MarkPanelPosition.y);
            m_pCheckMarkContainer->SetSize(markLayout.m_ContainerSize.width, markLayout.m_ContainerSize.height);
        }

        // Text
        {
            auto style = m_Style.GetTextStyle(IsEnabled(), IsFocused(), IsSelected());
            auto color = style.GetFontColor();
            auto fontSize = style.GetFontSize();
            m_pPanelText->SetVisibility(panel::PanelVisibility::Visible);
            m_pPanelText->SetColor(bgcolor);
            m_pPanelText->SetPosition(m_Layout.m_TextPanelPosition.x, m_Layout.m_TextPanelPosition.y);
            m_pPanelText->SetSize(m_Layout.m_TextPanelSize.width, m_Layout.m_TextPanelSize.height);
            m_pPanelText->SetTextColor(color);
            m_pPanelText->SetTextPosition(m_Layout.m_TextPlotPosition.x, m_Layout.m_TextPlotPosition.y);
            m_pPanelText->SetTextSize(fontSize);
        }
    }

}

