﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

class AppletMessageMenuResponse;

namespace ui{
    class IMenuItem;

    class IMenuPage
    {
    public:
        typedef uint32_t PageIndexType;

    public:
        virtual ~IMenuPage() NN_NOEXCEPT
        {
        };

        // メニューを閉じます。すべての親ページが閉じられます。
        virtual void CloseMenu() NN_NOEXCEPT = 0;

        // 現在のページを閉じます。親ページに移動します。
        virtual void ClosePage() NN_NOEXCEPT = 0;

        // 子ページを開きます
        virtual void OpenPage(PageIndexType pageIndex) NN_NOEXCEPT = 0;
        virtual void OpenPage(PageIndexType pageIndex, const std::shared_ptr<void>& pUserArgument) NN_NOEXCEPT = 0;

        // フォーカスを移動します。
        virtual void ChangeFocus(const std::shared_ptr<IMenuItem>& newItem) NN_NOEXCEPT = 0;

        // システムをスリープさせます。メニューは閉じられます。
        virtual void SleepSystem() NN_NOEXCEPT = 0;

        // システムをシャットダウンさせます。メニューは閉じられます。
        virtual void ShutdownSystem() NN_NOEXCEPT = 0;

        // メッセージを発生させます。メニューは閉じられます。
        virtual void SendMessage(const AppletMessageMenuResponse& message) NN_NOEXCEPT = 0;

        // UI の状態を外部から読み込んだ値に設定します。
        //
        // チェックボックスやドロップダウンリストなどの選択状態を持つ UI に対して
        // 設定された getter 関数を呼び出します。
        // getter 関数が設定されていない UI に対しては何も行いません。
        //
        // 各 MenuPage の OnEnteringPage() 関数内で呼び出した場合のみ読込を行います。
        // それ以外の関数から呼び出した場合は無視されます。
        virtual void LoadValue() NN_NOEXCEPT = 0;

        // UI の状態を外部に書き出します。
        //
        // チェックボックスやドロップダウンリストなどの選択状態を持つ UI に対して
        // 設定された setter 関数を呼び出します。
        // setter 関数が設定されていない UI に対しては何も行いません。
        virtual void StoreValue() NN_NOEXCEPT = 0;

        // UI のレイアウトの再計算を要求します。
        //
        // レイアウトの再計算は負荷が高いため必要なときにのみ行うようにします。
        // ユーザーカスタム (UserCustom) のサイズが変更された場合、明示的にこの関数を呼び出して再計算を行います。
        // それ以外の UI については必要に応じて自動的にレイアウトの再計算が行われるためこの関数の呼出は不要です。
        //
        // TODO:
        // と言いつつ、今の実装では状態に応じて UI のレイアウトが変化する場合、手動でこの関数を呼び出さないといけない。
        // レイアウトの再計算を複数回要求したとしても実際に再計算されるのは 1 回だけなので、必要なら手動で呼び出してしまっても構わない。
        virtual void UpdateLayout() NN_NOEXCEPT = 0;

        // パネルの更新を要求します。
        //
        // ユーザーカスタム (UserCustom) の描画内容が変更された場合、明示的にこの関数を呼び出してパネルの更新を行います。
        // それ以外の UI については必要に応じて自動的にパネルの更新が行われるためこの関数の呼出は不要です。
        //
        // TODO:
        // と言いつつ、今の実装では暗黙的に毎フレームパネルの更新を要求しているのだが…。
        // UpdateLayout に比べれば軽い処理なので気が向いたら端折る。
        virtual void UpdatePanel() NN_NOEXCEPT = 0;

        // 指定した UI の有効状態 (IsEnabled) を変更します。
        //
        // 複数の UI に対して呼び出した場合、呼び出したすべての UI の状態が変更されます。
        // 同一の UI に対して複数回呼び出した場合、最後の呼出の設定値が適用されます。
        virtual void SetEnablity(const std::shared_ptr<ui::IMenuItem>& pItem, bool isEnabled) NN_NOEXCEPT = 0;

    public:
        // （内部実装用）現在のメニューの上にメニューアイテムを重ねます。
        // この関数は UI の実装用です。コールバック関数から呼び出さないでください。
        // 子ページを開く場合、 OpenPage() を使用します。
        //
        // 重ねたメニューから PushMenuLayerImpl(), PopMenuLayerImpl() 以外を呼び出した場合の動作は未定義。
        virtual void PushMenuLayerImpl(
            const std::shared_ptr<IMenuItem>& pItem,
            const Position& anchor
        ) NN_NOEXCEPT = 0;

        // （内部実装用）現在のメニューアイテムを除外します。
        // この関数は UI の実装用です。コールバック関数から呼び出さないでください。
        // ページを閉じる場合、 ClosePage() を使用します。
        //
        // 重ねていないメニューからこの関数を呼び出した場合の動作は未定義。
        virtual void PopMenuLayerImpl() NN_NOEXCEPT = 0;

    };
}
