﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once
#include <nn/nn_Common.h>
#include <string>

namespace scene { namespace debug { namespace logutil {

/*!
    @brief      UTF-8 文字列のユーティリティクラスです。

    @details
                本クラスでは、以下の検証を行います。

                - バイトシーケンスの正当性
                - 完全な文字列であるかどうかの確認
                - 冗長なエンコードが存在しないかの確認

                文字列を 1 バイトずつ入力することができるため、文字列データをすべてメモリ上に展開する必要はありません。

                1 バイトずつ入力したい場合、 @ref Utf8StreamUtility::Push で入力します。@n
                入力完了時に @ref Utf8StreamUtility::End を呼び出すことで、完全な文字列であるかどうかの確認が行われます。
*/
class Utf8StreamUtility
{
public:
    /*!
        @brief      コンストラクタです。
    */
    Utf8StreamUtility() NN_NOEXCEPT;

    /*!
        @brief      クリアします。
    */
    void Clear() NN_NOEXCEPT;

    /*!
        @brief      1 文字入力します。

        @param[in]  c   文字。

        @return     正しい UTF-8 文字列が入力され続けているかどうか。

        @details
                    終端文字を入力した場合、 @ref End の結果を返します。@n
                    不正なバイトシーケンスや冗長なエンコードの存在を確認した場合、false を返します。
    */
    bool Push(char c) NN_NOEXCEPT;

    /*!
        @brief      入力の完了を宣言します。

        @return     正しい UTF-8 文字列が入力され続けていたかどうか。

        @details
                    完全な文字列でなかった場合、false を返します。
    */
    bool End() NN_NOEXCEPT;

    /*!
        @brief      文字数を取得します。

        @return     文字数。

        @details
                    本クラスは、入力された文字列から文字数をカウントします。@n
                    入力されたバイト数ではないことに注意してください。
    */
    int32_t GetCount() NN_NOEXCEPT;

public:
    /*!
        @brief      文字列を検査します。

        @param[in]  string  文字列。

        @return     正しい UTF-8 文字列かどうか。

        @pre
            - string != nullptr
    */
    static bool Verify(const char* string) NN_NOEXCEPT;

    /*!
        @brief      文字列を検査します。

        @param[in]  outCount    文字数。
        @param[in]  string      文字列。

        @return     正しい UTF-8 文字列かどうか。

        @pre
            - string != nullptr
    */
    static bool Verify(int* outCount, const char* string) NN_NOEXCEPT;

    /*!
        @brief      指定したバイト数の文字列を検査します。

        @param[in]  string  文字列。
        @param[in]  length  文字列のバイト数。

        @return     正しい UTF-8 文字列かどうか。

        @pre
            - string != nullptr
    */
    static bool Verify(const char* string, size_t length) NN_NOEXCEPT;

    /*!
        @brief      指定したバイト数の文字列を検査します。

        @param[in]  outCount    文字数。
        @param[in]  string      文字列。
        @param[in]  length      文字列のバイト数。

        @return     正しい UTF-8 文字列かどうか。

        @pre
            - string != nullptr
    */
    static bool Verify(int* outCount, const char* string, size_t length) NN_NOEXCEPT;

    /*!
    @brief      バッファに収まる範囲内で、指定した最大文字数分の文字列をコピーします。

    @param[in]  output      コピー先のバッファ。
    @param[in]  outputSize  コピー先のバッファサイズ。
    @param[in]  input       コピー元。
    @param[in]  maxCount    コピーする最大文字数。

    @return     コピーしたバイト数。

    @pre
    - output != nullptr
    - outputSize > 0
    - input != nullptr
    - maxCount > 0

    @details
    不正な UTF-8 文字列を指定した場合、-1 が返ります。

    マルチバイト文字のバイトシーケンスが未完了の状態で終端文字を検知した場合、不正とは判定されません。@n
    未完了のバイト列は無視されます。

    例：{'\xE3', '\x81', '\x82', '\xE3', '\x81', 0}; // 「あ」＋「い」の 2 バイト目まで。

    上記の例では、maxCount を 1 に指定した場合「あ」の 3 バイトだけコピーされます。
    */
    static int Substring(std::string* output, size_t outputSize, const char* input, int maxCount) NN_NOEXCEPT;

    /*!
    @brief      バッファに収まる範囲内で、指定した最大文字数分の文字列を切り出し、文字列を挿すポインタを進めます。

    @param[in]  output      切り出す先のバッファ。
    @param[in]  outputSize  切り出す先のバッファサイズ。
    @param[in]  input       切り出す元のポインタ。切り出した直後のポインタを返す。
    @param[in]  maxCount    切り出す最大文字数。

    @return     切り出したバイト数。

    @pre
    - output != nullptr
    - outputSize > 0
    - input != nullptr
    - *input != nullptr
    - maxCount > 0

    @details
    不正な UTF-8 文字列を指定した場合、-1 が返ります。

    マルチバイト文字のバイトシーケンスが未完了の状態で終端文字を検知した場合、不正とは判定されません。@n
    未完了のバイト列は無視されます。

    例：{'\xE3', '\x81', '\x82', '\xE3', '\x81', 0}; // 「あ」＋「い」の 2 バイト目まで。

    上記の例では、maxCount を 1 に指定した場合「あ」の 3 バイトだけコピーされます。
    */
    static int Slice(std::string* output, size_t outputSize, const char** input, int maxCount) NN_NOEXCEPT;

    static int GetBytesSizeOfWidth(const char* input, int maxCount) NN_NOEXCEPT;

    static int SliceSubstring(std::string* output, size_t outputSize, const char* input, const std::string separator, int maxCount) NN_NOEXCEPT;

private:
    //
    nn::Bit8 m_Code[4];
    //
    int32_t m_Count;
    int32_t m_Bytes;
    int32_t m_ExpectedBytes;
};

}}}
