﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <string>

#include <nn/nn_Assert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/font/font_ScalableFont.h>

#include "TextReader.h"
#include "Utf8StreamUtility.h"

namespace scene { namespace debug { namespace logutil {

TextReader::TextReader() NN_NOEXCEPT
    : m_ReaderImpl()
    , m_LineWidthMax(0)
    , m_CharWidthGetterFunction(nullptr)
    , m_CharWidthGetterUserPtr(nullptr)
{
}

void TextReader::Initialize(
    const char* binaryData,
    size_t dataSize,
    float lineWidthMax,
    float (*charWidthGetterFunction)(const char*, size_t, void*),
    void* charWidthGetterUserPtr
) NN_NOEXCEPT
{
    m_ReaderImpl = std::make_unique<TextReaderImpl>(binaryData, dataSize);

    m_LineWidthMax = lineWidthMax;
    m_CharWidthGetterFunction = charWidthGetterFunction;
    m_CharWidthGetterUserPtr  = charWidthGetterUserPtr;
}

float TextReader::GetCharWidth(const char* character, size_t length) const NN_NOEXCEPT
{
    if(m_CharWidthGetterFunction)
    {
        return m_CharWidthGetterFunction(character, length, m_CharWidthGetterUserPtr);
    }
    else
    {
        return 0;
    }
}

// offset から 1 文字を取得
nn::Result TextReader::ReadUtf8Character(std::string* pOutChar, int64_t* pOutNextOffset, int64_t offset) NN_NOEXCEPT
{
    pOutChar->clear();
    *pOutNextOffset = offset;

    Utf8StreamUtility utf8Stream;
    size_t byteSizeSum = 0;

    // Utf-8最大4バイト分保持用
    char buffer[4] = {};

    while (!m_ReaderImpl->IsEof(offset + byteSizeSum))
    {
        // 1 バイト取得
        NN_RESULT_DO(m_ReaderImpl->GetByte(&(buffer[byteSizeSum]), offset + byteSizeSum));

        // 不正な UTF-8 シーケンス、またはヌル終端文字である場合
        if (false == utf8Stream.Push(buffer[byteSizeSum]) || '\0' == buffer[byteSizeSum])
        {
            // Utf-8 の開始バイトを "\\xFF" の形にサニタイジング
            *pOutChar += Sanitize(buffer[0]);
            *pOutNextOffset = offset + 1;
            NN_RESULT_SUCCESS;
        }

        // utf8 の1文字の最後の 1バイト が追加された
        if (utf8Stream.GetCount() == 1)
        {
            *pOutChar += std::string(buffer, byteSizeSum + 1);
            *pOutNextOffset = offset + byteSizeSum + 1;
            NN_RESULT_SUCCESS;
        }

        byteSizeSum++;

        NN_ASSERT(4 != byteSizeSum, "The logreader::Utf8StreamUtility::Push() cannot classify the utf-8.");
    }

    // 2バイト目以降で EOF に達したとき
    // TODO:サニタイズされない処理に
    if (byteSizeSum != 0)
    {
        *pOutChar += Sanitize(buffer[0]);
        *pOutNextOffset = offset + 1;
    }

    NN_RESULT_SUCCESS;
}

// オフセットから 1 行を取得
nn::Result TextReader::ReadLine(std::string* pOutLine, int64_t* pOutNextOffset, int64_t offset) NN_NOEXCEPT
{
    pOutLine->clear();
    *pOutNextOffset = offset;

    std::string oneUtf8Character;
    int64_t lastOffset;
    float lineWidth = 0;
    while (!m_ReaderImpl->IsEof(*pOutNextOffset))
    {
        // 1 文字読む
        NN_RESULT_DO(ReadUtf8Character(&oneUtf8Character, &lastOffset, *pOutNextOffset));

        float charWidth = GetCharWidth(oneUtf8Character.c_str(), oneUtf8Character.size());

        if (oneUtf8Character[0] == '\n')        // \n (最大文字幅に達した直後の \n も含む)
        {
            *pOutNextOffset = lastOffset;
            break;
        }
        else if (oneUtf8Character[0] == '\r')   // \r (最大文字幅に達した直後の \r も含む)
        {
            // オフセット更新のみ、読み飛ばし扱い
            *pOutNextOffset = lastOffset;
        }
        else if (lineWidth == 0 && charWidth > m_LineWidthMax) // 1 文字で最大文字幅を超える場合は採用する
        {
            *pOutNextOffset = lastOffset;
            *pOutLine += oneUtf8Character;
            lineWidth += charWidth;
        }
        else if (lineWidth + charWidth > m_LineWidthMax) // 合計で最大文字幅を超える場合はキャンセル。
        {
            break;
        }
        else
        {
            *pOutNextOffset = lastOffset;
            *pOutLine += oneUtf8Character;
            lineWidth += charWidth;
        }
    }

    NN_RESULT_SUCCESS;
}

//float TextReader::GetCharWidth(const char* utf8Char) NN_NOEXCEPT
//{
//    uint32_t utf32Char = 0;
//    auto result = nn::util::ConvertCharacterUtf8ToUtf32(&utf32Char, utf8Char);
//    NN_ASSERT(result == nn::util::CharacterEncodingResult_Success);
//    NN_UNUSED(result);
//
//    return m_ScalableFont.GetCharWidth(utf32Char) * m_FontScale;
//}

std::string Sanitize(char inputData) NN_NOEXCEPT
{
    // Utf-8 の開始バイトを "\\xFF" の形にサニタイジング
    char sanitizingBuffer[5];
    nn::util::SNPrintf(sanitizingBuffer, 5, "\\x%02X", inputData & 0x000000FF);
    return std::string(sanitizingBuffer, 4);
}


}}}
