﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <vector>
#include <memory>
#include <functional>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include "debug_ActivityCommon.h"
#include "../../panel/panel_IPanel.h"
#include "../../panel/panel_PanelContainer.h"
#include "../../panel/panel_PanelText.h"
#include "../../ui/ui_Types.h"

namespace scene{ namespace debug{

    class LayoutEditor
    {
    public:
        struct ModuleHandle
        {
        public:
            bool IsValid() const NN_NOEXCEPT { return this->value >= 0; }
            bool IsInvalid() const NN_NOEXCEPT { return !this->IsValid(); }

        public:
            int64_t value = -1;
        };

    private:
        struct Entry
        {
            std::shared_ptr<ActivityCommon> pCommon;
            std::function<ui::Size ()> sizeGetterFunction;

            std::shared_ptr<panel::PanelText> pPanel;
            bool isActive = false;
        };

    public:
        LayoutEditor() NN_NOEXCEPT;
        void SetViewSize(const ui::Size& size) NN_NOEXCEPT;
        void SetCanvasSize(const ui::Size& size) NN_NOEXCEPT;
        void SetCanvasColor(const nn::util::Color4f& value) NN_NOEXCEPT;
        void SetModuleColorInactive(const nn::util::Color4f& value) NN_NOEXCEPT;
        void SetModuleColorActive(const nn::util::Color4f& value) NN_NOEXCEPT;

        ui::Size GetCanvasSize() const NN_NOEXCEPT;

        ModuleHandle RegisterModule(
            const std::shared_ptr<ActivityCommon>& pCommon,
            const std::function<ui::Size ()>& sizeGetterFunction
        ) NN_NOEXCEPT;

        template<typename ModuleActivity>
        ModuleHandle RegisterModule(const std::shared_ptr<ModuleActivity>& pActivity) NN_NOEXCEPT
        {
            return RegisterModule(
                std::shared_ptr<ActivityCommon>(pActivity, &pActivity->common),
                [pActivity]() -> ui::Size { return {pActivity->GetWidth(), pActivity->GetHeight()}; }
            );
        }

        ModuleHandle FindModule(const char* name) const NN_NOEXCEPT;

        ui::Size GetModuleSize(ModuleHandle h) const NN_NOEXCEPT;
        ui::Position GetModulePosition(ModuleHandle h) const NN_NOEXCEPT;
        void SetModulePosition(ModuleHandle h, const ui::Position& pos) NN_NOEXCEPT;
        void MoveModule(ModuleHandle h, const ui::Position& dpos) NN_NOEXCEPT;
        void SetModuleActivity(ModuleHandle h, bool value) NN_NOEXCEPT;

        ui::Size UpdateLayout() NN_NOEXCEPT;
        void UpdatePanel(const ui::Position& position) NN_NOEXCEPT;
        void UpdateInternalPanel() NN_NOEXCEPT;
        std::shared_ptr<panel::IPanel> GetPanel() NN_NOEXCEPT;

    private:
        std::vector<Entry> m_EntryList;

        std::shared_ptr<panel::PanelContainer> m_pContainer;
        ui::Size m_ViewSize = {1280, 720};
        ui::Size m_CanvasSize = {1280, 720};

        nn::util::Color4f m_CanvasColor         = {0, 0, 0, 1};
        nn::util::Color4f m_ModuleColorInactive = {1, 1, 1, 1};
        nn::util::Color4f m_ModuleColorActive   = {1, 0 ,0, 1};

    };

}}
