﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "debug_HistoricalGraph.h"

#include <new>
#include <limits>
#include <nn/nn_Assert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

namespace scene{ namespace debug{

    const size_t HistoricalGraph::RequiredMemorySizePerSeries;
    const size_t HistoricalGraph::RequiredMemoryAlignment;

    HistoricalGraph::SeriesEntry::SeriesEntry() NN_NOEXCEPT
    {
        m_PointPanel = std::make_shared<panel::PanelText>();
        m_PointPanel->SetVisibility(panel::PanelVisibility::Invisible);

        m_ValueMin = 0;
        m_ValueMax = 1;
        m_LineWidth = 1;
        m_LineColor = nn::util::Color4f(1, 1, 1, 1);

        m_CurrentValue = std::numeric_limits<float>::quiet_NaN();
        m_PreviousValue = std::numeric_limits<float>::quiet_NaN();
        m_PreviousYMin = 0;
        m_PreviousYMax = 0;
    }

    void HistoricalGraph::SeriesEntry::SetLineWidth(int value) NN_NOEXCEPT
    {
        m_LineWidth = std::max(value, 0);
    }

    void HistoricalGraph::SeriesEntry::SetLineColor(const nn::util::Color4f& value) NN_NOEXCEPT
    {
        m_LineColor = value;
    }

    void HistoricalGraph::SeriesEntry::SetValueRange(float valueMin, float valueMax) NN_NOEXCEPT
    {
        m_ValueMin = valueMin;
        m_ValueMax = valueMax;
    }

    void HistoricalGraph::SeriesEntry::ResetValue() NN_NOEXCEPT
    {
        m_CurrentValue = std::numeric_limits<float>::quiet_NaN();
        m_PreviousValue = std::numeric_limits<float>::quiet_NaN();
    }

    void HistoricalGraph::SeriesEntry::PushValue(float value) NN_NOEXCEPT
    {
        m_CurrentValue = value;
    }

    void HistoricalGraph::SeriesEntry::Update(int width, int height, int scrollSpeed) NN_NOEXCEPT
    {
        float current = m_CurrentValue;

        bool isVisible = true;

        int drawX = 0;
        int drawW = 0;
        int drawYMin = 0;
        int drawYMax = 0;
        if(
            isnan(current)
            || width <= 0
            || height <= 0
            || scrollSpeed == 0 // TORIAEZU: スクロール速度が 0 のときは描画しない
            || m_ValueMin == m_ValueMax
            || m_LineWidth <= 0
            )
        {
            isVisible = false;
        }
        else do
        {
            // 横方向を計算
            if(scrollSpeed > 0)
            {
                drawX = width - scrollSpeed; // 右端
                drawW = scrollSpeed;
            }
            else
            {
                drawX = 0;
                drawW = -scrollSpeed; // 左端
            }

            // 縦方向を計算
            {
                int curYMin = 0;
                int curYMax = 0;
                float tmpY = (current - m_ValueMin) * height / (m_ValueMax - m_ValueMin);
                if(isnan(tmpY))
                {
                    isVisible = false;
                    break;
                }
                else
                {
                    tmpY = std::max<float>(tmpY, -m_LineWidth);
                    tmpY = std::min<float>(tmpY, height + m_LineWidth);

                    curYMin = height - 1 - static_cast<int>(tmpY);
                    curYMax = curYMin + m_LineWidth - 1;
                }

                // グラフが連続するように描画範囲を広げる
                if(!isnan(m_PreviousValue))
                {
                    if(curYMin > m_PreviousYMax)
                    {
                        curYMin = m_PreviousYMax + 1;
                    }

                    if(curYMax < m_PreviousYMin)
                    {
                        curYMax = m_PreviousYMin - 1;
                    }
                    NN_ASSERT(curYMax - curYMin >= m_LineWidth - 1);
                }
                drawYMin = curYMin;
                drawYMax = curYMax;
            }
        }while(NN_STATIC_CONDITION(false));

        if(isVisible)
        {
            m_PointPanel->SetVisibility(panel::PanelVisibility::Visible);
            m_PointPanel->SetColor(m_LineColor);
            m_PointPanel->SetPosition(drawX, drawYMin);
            m_PointPanel->SetSize(drawW, drawYMax - drawYMin + 1);
            m_PreviousValue = m_CurrentValue;
            m_PreviousYMin = drawYMin;
            m_PreviousYMax = drawYMax;
        }
        else
        {
            m_PointPanel->SetVisibility(panel::PanelVisibility::Invisible);
            m_PreviousValue = std::numeric_limits<float>::quiet_NaN();
        }
    }

    std::shared_ptr<panel::IPanel> HistoricalGraph::SeriesEntry::GetPanel() NN_NOEXCEPT
    {
        return m_PointPanel;
    }

    //---------------------------------

    HistoricalGraph::HistoricalGraph() NN_NOEXCEPT
    {
        m_EntryCount = 0;
        m_pEntryList = nullptr;
        ResetParameterImpl();

        m_RootPanel = std::make_shared<panel::PanelContainer>();
        m_RootPanel->SetVisibility(panel::PanelVisibility::Transparent);
        m_RootPanel->SetSize(0, 0);
    }

    void HistoricalGraph::Initialize(int entryCount, void* memory, size_t memorySize) NN_NOEXCEPT
    {
        NN_ASSERT_GREATER_EQUAL(memorySize, entryCount * RequiredMemorySizePerSeries);
        NN_ASSERT_ALIGNED(memory, RequiredMemoryAlignment);

        m_HistoryPanel = std::make_shared<panel::PanelHistorical>();
        m_HistoryPanel->SetVisibility(panel::PanelVisibility::Visible);

        char* p = reinterpret_cast<char*>(memory);
        for(int i = 0; i < entryCount; i++)
        {
            new(p + sizeof(SeriesEntry) * i) SeriesEntry();
        }

        m_EntryCount = entryCount;
        m_pEntryList = reinterpret_cast<SeriesEntry*>(memory);
        ResetParameterImpl();

        m_RootPanel->ClearChildren();
        m_RootPanel->AddChild(m_HistoryPanel);
        for(int i = 0; i < entryCount; i++)
        {
            m_RootPanel->AddChild(m_pEntryList[i].GetPanel());
        }

    }

    void HistoricalGraph::Finalize() NN_NOEXCEPT
    {
        m_RootPanel->ClearChildren();

        for(int i = 0; i < m_EntryCount; i++)
        {
            (m_pEntryList + i)->~SeriesEntry();
        }
        m_HistoryPanel.reset();

        m_EntryCount = 0;
        m_pEntryList = nullptr;
        ResetParameterImpl();
    }

    void HistoricalGraph::ResetParameterImpl() NN_NOEXCEPT
    {
        m_PositionX = 0;
        m_PositionY = 0;
        m_Width = 0;
        m_Height = 0;
        m_ScroollSpeed = 1;
        m_BackgroundColor = nn::util::Color4f(0, 0, 0, 0);
    }

    void HistoricalGraph::SetPosition(int x, int y) NN_NOEXCEPT
    {
        m_PositionX = x;
        m_PositionY = y;
    }

    void HistoricalGraph::SetSize(int width, int height) NN_NOEXCEPT
    {
        m_Width = width;
        m_Height = height;
    }

    void HistoricalGraph::SetBackgroundColor(const nn::util::Color4f& value) NN_NOEXCEPT
    {
        m_BackgroundColor = value;
    }

    void HistoricalGraph::SetScrollSpeed(int value) NN_NOEXCEPT
    {
        m_ScroollSpeed = value;
    }

    void HistoricalGraph::SetEntryLineWidth(int entryIndex, int value) NN_NOEXCEPT
    {
        if(entryIndex < 0 || entryIndex >= m_EntryCount)
        {
            return;
        }

        m_pEntryList[entryIndex].SetLineWidth(value);
    }

    void HistoricalGraph::SetEntryLineColor(int entryIndex, const nn::util::Color4f& value) NN_NOEXCEPT
    {
        if(entryIndex < 0 || entryIndex >= m_EntryCount)
        {
            return;
        }

        m_pEntryList[entryIndex].SetLineColor(value);
    }

    void HistoricalGraph::SetValueRange(int entryIndex, float valueMin, float valueMax) NN_NOEXCEPT
    {
        if(entryIndex < 0 || entryIndex >= m_EntryCount)
        {
            return;
        }

        m_pEntryList[entryIndex].SetValueRange(valueMin, valueMax);
    }

    void HistoricalGraph::ResetEntryValue(int entryIndex) NN_NOEXCEPT
    {
        if(entryIndex < 0 || entryIndex >= m_EntryCount)
        {
            return;
        }

        m_pEntryList[entryIndex].ResetValue();
    }

    void HistoricalGraph::PushEntryValue(int entryIndex, float value) NN_NOEXCEPT
    {
        if(entryIndex < 0 || entryIndex >= m_EntryCount)
        {
            return;
        }

        m_pEntryList[entryIndex].PushValue(value);
    }

    void HistoricalGraph::Update() NN_NOEXCEPT
    {
        m_RootPanel->SetPosition(m_PositionX, m_PositionY);
        m_RootPanel->SetSize(m_Width, m_Height);

        m_HistoryPanel->SetSize(m_Width, m_Height);
        m_HistoryPanel->SetColor(m_BackgroundColor);
        m_HistoryPanel->SetOffset(-m_ScroollSpeed, 0);

        for(int i = 0; i < m_EntryCount; i++)
        {
            m_pEntryList[i].Update(m_Width, m_Height, m_ScroollSpeed);
        }
    }

    std::shared_ptr<panel::IPanel> HistoricalGraph::GetPanel() NN_NOEXCEPT
    {
        return m_RootPanel;
    }

    void HistoricalGraph::SetPanelName(const std::string& value) NN_NOEXCEPT
    {
        m_RootPanel->SetPanelName(value);
    }

}}

