﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "SceneStack.h"

#include "SceneFactory.h"

namespace scene{

    SceneStack::SceneStack() NN_NOEXCEPT
    {
    }

    SceneStack::SceneStack(SceneIndexType initialScene, const std::shared_ptr<void>& initialArg) NN_NOEXCEPT
    {
        PushScene(initialScene, initialArg);
    }

    bool SceneStack::IsEmpty() const NN_NOEXCEPT
    {
        return m_Stack.size() == 0;
    }

    void SceneStack::Clear() NN_NOEXCEPT
    {
        // 後ろから破棄する
        while(m_Stack.size() > 0)
        {
            m_Stack.back()->OnDestroying();
            m_Stack.pop_back();
        }
    }

    void SceneStack::ReplaceScene(SceneIndexType scene, const std::shared_ptr<void>& arg) NN_NOEXCEPT
    {
        //NN_SDK_LOG("replace scene %llu ->\n", m_Stack.size());
        if(m_Stack.size() > 0)
        {
            m_Stack.back()->OnDestroying();
            m_Stack.pop_back();
        }
        {
            auto pScene = SceneFactory::Create(scene, arg);
            NN_SDK_ASSERT_NOT_NULL(pScene); // TORIAEZU
            m_Stack.push_back(pScene);
            pScene->OnCreated();
        }
    }

    void SceneStack::PushScene(SceneIndexType scene, const std::shared_ptr<void>& arg) NN_NOEXCEPT
    {
        //NN_SDK_LOG("push scene %llu ->\n", m_Stack.size());
        auto pScene = SceneFactory::Create(scene, arg);
        NN_SDK_ASSERT_NOT_NULL(pScene); // TORIAEZU
        m_Stack.push_back(pScene);
        pScene->OnCreated();
    }

    void SceneStack::PopScene() NN_NOEXCEPT
    {
        //NN_SDK_LOG("pop scene %llu ->\n", m_Stack.size());
        if(m_Stack.size() > 0)
        {
            m_Stack.back()->OnDestroying();
            m_Stack.pop_back();
        }
    }

    bool SceneStack::ProcessUpdateResult(const SceneUpdateResult& updateResult) NN_NOEXCEPT
    {
        if(updateResult.switchMode == SceneSwitchMode_Replace)
        {
            ReplaceScene(updateResult.nextSceneIndex, updateResult.pNextSceneArgument);
            return true;
        }
        else if(updateResult.switchMode == SceneSwitchMode_Push)
        {
            PushScene(updateResult.nextSceneIndex, updateResult.pNextSceneArgument);
            return true;
        }
        else if(updateResult.switchMode == SceneSwitchMode_Pop)
        {
            PopScene();
            return true;
        }
        else if(updateResult.switchMode == SceneSwitchMode_Incomplete)
        {
            // 単にリトライさせる
            return true;
        }
        return false;
    }

    SceneUpdateResult SceneStack::UpdateTopScene() NN_NOEXCEPT
    {
        if(m_Stack.size() == 0)
        {
            return SceneUpdateResult::GetKeepResult();
        }
        return m_Stack.back()->Update();
    }

    SceneUpdateResult SceneStack::UpdateAllScene() NN_NOEXCEPT
    {
        if (m_Stack.size() == 0)
        {
            return SceneUpdateResult::GetKeepResult();
        }

        for (auto e : m_Stack)
        {
            e->Update();
        }

        // とりあえず Keep にしておく
        return SceneUpdateResult::GetKeepResult();
    }

    nn::util::Color4f SceneStack::GetBackgroundColor() const NN_NOEXCEPT
    {
        for(int i = static_cast<int>(m_Stack.size()) - 1; i >= 0; i--)
        {
            if(!m_Stack[i]->IsTransparent())
            {
                return m_Stack[i]->GetBackgroundColor();
            }
        }
        return nn::util::Color4f(0, 0, 0, 0);
    }

    void SceneStack::GatherPanel(std::vector<std::shared_ptr<panel::IPanel>>& list) NN_NOEXCEPT
    {
        if(m_Stack.size() == 0)
        {
            return;
        }

        int i = 0;
        int n = static_cast<int>(m_Stack.size());
        // 不透明なシーンが出てくるまで遡る
        for(i = n - 1; i >= 0; i--)
        {
            if(!m_Stack[i]->IsTransparent())
            {
                break;
            }
        }
        if(i < 0)
        {
            i = 0;
        }
        // 奥にあるシーンから登録
        for(; i < n; i++)
        {
            if(auto pPanel = m_Stack[i]->GetPanel())
            {
                list.push_back(pPanel);
            }
        }
    }

}
