﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_Assert.h>
#include "panel_IPanel.h"
#include "panel_PanelOcclusion.h"

namespace panel
{

    class PanelFunction
    {
    public:
        // グローバル座標系における Panel の左上座標を取得する。
        static nn::util::Vector3f GetPanelGlobalPosition(const std::shared_ptr<IPanel>& panel) NN_NOEXCEPT
        {
            NN_ASSERT_NOT_NULL(panel);
            if(auto parent = panel->GetParent())
            {
                return GetPanelGlobalPosition(parent) + panel->GetPosition();
            }
            else
            {
                return panel->GetPosition();
            }
        }

        // グローバル座標系における Panel の矩形を取得する。
        static PanelRectangle GetPanelGlobalRectangle(const std::shared_ptr<IPanel>& panel) NN_NOEXCEPT
        {
            NN_ASSERT_NOT_NULL(panel);
            auto position = GetPanelGlobalPosition(panel);
            auto size = panel->GetSize();

            PanelRectangle value = {};
            value.x      = static_cast<int>(position.GetX());
            value.y      = static_cast<int>(position.GetY());
            value.width  = static_cast<int>(size.GetX());
            value.height = static_cast<int>(size.GetY());
            return value;
        }

        // ローカル座標系における Panel の矩形を取得する。
        static PanelRectangle GetPanelLocalRectangle(const std::shared_ptr<IPanel>& panel) NN_NOEXCEPT
        {
            NN_ASSERT_NOT_NULL(panel);
            auto position = panel->GetPosition();
            auto size = panel->GetSize();

            PanelRectangle value = {};
            value.x      = static_cast<int>(position.GetX());
            value.y      = static_cast<int>(position.GetY());
            value.width  = static_cast<int>(size.GetX());
            value.height = static_cast<int>(size.GetY());
            return value;
        }

        // グローバル座標系で表された矩形範囲をパネルのローカル座標系で測った値を返す。
        static PanelRectangle TransformGlobalToLocalRectangle(const std::shared_ptr<IPanel>& panel, const PanelRectangle& rect) NN_NOEXCEPT
        {
            NN_ASSERT_NOT_NULL(panel);
            auto grect = GetPanelGlobalRectangle(panel);

            PanelRectangle value = {};
            value.x = rect.x - grect.x;
            value.y = rect.y - grect.y;
            value.width = rect.width;
            value.height = rect.height;
            return value;
        }

        // すべての子ノード child に対して f(child) を呼び出す。
        // f(child) が false を返した場合中断する。
        template<typename F>
        static void ForeachChildren(const std::shared_ptr<IPanelContainer>& panel, F f) NN_NOEXCEPT
        {
            if(panel == nullptr)
            {
                return;
            }

            int n = panel->GetChildrenCount();
            for(int i = 0; i < n; i++)
            {
                auto child = panel->GetChild(i);
                bool v = f(child);
                if(!v)
                {
                    break;
                }
            }
        }

        // 全ての弟ノード sibling に対して f(sibling) を呼び出す。
        // f(child) が false を返した場合中断する。
        template<typename F>
        static void ForeachFollowingSibling(const std::shared_ptr<IPanel>& panel, F f) NN_NOEXCEPT
        {
            if(panel == nullptr)
            {
                return;
            }

            if(auto parent = panel->GetParent())
            {
                int n = parent->GetChildrenCount();
                int iSelf = parent->GetChildIndex(panel);
                for(int i = iSelf + 1; i < n; i++)
                {
                    auto sibling = parent->GetChild(i);
                    bool v = f(sibling);
                    if(!v)
                    {
                        break;
                    }
                }
            }
        }

        // 全ての兄ノード sibling に対して f(sigling) を呼び出す。
        // f(child) が false を返した場合中断する。
        template<typename F>
        static void ForeachPrecedingSibling(const std::shared_ptr<IPanel>& panel, F f) NN_NOEXCEPT
        {
            if(panel == nullptr)
            {
                return;
            }

            if(auto parent = panel->GetParent())
            {
                int iSelf = parent->GetChildIndex(panel);
                for(int i = 0; i < iSelf; i++)
                {
                    auto sibling = parent->GetChild(i);
                    bool v = f(sibling);
                    if(!v)
                    {
                        break;
                    }
                }
            }
        }


        // パネルとその子が最前面に描画されるかを取得する。
        // 以下の全ての条件を満たす場合最前面
        //
        // - 自身が Visible または Transparent である。
        // - すべての先祖が Visible または Transparent である。
        // - 自身の弟または先祖の弟に、弟自身またはその子が描画されるものであって描画範囲が重複するものが存在しない
        //
        // 最前面のパネルの矩形範囲には他のパネルの描画結果が含まれないことが保証されます。
        // ただし crop によって描画されない場合があります。
        static PanelOcclusion CheckPanelOcclusion(const std::shared_ptr<IPanel>& panel) NN_NOEXCEPT
        {
            // nullptr は NoRegion
            if(panel == nullptr)
            {
                return PanelOcclusion_NoRegion;
            }

            // 自身が Visible または Transparent でなければ NoRegion
            if(panel->GetVisibility() != PanelVisibility::Visible && panel->GetVisibility() != PanelVisibility::Transparent)
            {
                return PanelOcclusion_NoRegion;
            }

            auto selfRect = GetPanelGlobalRectangle(panel);
            // 自身の描画範囲が空集合なら NoRegion
            if(selfRect.IsEmpty())
            {
                return PanelOcclusion_NoRegion;
            }

            // すべての先祖が Visible または Transparent でなければ Invalid
            for(auto p = panel->GetParent(); p != nullptr; p = p->GetParent())
            {
                if(p->GetVisibility() != PanelVisibility::Visible && p->GetVisibility() != PanelVisibility::Transparent)
                {
                    return PanelOcclusion_NoRegion;
                }
            }

            // 自身の弟または先祖の弟との遮蔽をチェック。
            // BG: 弟自身またはその子孫が描画されるものであって、その描画範囲がこのパネルの描画範囲を完全に含むものが 1 つ以上存在する
            // FG: 弟自身またはその子孫が描画されるものであって、その描画範囲とこのパネルの描画範囲が部分的にでも重複するものが 1 つも存在しない
            // Partial: それ以外
            PanelOcclusion result = PanelOcclusion_Foreground;
            {
                for(auto p = panel; p != nullptr; p = p->GetParent())
                {
                    ForeachFollowingSibling(
                        p,
                        [&](const std::shared_ptr<IPanel>& sibling)
                        {
                            CheckPanelOcclusionImpl(result, sibling, selfRect);
                            return result != PanelOcclusion_Background;
                        }
                    );
                }

                if(result == PanelOcclusion_Background)
                {
                    return result;
                }
            }

            return result;
        }

    private:
        static void CheckPanelOcclusionImpl(PanelOcclusion& refResult, const std::shared_ptr<IPanel>& child, const PanelRectangle& selfRect) NN_NOEXCEPT
        {
            if(child->GetVisibility() == PanelVisibility::Visible)
            {
                auto rect = GetPanelGlobalRectangle(child);
                auto intersect = selfRect.Product(rect);

                if(intersect == selfRect) // 完全に含まれる
                {
                    refResult = PanelOcclusion_Background;
                }
                else if(!intersect.IsEmpty()) // 部分的に含まれる
                {
                    refResult = PanelOcclusion_Partial;
                }
            }
            else if(child->GetVisibility() == PanelVisibility::Transparent)
            {
                // Transparent なコンテナなら更に中を見る
                if(auto pContainer = child->CastToContainer())
                {
                    ForeachChildren(
                        pContainer,
                        [&](const std::shared_ptr<IPanel>& c)
                        {
                            CheckPanelOcclusionImpl(refResult, c, selfRect);
                            return refResult != PanelOcclusion_Background;
                        }
                    );
                }
            }
        }

    public:
        // パネルの型名を表す文字列を取得します（Release では空文字列）
        static const char* GetPanelTypeNameStringForDevelop(PanelTypeType type) NN_NOEXCEPT;
        // パネルツリーの状態を出力します（Release では空関数）
        static void PrintPanelTreeForDevelop(const std::shared_ptr<IPanel>& panel, int indent = 0) NN_NOEXCEPT;
    };
}

