﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/gfx.h>

#include "../panel_IPanel.h"
#include "../panel_PanelContainer.h"
#include "../panel_PanelText.h"
#include "../panel_PanelHistorical.h"
#include "../panel_PanelSharedTexture.h"
#include "../panel_PanelOcclusion.h"

namespace panel{ namespace detail{
    class Drawer;

    struct RendererOption
    {
        // 描画するパネルの情報を出力するインターバル[frames]。
        // 0 以下の場合出力しない。
        int dumpingPanelInterval = 0;

        // 描画したパネルの数を出力するか。
        bool printRenderedPanelCount = false;
    };

    class Renderer
    {
    public:
        Renderer() NN_NOEXCEPT;

        void Initialize(Drawer* pDrawer) NN_NOEXCEPT;
        void Finalize() NN_NOEXCEPT;

        void SetOption(const RendererOption& option) NN_NOEXCEPT;
        void ResetOption() NN_NOEXCEPT;

        // 直前の RenderPanel() で描画されたパネルの数を取得します。
        // RenderPanel() で渡した panel のツリーから Invisible, Transparent のパネルを除外したノード数を返します。
        // 一度も RenderPanel() を呼び出していない場合 0 を返します。
        int GetRenderedPanelCount() const NN_NOEXCEPT;

        void RenderPanel(
            nn::gfx::CommandBuffer* pCommandBuffer,
            nn::gfx::Texture* pCurrentRenderTarget,
            const std::shared_ptr<IPanel>& panel,
            const PanelRectangle& cropRect,
            nn::gfx::Texture* pPreviousRenderTarget
        ) NN_NOEXCEPT;

    private:
        // 入口。再帰呼び出しする場合はこれを呼ぶこと。
        bool RenderPanelImpl(
            nn::gfx::CommandBuffer* pCommandBuffer,
            nn::gfx::Texture* pCurrentRenderTarget,
            const std::shared_ptr<IPanel>& panel,
            const PanelRectangle& cropRect,
            nn::gfx::Texture* pPreviousRenderTarget
        ) NN_NOEXCEPT;

        // 前回の描画結果をコピペする。
        // 成功したら true を返す。成功した場合、パネルの描画キャッシュが更新される。
        // 失敗した場合はちゃんと描画すること。
        bool RecyclePreviousRendering(
            nn::gfx::CommandBuffer* pCommandBuffer,
            nn::gfx::Texture* pCurrentRenderTarget,
            const std::shared_ptr<IPanel>& panel,
            const PanelRectangle& cropRect,
            PanelOcclusion occlusion,
            nn::gfx::Texture* pPreviousRenderTarget
        ) NN_NOEXCEPT;

        bool RenderPanelContainerImpl(
            nn::gfx::CommandBuffer* pCommandBuffer,
            nn::gfx::Texture* pCurrentRenderTarget,
            const std::shared_ptr<PanelContainer>& panel,
            const PanelRectangle& cropRect,
            PanelOcclusion occlusion,
            nn::gfx::Texture* pPreviousRenderTarget
        ) NN_NOEXCEPT;

        bool RenderPanelDefaultImpl(
            nn::gfx::CommandBuffer* pCommandBuffer,
            const std::shared_ptr<IPanel>& panel,
            const PanelRectangle& cropRect,
            PanelOcclusion occlusion
        ) NN_NOEXCEPT;

        bool RenderPanelTextImpl(
            nn::gfx::CommandBuffer* pCommandBuffer,
            const std::shared_ptr<PanelText>& panel,
            const PanelRectangle& cropRect,
            PanelOcclusion occlusion
        ) NN_NOEXCEPT;

        bool RenderPanelHistoricalImpl(
            nn::gfx::CommandBuffer* pCommandBuffer,
            nn::gfx::Texture* pCurrentRenderTarget,
            const std::shared_ptr<PanelHistorical>& panel,
            const PanelRectangle& cropRect,
            PanelOcclusion occlusion,
            nn::gfx::Texture* pPreviousRenderTarget
        ) NN_NOEXCEPT;

        bool RenderPanelSharedTextureImpl(
            nn::gfx::CommandBuffer* pCommandBuffer,
            const std::shared_ptr<PanelSharedTexture>& panel,
            const PanelRectangle& cropRect,
            PanelOcclusion occlusion
        ) NN_NOEXCEPT;
    private:
        Drawer* m_pDrawer = nullptr;
        int64_t m_FrameCount = 0;
        int m_RenderedPanelCount = 0;

        RendererOption m_Option = {};

    };

}}
