﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "panel_Renderer.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include "../../framework/Framework.h"
#include "../panel_PanelFunction.h"
#include "panel_Drawer.h"
#include "panel_DebugConstant.h"

namespace panel{ namespace detail{

    Renderer::Renderer() NN_NOEXCEPT
    {
        m_pDrawer = nullptr;
        m_Option = {};
    }

    void Renderer::Initialize(Drawer* pDrawer) NN_NOEXCEPT
    {
        m_pDrawer = pDrawer;
    }

    void Renderer::Finalize() NN_NOEXCEPT
    {
        m_pDrawer = nullptr;
    }

    void Renderer::SetOption(const RendererOption& option) NN_NOEXCEPT
    {
        m_Option = option;
    }

    void Renderer::ResetOption() NN_NOEXCEPT
    {
        m_Option = {};
    }

    int Renderer::GetRenderedPanelCount() const NN_NOEXCEPT
    {
        return m_RenderedPanelCount;
    }

    //-----------------------------------------------------------------------

    void Renderer::RenderPanel(
        nn::gfx::CommandBuffer* pCommandBuffer,
        nn::gfx::Texture* pCurrentRenderTarget,
        const std::shared_ptr<IPanel>& panel,
        const PanelRectangle& cropRect,
        nn::gfx::Texture* pPreviousRenderTarget
    ) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(m_pDrawer);

        // TORIAEZU:
        m_FrameCount++;
        m_RenderedPanelCount = 0;

        if(m_Option.dumpingPanelInterval > 0)
        {
            if(m_FrameCount % m_Option.dumpingPanelInterval == 0)
            {
                NN_DEVOVL_PANEL_LOG_RENDER("Frame=%lld\n", m_FrameCount);
                PanelFunction::PrintPanelTreeForDevelop(panel, 1);
            }
        }
        RenderPanelImpl(pCommandBuffer, pCurrentRenderTarget, panel, cropRect, pPreviousRenderTarget);

        if(m_Option.printRenderedPanelCount)
        {
            NN_DEVOVL_PANEL_LOG_RENDER("Frame=%lld #panel=%lld\n", m_FrameCount, m_RenderedPanelCount);
        }
    }

    //-----------------------------------------------------------------------

    namespace {

        void UpdateRenderingCacheImpl(const std::shared_ptr<IPanel>& panel, const PanelRectangle& globalRenderedRectangle, PanelOcclusion occlusion) NN_NOEXCEPT
        {
            panel->NotifyRenderingComplete(
                occlusion == PanelOcclusion_Foreground,
                PanelFunction::TransformGlobalToLocalRectangle(panel, globalRenderedRectangle),
                globalRenderedRectangle
            );
        }

        void InvalidateRenderingCacheRecursivelyImpl(const std::shared_ptr<IPanel>& panel, PanelOcclusion occlusion) NN_NOEXCEPT
        {
            UpdateRenderingCacheImpl(panel, {}, occlusion);
            if(auto p = panel->CastToContainer())
            {
                int n = p->GetChildrenCount();
                for(int i = 0; i < n; i++)
                {
                    auto child = p->GetChild(i);
                    InvalidateRenderingCacheRecursivelyImpl(child, occlusion);
                }
            }
        }
    }

    bool Renderer::RenderPanelImpl(
        nn::gfx::CommandBuffer* pCommandBuffer,
        nn::gfx::Texture* pCurrentRenderTarget,
        const std::shared_ptr<IPanel>& panel,
        const PanelRectangle& cropRect,
        nn::gfx::Texture* pPreviousRenderTarget
    ) NN_NOEXCEPT
    {
        if(panel == nullptr)
        {
            return true;
        }

        // 表示されないものは描画しない
        auto occlusion = PanelFunction::CheckPanelOcclusion(panel);
        if(occlusion == PanelOcclusion_NoRegion ||
            occlusion == PanelOcclusion_Background )
        {
            // 子孫を含めてキャッシュを破棄
            InvalidateRenderingCacheRecursivelyImpl(panel, occlusion);
            return true;
        }

        // Visible なものを数える
        if(panel->GetVisibility() == panel::PanelVisibility::Visible)
        {
            m_RenderedPanelCount++;
        }

        // TORIAEZU:
        //   Visible の場合だけコピーできるか試す
        //   Transparent の場合はコピーしない
        if(panel->GetVisibility() == panel::PanelVisibility::Visible)
        {
            if(RecyclePreviousRendering(pCommandBuffer, pCurrentRenderTarget, panel, cropRect, occlusion, pPreviousRenderTarget))
            {
                return true;
            }
        }

        // dispatch
        bool isSuccess = false;
        if(auto pPanelContainer = PanelContainer::IsContainer(panel))
        {
            isSuccess = RenderPanelContainerImpl(pCommandBuffer, pCurrentRenderTarget, pPanelContainer, cropRect, occlusion, pPreviousRenderTarget);
        }
        else if(auto pPanelText = PanelText::IsText(panel))
        {
            isSuccess = RenderPanelTextImpl(pCommandBuffer, pPanelText, cropRect, occlusion);
        }
        else if(auto pPanelHistorical = PanelHistorical::IsHistorical(panel))
        {
            isSuccess = RenderPanelHistoricalImpl(pCommandBuffer, pCurrentRenderTarget, pPanelHistorical, cropRect, occlusion, pPreviousRenderTarget);
        }
        else if(auto pPanelSharedTexture = PanelSharedTexture::IsSharedTexture(panel))
        {
            isSuccess = RenderPanelSharedTextureImpl(pCommandBuffer, pPanelSharedTexture, cropRect, occlusion);
        }
        else
        {
            isSuccess = RenderPanelDefaultImpl(pCommandBuffer, panel, cropRect, occlusion);
        }

        return isSuccess;
    }

    //-----------------------------------------------------------------------


    bool Renderer::RecyclePreviousRendering(
        nn::gfx::CommandBuffer* pCommandBuffer,
        nn::gfx::Texture* pCurrentRenderTarget,
        const std::shared_ptr<IPanel>& panel,
        const PanelRectangle& cropRect,
        PanelOcclusion occlusion,
        nn::gfx::Texture* pPreviousRenderTarget
    ) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(panel);

        // テクスチャが登録されていないとコピーできない。
        if(!pCurrentRenderTarget || !pPreviousRenderTarget)
        {
            return false;
        }

        // 再描画が必要ならコピーしない。
        if(panel->CheckRedrawRequired())
        {
            return false;
        }

        auto cache = panel->GetRenderingRecycleCache();

        // 前回最前面でなければコピーしない。
        if(!cache.IsForegroud())
        {
            return false;
        }

        // 今回の描画範囲を計算
        auto grect = cropRect.Product(PanelFunction::GetPanelGlobalRectangle(panel));
        auto lrect = PanelFunction::TransformGlobalToLocalRectangle(panel, grect);

        // 今回の描画内容が前回の描画範囲に含まれているか確認。
        //
        // TORIAEZU: 完全に一致する場合だけキャッシュを利用する。
        //           本当は lrect \in cache.LocalRectangle であれば ok。
        if(lrect != cache.GetLocalRectangle())
        {
            return false;
        }

        m_pDrawer->DrawWithCopyTexture(
            pCommandBuffer,
            pCurrentRenderTarget,
            pPreviousRenderTarget,
            grect,
            cache.GetGlobalRectangle()
        );

        // キャッシュを更新
        UpdateRenderingCacheImpl(panel, grect, occlusion);

        return true;
    }


    bool Renderer::RenderPanelContainerImpl(
        nn::gfx::CommandBuffer* pCommandBuffer,
        nn::gfx::Texture* pCurrentRenderTarget,
        const std::shared_ptr<PanelContainer>& panel,
        const PanelRectangle& cropRect,
        PanelOcclusion occlusion,
        nn::gfx::Texture* pPreviousRenderTarget
    ) NN_NOEXCEPT
    {
        auto visibility = panel->GetVisibility();
        if(visibility == PanelVisibility::Invisible)
        {
            // 描画対象でない。
            UpdateRenderingCacheImpl(panel, {}, occlusion);
            return true;
        }

        auto rect = PanelFunction::GetPanelGlobalRectangle(panel);
        auto srect = rect.Product(cropRect);

        // Visible なら背景を描く
        if(visibility == PanelVisibility::Visible)
        {
            m_pDrawer->DrawWithShader(pCommandBuffer, rect, cropRect, panel->GetColor(), nullptr, false);
        }

        // 子を描画する
        bool isComplete = true;
        {
            auto childCrop = rect.Product(cropRect);
            int n = panel->GetChildrenCount();
            for(int i = 0; i < n; i++)
            {
                if(auto child = panel->GetChild(i))
                {
                    isComplete &= RenderPanelImpl(pCommandBuffer, pCurrentRenderTarget, child, childCrop, pPreviousRenderTarget);
                }
            }
        }

        // 失敗した子がなければ成功
        if(isComplete)
        {
            UpdateRenderingCacheImpl(panel, srect, occlusion);
        }
        else
        {
            UpdateRenderingCacheImpl(panel, {}, occlusion);
        }

        return isComplete;
    }

    //----------------------------------------------------------------------

    bool Renderer::RenderPanelDefaultImpl(
        nn::gfx::CommandBuffer* pCommandBuffer,
        const std::shared_ptr<IPanel>& panel,
        const PanelRectangle& cropRect,
        PanelOcclusion occlusion
    ) NN_NOEXCEPT
    {
        if(panel->GetVisibility() != PanelVisibility::Visible)
        {
            // 描画対象でない
            UpdateRenderingCacheImpl(panel, {}, occlusion);
            return true;
        }

        auto rect = PanelFunction::GetPanelGlobalRectangle(panel);
        auto srect = rect.Product(cropRect);
        m_pDrawer->DrawWithShader(pCommandBuffer, rect, cropRect, panel->GetColor(), nullptr, false);

        UpdateRenderingCacheImpl(panel, srect, occlusion);
        return true;
    }

    bool Renderer::RenderPanelTextImpl(
        nn::gfx::CommandBuffer* pCommandBuffer,
        const std::shared_ptr<PanelText>& panel,
        const PanelRectangle& cropRect,
        PanelOcclusion occlusion
    ) NN_NOEXCEPT
    {
        if(panel->GetVisibility() != PanelVisibility::Visible)
        {
            // 描画対象でない
            UpdateRenderingCacheImpl(panel, {}, occlusion);
            return true;
        }

        auto rect = PanelFunction::GetPanelGlobalRectangle(panel);
        auto srect = rect.Product(cropRect);

        // 背景
        {
            m_pDrawer->DrawWithShader(pCommandBuffer, rect, cropRect, panel->GetColor(), nullptr, false);
        }

        // テキスト
        bool isTextComplete = false;
        if(!panel->GetText().empty())
        {
            isTextComplete = m_pDrawer->DrawWithTextWriter(pCommandBuffer, panel, cropRect);
        }
        else
        {
            // 描く文字がなければ成功
            isTextComplete = true;
        }

        bool isScuccess = isTextComplete;

        if(isScuccess)
        {
            UpdateRenderingCacheImpl(panel, srect, occlusion);
        }
        else
        {
            UpdateRenderingCacheImpl(panel, {}, occlusion);
        }

        return isScuccess;
    }

    bool Renderer::RenderPanelHistoricalImpl(
        nn::gfx::CommandBuffer* pCommandBuffer,
        nn::gfx::Texture* pCurrentRenderTarget,
        const std::shared_ptr<PanelHistorical>& panel,
        const PanelRectangle& cropRect,
        PanelOcclusion occlusion,
        nn::gfx::Texture* pPreviousRenderTarget
    ) NN_NOEXCEPT
    {
        if(panel->GetVisibility() != PanelVisibility::Visible)
        {
            // 描画対象でない
            UpdateRenderingCacheImpl(panel, {}, occlusion);
            return true;
        }

        auto rect = PanelFunction::GetPanelGlobalRectangle(panel);
        auto srect = rect.Product(cropRect);

        // 描画範囲が空集合なら無視
        if(srect.IsEmpty())
        {
            return true;
        }

        // 背景色でクリア
        m_pDrawer->DrawWithShader(pCommandBuffer, rect, cropRect, panel->GetColor(), nullptr, false);

        // 履歴が有効なら引き継ぐ
        if(panel->IsHistoryValid())
        {
            // コピー領域を計算（グローバル座標系）
            PanelRectangle srcRect = {};
            PanelRectangle dstRect = {};
            int offsetX = panel->GetOffsetX();
            int offsetY = panel->GetOffsetY();
            // ヨコ
            if(offsetX > 0)
            {
                srcRect.x     = srect.x;
                srcRect.width = srect.width - offsetX;
                dstRect.x     = srect.x + offsetX;
                dstRect.width = srect.width - offsetX;

            }
            else
            {
                srcRect.x     = srect.x - offsetX;
                srcRect.width = srect.width + offsetX;
                dstRect.x     = srect.x;
                dstRect.width = srect.width + offsetX;
            }
            // タテ
            if(offsetY > 0)
            {
                srcRect.y      = srect.y;
                srcRect.height = srect.height - offsetY;
                dstRect.y      = srect.y + offsetY;
                dstRect.height = srect.height - offsetY;

            }
            else
            {
                srcRect.y      = srect.y - offsetY;
                srcRect.height = srect.height + offsetY;
                dstRect.y      = srect.y;
                dstRect.height = srect.height + offsetY;
            }

            if(!srcRect.IsEmpty() && !dstRect.IsEmpty())
            {
                NN_SDK_ASSERT_EQUAL(srcRect.width, dstRect.width);
                NN_SDK_ASSERT_EQUAL(srcRect.height, dstRect.height);
                m_pDrawer->DrawWithCopyTexture(
                    pCommandBuffer,
                    pCurrentRenderTarget,
                    pPreviousRenderTarget,
                    dstRect,
                    srcRect
                );
            }
        }
        UpdateRenderingCacheImpl(panel, srect, occlusion);
        return true;
    }

    bool Renderer::RenderPanelSharedTextureImpl(
        nn::gfx::CommandBuffer* pCommandBuffer,
        const std::shared_ptr<PanelSharedTexture>& panel,
        const PanelRectangle& cropRect,
        PanelOcclusion occlusion
    ) NN_NOEXCEPT
    {
        if(panel->GetVisibility() != PanelVisibility::Visible)
        {
            // 描画対象でない
            UpdateRenderingCacheImpl(panel, {}, occlusion);
            return true;
        }

        auto rect = PanelFunction::GetPanelGlobalRectangle(panel);
        auto srect = rect.Product(cropRect);
        m_pDrawer->DrawWithShader(pCommandBuffer, rect, cropRect, panel->GetColor(), panel->GetTexture(), false);

        UpdateRenderingCacheImpl(panel, srect, occlusion);
        return true;
    }


}}
