﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "TextWriterPool.h"

#include "Framework.h"
#include "../Config.h"
#include <nn/util/util_Color.h>
#include <nn/util/util_MatrixApi.h>
#include <nn/util/util_Matrix.h>

namespace framework{

    namespace{
        static const uint32_t TextureCacheWidth = 512;
        static const uint32_t TextureCacheHeight = 1024;

        static const int CacheFontSize = 32;
        //static const int CacheIconSize = 32;

        void* MemoryAllocate(size_t size, size_t alignment, void*) NN_NOEXCEPT
        {
            return Framework::Allocate(size, alignment);
        }

        void MemoryFree(void* ptr, void*) NN_NOEXCEPT
        {
            Framework::Free(ptr);
        }

        bool AllocateAndRegisterTextureViewSlot(nn::gfx::DescriptorSlot* pOutSlot, const nn::gfx::TextureView& textureView, void*) NN_NOEXCEPT
        {
            int index = Framework::AllocateTextureDescriptorSlot();
            auto pPool = Framework::GetTextureDescriptorPool();
            pPool->SetTextureView(index, &textureView);
            pPool->GetDescriptorSlot(pOutSlot, index);
            return true;
        }

        void UnregisterAndFreeTextureViewSlot(nn::gfx::DescriptorSlot* pSlot, const nn::gfx::TextureView&, void*) NN_NOEXCEPT
        {
            auto pPool = Framework::GetTextureDescriptorPool();
            auto index = pPool->GetDescriptorSlotIndex(*pSlot);
            Framework::FreeTextureDescriptorSlot(index);
        }

        bool AllocateAndRegisterSamplerSlot(nn::gfx::DescriptorSlot* pOutSlot, const nn::gfx::Sampler& sampler, void*) NN_NOEXCEPT
        {
            int index = Framework::AllocateSamplerDescriptorSlot();
            auto pPool = Framework::GetSamplerDescriptorPool();
            pPool->SetSampler(index, &sampler);
            pPool->GetDescriptorSlot(pOutSlot, index);
            return true;
        }

        void UnregisterAndFreeSamplerSlot(nn::gfx::DescriptorSlot* pSlot, const nn::gfx::Sampler&, void*) NN_NOEXCEPT
        {
            auto pPool = Framework::GetSamplerDescriptorPool();
            auto index = pPool->GetDescriptorSlotIndex(*pSlot);
            Framework::FreeSamplerDescriptorSlot(index);
        }
    }

    void TextWriterPool::Initialize(const TextWriterPoolInfo& info) NN_NOEXCEPT
    {
        NN_DEVOVL_FW_LOG("Initializing TextWriterPool...\n");
        auto pDevice = Framework::GetDevice();

        // テクスチャキャッシュの初期化
        NN_DEVOVL_FW_LOG("  initilaizing TextureCache sizeof(arg=%llu)\n", sizeof(nn::font::TextureCache::InitializeArg));
        {
            auto arg = std::make_unique<nn::font::TextureCache::InitializeArg>();
            NN_ABORT_UNLESS_NOT_NULL(arg);
            arg->SetDefault();
            arg->allocateFunction = MemoryAllocate;
            arg->pUserDataForAllocateFunction = nullptr;
            arg->textureCacheWidth = TextureCacheWidth;
            arg->textureCacheHeight = TextureCacheHeight;
            arg->noPlotWorkMemorySize = 128 * 1024;

            //arg->fontCount = TextWriterUsage_Count;
            //{
            //    arg->fontListCount [TextWriterUsage_Text]    = 1;
            //    arg->pFontDatas    [TextWriterUsage_Text][0] = info.pFontDataList[0];
            //    arg->pFontDataSizes[TextWriterUsage_Text][0] = info.fontDataSizeList[0];
            //    arg->pFontDataTypes[TextWriterUsage_Text][0] = nn::font::TextureCache::FontDataType_Ttf;
            //}
            //{
            //    arg->fontListCount [TextWriterUsage_Icon]    = 1;
            //    arg->pFontDatas    [TextWriterUsage_Icon][0] = info.pFontDataList[1];
            //    arg->pFontDataSizes[TextWriterUsage_Icon][0] = info.fontDataSizeList[1];
            //    arg->pFontDataTypes[TextWriterUsage_Icon][0] = nn::font::TextureCache::FontDataType_Ttf;
            //}
            arg->fontCount = 1;
            {
                arg->fontListCount [TextWriterUsage_Text]    = 2;
                arg->pFontDatas    [TextWriterUsage_Text][0] = info.pFontDataList[0];
                arg->pFontDataSizes[TextWriterUsage_Text][0] = info.fontDataSizeList[0];
                arg->pFontDataTypes[TextWriterUsage_Text][0] = nn::font::TextureCache::FontDataType_Ttf;
                arg->pFontDatas    [TextWriterUsage_Text][1] = info.pFontDataList[1];
                arg->pFontDataSizes[TextWriterUsage_Text][1] = info.fontDataSizeList[1];
                arg->pFontDataTypes[TextWriterUsage_Text][1] = nn::font::TextureCache::FontDataType_Ttf;
            }

            m_TextureCache.Initialize(pDevice, *arg);
            m_TextureCache.RegisterTextureViewToDescriptorPool(AllocateAndRegisterTextureViewSlot, nullptr);
        }

        // スケーラブルフォントの初期化
        NN_DEVOVL_FW_LOG("  initilaizing ScalableFont(sizeof(arg)=%llu)\n", sizeof(nn::font::ScalableFont::InitializeArg));
        {
            // テキスト用
            NN_DEVOVL_FW_LOG("    for text\n");
            auto arg = std::make_unique<nn::font::ScalableFont::InitializeArg>();
            NN_ABORT_UNLESS_NOT_NULL(arg);
            arg->SetDefault();
            arg->pTextureCache = &m_TextureCache;
            arg->fontSize = CacheFontSize;
            arg->fontFace = TextWriterUsage_Text;
            arg->isAlternateCharSpaceWithOriginalWidthForNotReadyChar = true;
            m_ScalableFontList[TextWriterUsage_Text].Initialize(*arg);
        }
        //{
        //    // アイコン用
        //    NN_DEVOVL_FW_LOG("    for icon\n");
        //    auto arg = std::make_unique<nn::font::ScalableFont::InitializeArg>();
        //    NN_ABORT_UNLESS_NOT_NULL(arg);
        //    arg->SetDefault();
        //    arg->pTextureCache = &m_TextureCache;
        //    arg->fontSize = CacheIconSize;
        //    arg->fontFace = TextWriterUsage_Icon;
        //    arg->isAlternateCharSpaceWithOriginalWidthForNotReadyChar = true;
        //    m_ScalableFontList[TextWriterUsage_Icon].Initialize(*arg);
        //}

        // ASCII 文字を登録
        NN_DEVOVL_FW_LOG("  registering ASCII glyph\n");
        for(uint16_t c = 0; c < 127; c++)
        {
            m_ScalableFontList[TextWriterUsage_Text].RegisterGlyph(c, 0);
        }
        m_TextureCache.UpdateTextureCache();
        m_TextureCache.CompleteTextureCache();

        // RectDrawer の初期化
        NN_DEVOVL_FW_LOG("  initializing RectDrawer\n");
        {
            size_t workBufferSize = nn::font::RectDrawer::GetWorkBufferSize(pDevice, info.charCountMax);
            void* pWorkBuffer = Framework::Allocate(workBufferSize, NN_ALIGNOF(std::max_align_t));
            //NN_SDK_LOG("[devovl] RectDrawer WorkSize=%lld\n", workBufferSize);
            NN_ABORT_UNLESS(m_RectDrawer.Initialize(pDevice, pWorkBuffer, info.charCountMax));
            m_RectDrawer.RegisterSamplerToDescriptorPool(AllocateAndRegisterSamplerSlot, nullptr);
            m_pRectDrawerWorkBuffer = pWorkBuffer;
        }

        m_CharCountMax = info.charCountMax;
        m_WriterCount = info.writerCount;

        NN_DEVOVL_FW_LOG("  initializing TextWriters\n");
        m_EntryList.resize(info.writerCount);
        for(auto& e : m_EntryList)
        {
            e.writer.Initialize(this);
            e.isUsed = false;
        }

        NN_DEVOVL_FW_LOG("Initializing TextWriterPool...complete\n");
    }

    void TextWriterPool::Finalize() NN_NOEXCEPT
    {
        auto pDevice = Framework::GetDevice();

        for(auto& e : m_EntryList)
        {
            NN_ASSERT(!e.isUsed);
            e.writer.Finalize();
        }

        m_RectDrawer.UnregisterSamplerFromDescriptorPool(UnregisterAndFreeSamplerSlot, nullptr);
        m_RectDrawer.Finalize(pDevice);
        for(auto& e : m_ScalableFontList)
        {
            e.Finalize(pDevice);
        }
        m_TextureCache.UnregisterTextureViewFromDescriptorPool(UnregisterAndFreeTextureViewSlot, nullptr);
        m_TextureCache.Finalize(pDevice, MemoryFree, nullptr);

        Framework::Free(m_pRectDrawerWorkBuffer);
    }

    nn::font::ScalableFont* TextWriterPool::GetScalableFont(TextWriterUsage usage) NN_NOEXCEPT
    {
        return &m_ScalableFontList[usage];
    }

    std::shared_ptr<PoolTextWriterHolder> TextWriterPool::Acquire() NN_NOEXCEPT
    {
        int index = -1;
        TextWriter* pWriter = nullptr;
        for(int i = 0; i < m_WriterCount; i++)
        {
            auto& e = m_EntryList[i];
            if(!e.isUsed)
            {
                index = i;
                pWriter = &e.writer;
                e.isUsed = true;
                break;
            }
        }

        if(index < 0)
        {
            NN_DEVOVL_FW_LOG("[TextWriterPool] Acquire text writer failed\n");
            return nullptr;
        }

        // 状態をクリアしてから返す。
        pWriter->Reset();
        return std::make_shared<PoolTextWriterHolder>(index, pWriter, this);
    }

    void TextWriterPool::UpdateFontCacheUtf8Impl(const char* pCode, TextWriterUsage usage) NN_NOEXCEPT
    {
        auto count = m_ScalableFontList[usage].RegisterGlyphsUtf8(pCode, -1);
        if (count > 0)
        {
            m_TextureCache.UpdateTextureCache();
            m_TextureCache.CompleteTextureCache();
        }
    }

    //-----------------------------

    void TextWriter::Initialize(TextWriterPool* pTextWriterPool) NN_NOEXCEPT
    {
        auto pDevice = Framework::GetDevice();
        m_pTextWriterPool = pTextWriterPool;
        {
            auto dsbArg = std::make_unique<nn::font::DispStringBuffer::InitializeArg>();
            NN_ABORT_UNLESS_NOT_NULL(dsbArg);
            dsbArg->SetCharCountMax(pTextWriterPool->m_CharCountMax);

            size_t drawBufferSize = nn::font::DispStringBuffer::GetRequiredDrawBufferSize(*dsbArg);
            void* pDrawBufferMemory = Framework::Allocate(drawBufferSize, NN_ALIGNOF(std::max_align_t));
            //NN_SDK_LOG("[devovl]DsipStringBuffer drawBufferSize=%lld\n", drawBufferSize);
            dsbArg->SetDrawBuffer(pDrawBufferMemory);
            m_pDrawBufferMemory = pDrawBufferMemory;

            auto gbArg = std::make_unique<nn::font::GpuBuffer::InitializeArg>();
            gbArg->SetGpuAccessFlag(nn::gfx::GpuAccess_ConstantBuffer);
            size_t constantBufferSize = nn::font::DispStringBuffer::GetRequiredConstantBufferSize(pDevice, *dsbArg);
            //NN_SDK_LOG("[devovl]DsipStringBuffer constantBufferSize=%lld\n", constantBufferSize);
            auto offset = Framework::AllocateConstantBufferPoolMemory(constantBufferSize, NN_ALIGNOF(std::max_align_t));
            gbArg->SetBufferSize(constantBufferSize);
            gbArg->SetBufferCount(1);
            gbArg->SetMemoryPool(Framework::GetConstantBufferMemoryPool());
            gbArg->SetMemoryPoolOffset(offset);
            gbArg->SetAllocator(MemoryAllocate, nullptr);
            gbArg->SetAllocateSyncFlag(false);
            NN_ABORT_UNLESS(m_GpuBuffer.Initialize(pDevice, *gbArg));
            m_GpuBufferOffset = offset;

            dsbArg->SetConstantBuffer(&m_GpuBuffer);
            NN_ABORT_UNLESS(m_DisplayStringBuffer.Initialize(pDevice, *dsbArg));
        }

        Reset();
    }

    void TextWriter::Finalize() NN_NOEXCEPT
    {
        auto pDevice = Framework::GetDevice();

        m_GpuBuffer.Finalize(pDevice, MemoryFree, nullptr);
        m_DisplayStringBuffer.Finalize(pDevice);

        Framework::Free(m_pDrawBufferMemory);
        Framework::FreeConstantBufferPoolMemory(m_GpuBufferOffset);
    }

    void TextWriter::Reset() NN_NOEXCEPT
    {
        m_ViewportSizeX = 1280;
        m_ViewportSizeY = 720;
        m_PositionX = 0;
        m_PositionY = 0;
        m_FontSize = 10;
        m_Usage = TextWriterUsage_Text;
        m_Color = { 1, 1, 1, 1 };
        m_DisplayStringBuffer.SetCharCount(0);
    }

    void TextWriter::SetViewportSize(float w, float h) NN_NOEXCEPT
    {
        m_ViewportSizeX = w;
        m_ViewportSizeY = h;
    }

    void TextWriter::SetPosition(float x, float y) NN_NOEXCEPT
    {
        m_PositionX = x;
        m_PositionY = y;
    }

    void TextWriter::SetFontSize(float size) NN_NOEXCEPT
    {
        m_FontSize = size;
    }

    void TextWriter::SetColor(const nn::util::Color4fType& color) NN_NOEXCEPT
    {
        m_Color = color;
    }

    void TextWriter::Print(const char* str) NN_NOEXCEPT
    {
        m_pTextWriterPool->UpdateFontCacheUtf8Impl(str, m_Usage);

        nn::font::TextWriter writer;
        writer.SetDispStringBuffer(&m_DisplayStringBuffer);
        writer.SetFont(m_pTextWriterPool->GetScalableFont(m_Usage));
        writer.SetCursor(m_PositionX, m_PositionY);
        writer.SetScale(m_FontSize / CacheFontSize, m_FontSize / CacheFontSize);
        nn::util::Unorm8x4 color = nn::util::Color4u8(m_Color);
        writer.SetTextColor(color);

        writer.StartPrint();
        {
            writer.Print(str);
        }
        writer.EndPrint();

        m_GpuBuffer.Map(0);
        {
            nn::util::MatrixT4x4fType m = NN_UTIL_MATRIX_T4X4F_INITIALIZER(1,0,0,0, 0,1,0,0, 0,0,1,0, 0,0,0,1);
            nn::util::MatrixOrthographicOffCenterRightHanded(
                &m,
                0,
                m_ViewportSizeX,
                m_ViewportSizeY,
                0,
                0,   // nearZ
                1000 // farZ
            );
            m_DisplayStringBuffer.BuildConstantBuffer(m);
        }
        m_GpuBuffer.Unmap();
        m_GpuBuffer.SetGpuAccessBufferIndex(0);
    }

    void TextWriter::MakeCommand(nn::gfx::CommandBuffer* pCommandBuffer) NN_NOEXCEPT
    {
        m_pTextWriterPool->m_RectDrawer.Draw(*pCommandBuffer, m_DisplayStringBuffer);
    }

    //----------------------------------

    PoolTextWriterHolder::PoolTextWriterHolder(int index, TextWriter* pWriter, TextWriterPool* pPool) NN_NOEXCEPT
        : m_Index(index)
        , m_pWriter(pWriter)
        , m_pPool(pPool)
    {
    }

    PoolTextWriterHolder::~PoolTextWriterHolder() NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(m_pPool);
        NN_SDK_ASSERT_NOT_NULL(m_pWriter);

        NN_SDK_ASSERT(m_pPool->m_EntryList[m_Index].isUsed);
        m_pPool->m_EntryList[m_Index].isUsed = false;
    }

    TextWriter* PoolTextWriterHolder::Get() NN_NOEXCEPT
    {
        return m_pWriter;
    }

    TextWriterPool * PoolTextWriterHolder::GetPool()
    {
        return m_pPool;
    }

}
