﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Hid.h"

#include <cmath>
#include <array>
#include <nn/nn_Log.h>
#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/hid_DebugPad.h>

//#define FRAMEWORK_LOG_HID(...) NN_LOG("[hid]" __VA_ARGS__)

#ifndef FRAMEWORK_LOG_HID
#define FRAMEWORK_LOG_HID(...)
#endif

namespace {
    nn::hid::NpadIdType g_NpadIds[] = {
        nn::hid::NpadId::No1,
        nn::hid::NpadId::No2,
        nn::hid::NpadId::No3,
        nn::hid::NpadId::No4,
        nn::hid::NpadId::No5,
        nn::hid::NpadId::No6,
        nn::hid::NpadId::No7,
        nn::hid::NpadId::No8,
        nn::hid::NpadId::Handheld
    };

    NN_STATIC_ASSERT(HidNpadCountMax == sizeof(g_NpadIds) / sizeof(nn::hid::NpadIdType));

    HidButtonHandleParameter g_ButtonHandleParameter ={};

    std::array<HidNpadState, HidNpadCountMax> g_NpadInputList = {};

    HidButtons<int> g_ButtonPressedFrameCounts = {};
    HidButtons<int> g_PrevButtonInput = {};

    HidButtonState  g_CurrentButtonState = {};

    HidTouchState g_TouchState;
    int64_t g_PrevSamplingNumber = 0;

    //----------------------------------------------------------------------------------------------

    // 押されたボタンを取得
    // 押されたボタンは 1、それ以外は 0 になる
    template<typename TState>
    void ExtractNpadInputImpl(HidButtons<int>* pOutValue, const TState& state) NN_NOEXCEPT
    {
        typedef nn::hid::NpadJoyButton ButtonSet;
    #define NN_DEVOVL_HID_EXTRACT_BUTTON_INPUT(BUTTON)  \
        pOutValue->BUTTON = state.buttons.Test(ButtonSet::BUTTON::Index) ? 1 : 0;

        NN_DEVOVL_HID_FOREACH_JOY_BUTTON(NN_DEVOVL_HID_EXTRACT_BUTTON_INPUT);

    #undef NN_DEVOLV_HID_EXTRACT_BUTTON_INPUT
    }

    // 押されたボタンを取得
    // 押されたボタンは 1、それ以外は 0 になる
    template<typename TState>
    void ExtractDebugPadInputImpl(HidButtons<int>* pOutValue, const TState& state) NN_NOEXCEPT
    {
        typedef nn::hid::DebugPadButton ButtonSet;
    #define NN_DEVOVL_HID_EXTRACT_BUTTON_INPUT(BUTTON)  \
        pOutValue->BUTTON = state.buttons.Test(ButtonSet::BUTTON::Index) ? 1 : 0;

        NN_DEVOVL_HID_FOREACH_DEBUGPAD_BUTTON(NN_DEVOVL_HID_EXTRACT_BUTTON_INPUT);

    #undef NN_DEVOLV_HID_EXTRACT_BUTTON_INPUT
    }

    // ボタンの押下状態をマージする
    void MergeButtonInputImpl(HidButtons<int>* pDstValue, const HidButtons<int>& src) NN_NOEXCEPT
    {
    #define NN_DEVOVL_HID_MERGE_BUTTON_INPUT(BUTTON)    \
        pDstValue->BUTTON |= src.BUTTON;

        NN_DEVOVL_HID_FOREACH_BUTTON(NN_DEVOVL_HID_MERGE_BUTTON_INPUT);

    #undef NN_DEVOVL_HID_MERGE_BUTTON_INPUT
    }

    template<typename TState>
    void ExtractAnalogStickInputImpl(HidAnalogSticks* pOutValue, const TState& state) NN_NOEXCEPT
    {
        float lx = state.analogStickL.x / +32768.f;
        float ly = state.analogStickL.y / -32768.f;
        float rx = state.analogStickR.x / +32768.f;
        float ry = state.analogStickR.y / -32768.f;

        pOutValue->stickL.x = lx;
        pOutValue->stickL.y = ly;
        pOutValue->stickR.x = rx;
        pOutValue->stickR.y = ry;
    }

    void MergeAnalogStickInputImpl(HidAnalogSticks* pDstValue, const HidAnalogSticks& src) NN_NOEXCEPT
    {
        pDstValue->stickL.x += src.stickL.x;
        pDstValue->stickL.y += src.stickL.y;
        pDstValue->stickR.x += src.stickR.x;
        pDstValue->stickR.y += src.stickR.y;
    }

    void ExtractCurrentInput(
        HidButtons<int>* pOutMergedValue,
        HidAnalogSticks* pOutMergedAnalogValue,
        std::array<HidNpadState, HidNpadCountMax>* pOutNpadInputList
    ) NN_NOEXCEPT
    {
        HidButtons<int> mergedButton = {};
        HidAnalogSticks mergedAnalogStick = {};

        // Npad
        for(int i = 0; i < HidNpadCountMax; i++)
        {
            HidButtons<int> button = {};
            HidAnalogSticks stick = {};
            nn::Bit8 ledPattern = {};

            //現在有効な操作形態(NpadStyleSet)を取得
            nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(g_NpadIds[i]);
            ledPattern = nn::hid::GetPlayerLedPattern(g_NpadIds[i]);

            // フルキー操作
            if (style.Test<nn::hid::NpadStyleFullKey>() == true)
            {
                nn::hid::NpadFullKeyState state = {};
                nn::hid::GetNpadState(&state, g_NpadIds[i]);
                ExtractNpadInputImpl(&button, state);
                ExtractAnalogStickInputImpl(&stick, state);
                MergeButtonInputImpl(&mergedButton, button);
                MergeAnalogStickInputImpl(&mergedAnalogStick, stick);
                (*pOutNpadInputList)[i] = {HidNpadStyle_FullKey, button, stick, ledPattern};
            }
            // ジョイコン操作
            else if (style.Test<nn::hid::NpadStyleJoyDual>() == true)
            {
                nn::hid::NpadJoyDualState state = {};
                nn::hid::GetNpadState(&state, g_NpadIds[i]);
                ExtractNpadInputImpl(&button, state);
                ExtractAnalogStickInputImpl(&stick, state);
                MergeButtonInputImpl(&mergedButton, button);
                MergeAnalogStickInputImpl(&mergedAnalogStick, stick);

                // 持ち方を検査
                HidNpadStyle s = HidNpadStyle_JoyDual;
                bool hasLeft  = state.attributes.Test<nn::hid::NpadJoyAttribute::IsLeftConnected>();
                bool hasRight = state.attributes.Test<nn::hid::NpadJoyAttribute::IsRightConnected>();
                switch((hasLeft ? 1 : 0) + (hasRight ? 2 : 0))
                {
                case 1: s = HidNpadStyle_JoyDualLeftOnly; break;
                case 2: s = HidNpadStyle_JoyDualRightOnly; break;
                default: ;
                }

                (*pOutNpadInputList)[i] = {s, button, stick, ledPattern};
            }
            else if (style.Test<nn::hid::NpadStyleJoyLeft>() == true)
            {
                nn::hid::NpadJoyLeftState state = {};
                nn::hid::GetNpadState(&state, g_NpadIds[i]);
                ExtractNpadInputImpl(&button, state);
                ExtractAnalogStickInputImpl(&stick, state);
                MergeButtonInputImpl(&mergedButton, button);
                MergeAnalogStickInputImpl(&mergedAnalogStick, stick);
                (*pOutNpadInputList)[i] = {HidNpadStyle_JoyLeft, button, stick, ledPattern};
            }
            else if (style.Test<nn::hid::NpadStyleJoyRight>() == true)
            {
                nn::hid::NpadJoyRightState state = {};
                nn::hid::GetNpadState(&state, g_NpadIds[i]);
                ExtractNpadInputImpl(&button, state);
                ExtractAnalogStickInputImpl(&stick, state);
                MergeButtonInputImpl(&mergedButton, button);
                MergeAnalogStickInputImpl(&mergedAnalogStick, stick);
                (*pOutNpadInputList)[i] = {HidNpadStyle_JoyRight, button, stick, ledPattern};
            }
            // 携帯機コントローラー操作
            else if (style.Test<nn::hid::NpadStyleHandheld>() == true)
            {
                nn::hid::NpadHandheldState state = {};
                nn::hid::GetNpadState(&state, g_NpadIds[i]);
                ExtractNpadInputImpl(&button, state);
                ExtractAnalogStickInputImpl(&stick, state);
                MergeButtonInputImpl(&mergedButton, button);
                MergeAnalogStickInputImpl(&mergedAnalogStick, stick);
                (*pOutNpadInputList)[i] = {HidNpadStyle_Handheld, button, stick, ledPattern};
            }
            else
            {
                (*pOutNpadInputList)[i] = {};
            }
        }

        // DebugPad
        {
            HidButtons<int> button = {};
            HidAnalogSticks stick = {};

            nn::hid::DebugPadState state = {};
            nn::hid::GetDebugPadState(&state);
            ExtractDebugPadInputImpl(&button, state);
            ExtractAnalogStickInputImpl(&stick, state);
            MergeButtonInputImpl(&mergedButton, button);
            MergeAnalogStickInputImpl(&mergedAnalogStick, stick);
        }

        *pOutMergedValue = mergedButton;
        *pOutMergedAnalogValue = mergedAnalogStick;
    }

    //----------------------------------------------------------------------------------------------

    // 押され続けているフレームを加算
    void UpdatePressedFrameCount(HidButtons<int>* pFrameCount, const HidButtons<int>& currentInput) NN_NOEXCEPT
    {
#define FRAMEWORK_HID_UPDATE_PRESSED_FRAME_COUNT(BUTTON)    \
        pFrameCount->BUTTON = (pFrameCount->BUTTON + currentInput.BUTTON) * currentInput.BUTTON;

        NN_DEVOVL_HID_FOREACH_BUTTON(FRAMEWORK_HID_UPDATE_PRESSED_FRAME_COUNT)

#undef FRAMEWORK_HID_UPDATE_PRESSED_FRAME_COUNT
    }

    //----------------------------------------------------------------------------------------------

    void GetCurrentButtonStateImpl(bool* pOutIsPressed, bool* pOutIsTriggered, bool* pOutIsDown, bool* pOutIsUp, int pressedCount, bool currentInput, bool prevInput) NN_NOEXCEPT
    {
        *pOutIsPressed = currentInput;
        *pOutIsDown = currentInput && !prevInput;
        *pOutIsUp   = !currentInput && prevInput;

        int begCount = g_ButtonHandleParameter.repeatTriggerFrameBeginCount;
        int interval = g_ButtonHandleParameter.repeatTriggerFrameInterval;

        *pOutIsTriggered = false;
        if(pressedCount == 1)
        {
            *pOutIsTriggered = true;
        }
        else if(begCount > 0)
        {
            if(pressedCount >= begCount &&
                ((pressedCount - begCount) % interval) == 0
                )
            {
                *pOutIsTriggered = true;
            }
        }
    }

    //----------------------------------------------------------------------------------------------

    void UpdateCurrentButtonState(HidButtonState* pState, const HidButtons<int>& frameCounts, const HidButtons<int>& currentInput, const HidAnalogSticks& currentStick, const HidButtons<int>& prevInput) NN_NOEXCEPT
    {
        HidButtonState state = {};

        // ボタン状態
#define FRAMEWORK_HID_GET_CURRENT_BUTTON_STATE(BUTTON) \
    GetCurrentButtonStateImpl(                         \
        &state.isPressed.BUTTON, \
        &state.isTriggered.BUTTON, \
        &state.isDown.BUTTON, \
        &state.isUp.BUTTON, \
        frameCounts.BUTTON, \
        currentInput.BUTTON != 0, \
        prevInput.BUTTON != 0 \
    );

        NN_DEVOVL_HID_FOREACH_BUTTON(FRAMEWORK_HID_GET_CURRENT_BUTTON_STATE)

#undef FRAMEWORK_HID_GET_CURRENT_BUTTON_STATE

        // スティック状態
        {
            auto v =currentStick.stickL;
            float abs2 = v.x * v.x + v.y * v.y;
            if(abs2 > 1)
            {
                float len = std::sqrtf(abs2);
                v.x /= len;
                v.y /= len;
            }
            state.analogSticks.stickL = v;
        }
        {
            auto v =currentStick.stickR;
            float abs2 = v.x * v.x + v.y * v.y;
            if(abs2 > 1)
            {
                float len = std::sqrtf(abs2);
                v.x /= len;
                v.y /= len;
            }
            state.analogSticks.stickR = v;
        }

        *pState = state;
    }

    //----------------------------------------------------------------------------------------------

    void UpdateTouchState(HidTouchState* pHidTouchState)
    {
        nn::hid::TouchScreenState<nn::hid::TouchStateCountMax> touchScreenStates[nn::hid::TouchScreenStateCountMax] = {};
        nn::hid::GetTouchScreenStates(touchScreenStates, sizeof(touchScreenStates) / sizeof(touchScreenStates[0]));

        HidTouchState s = {};

        // 最新の位置を取得する
        if(touchScreenStates[0].count > 0)
        {
            s.touchPositionX = touchScreenStates[0].touches[0].x;
            s.touchPositionY = touchScreenStates[0].touches[0].y;
            s.isHeld = true;
        }
        else
        {
            s.touchPositionX = -1;
            s.touchPositionY = -1;
            s.isHeld = false;
        }

        s.isReleased = false;

        // 前のサンプリングからの差分の値のみを読み込む
        for (int i = 0; i < touchScreenStates[0].samplingNumber - g_PrevSamplingNumber; i++)
        {
            if (i >= nn::hid::TouchScreenStateCountMax)
            {
                break;
            }

            nn::hid::TouchState touchState = touchScreenStates[i].touches[0];

            // タッチが開始された場合
            if (touchState.attributes.Test<nn::hid::TouchAttribute::Start>())
            {
                s.isPressed = true;
            }

            // 指が画面から離れた場合
            if (touchState.attributes.Test<nn::hid::TouchAttribute::End>())
            {
                // 指が画面から離れる直前の位置を保持する
                s.touchPositionX = touchState.x;
                s.touchPositionY = touchState.y;
                s.isReleased = true;
                s.isPressed = false;
                break;
            }
        }
        //NN_LOG("%s%s%s (%d,%d)\n",
        //    s.isHeld     ? "H" : "-",
        //    s.isPressed  ? "S" : "-",
        //    s.isReleased ? "E" : "-",
        //    s.touchPositionX,
        //    s.touchPositionY);

        g_PrevSamplingNumber = touchScreenStates[0].samplingNumber;
        *pHidTouchState = s;
    }

    //----------------------------------------------------------------------------------------------
}

void Hid::Initialize() NN_NOEXCEPT
{
    nn::hid::InitializeNpad();
    nn::hid::SetSupportedNpadStyleSet(
        nn::hid::NpadStyleFullKey::Mask |
        nn::hid::NpadStyleJoyDual::Mask |
        nn::hid::NpadStyleHandheld::Mask |
        nn::hid::NpadStyleJoyLeft::Mask |
        nn::hid::NpadStyleJoyRight::Mask
    );
    nn::hid::SetSupportedNpadIdType(g_NpadIds, HidNpadCountMax);
    nn::hid::SetNpadHandheldActivationMode(nn::hid::NpadHandheldActivationMode_Single);

    nn::hid::InitializeDebugPad();
    nn::hid::InitializeTouchScreen();
}

void Hid::Finalize() NN_NOEXCEPT
{
}

void Hid::SetParameter(const HidButtonHandleParameter& param) NN_NOEXCEPT
{
    g_ButtonHandleParameter = param;
}

void Hid::Update() NN_NOEXCEPT
{
    // マージ後の値
    HidButtons<int> newInput = {};
    HidAnalogSticks newStick = {};

    ExtractCurrentInput(&newInput, &newStick, &g_NpadInputList);
    UpdatePressedFrameCount(&g_ButtonPressedFrameCounts, newInput);
    UpdateCurrentButtonState(&g_CurrentButtonState, g_ButtonPressedFrameCounts, newInput, newStick, g_PrevButtonInput);
    g_PrevButtonInput = newInput;

    //NN_LOG("[devovl]stick L(% 4.2f,% 4.2f) R(% 4.2f,% 4.2f)\n",
    //    g_CurrentButtonState.analogSticks.stickL.x,
    //    g_CurrentButtonState.analogSticks.stickL.y,
    //    g_CurrentButtonState.analogSticks.stickR.x,
    //    g_CurrentButtonState.analogSticks.stickR.y
    //);

    UpdateTouchState(&g_TouchState);
}

HidButtonState Hid::GetButtonState() NN_NOEXCEPT
{
    return g_CurrentButtonState;
}

HidTouchState Hid::GetTouchState() NN_NOEXCEPT
{
    return g_TouchState;
}

int Hid::GetNpadCount() NN_NOEXCEPT
{
    return static_cast<int>(g_NpadInputList.size());
}

HidNpadState Hid::GetNpadState(int index) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(index, 0, HidNpadCountMax);
    return g_NpadInputList[index];
}

