﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <nn/nn_Common.h>
#include <nn/util/util_Color.h>
#include <nn/gfx.h>

#include "../Config.h"
#include "Rectangle.h"
#include "TexturePool.h"
#include "TextWriterPool.h"
#include "SharedTexture.h"

class Framework
{
public:

    // ----------------------------------------
    static void Initialize() NN_NOEXCEPT;
    static void Finalize() NN_NOEXCEPT;
    static void MakeInitializeCommand(nn::gfx::CommandBuffer* pCommandBuffer) NN_NOEXCEPT;

    // ----------------------------------------

    // 次のフレームバッファを取得します。
    static void WindowAcquireTexture(nn::gfx::Fence* pOutFence) NN_NOEXCEPT;
    // フレームバッファを提出します。
    static void QueuePresentTexture() NN_NOEXCEPT;
    // キューを Flush します。
    static void QueueFlush() NN_NOEXCEPT;
    // キューの実行完了を待ちます
    static void WaitQueueFinish() NN_NOEXCEPT;
    // コマンドを提出します。
    static void QueueSubmitFrameCommand() NN_NOEXCEPT;
    // フェンスがシグナルされるまで GPU を待機させます。
    static void QueueWaitFence(nn::gfx::Fence* pFence) NN_NOEXCEPT;
    // GPU にフェンスをシグナルさせます。
    static void QueueSignalFence(nn::gfx::Fence* pFence) NN_NOEXCEPT;
    // フェンスがシグナルされるまで CPU をブロックします。
    static void WaitFence(nn::gfx::Fence* pFence) NN_NOEXCEPT;
    // コマンドの積み込みを開始します。
    static void BeginFrameCommand() NN_NOEXCEPT;
    // コマンドの積み込みを終了します。
    static void EndFrameCommand() NN_NOEXCEPT;

    // 前回の描画範囲をクリアします。
    static void MakeClearPreviousRendererdRegionCommand(nn::gfx::CommandBuffer* pCommnadBuffer, const nn::util::Color4f& color) NN_NOEXCEPT;
    // 次回描画時にクリアする範囲を登録します
    static void RegisterClearRegion(const framework::Rectangle& rect) NN_NOEXCEPT;

    // Crop 範囲を設定します。
    static void SetWindowCrop(const framework::Rectangle& rect) NN_NOEXCEPT;

    // フレームバッファをレンダーターゲット設定します。
    static void MakeSetFrameBufferRenderTargetCommand(nn::gfx::CommandBuffer* pCommandBuffer) NN_NOEXCEPT;
    // コマンドで使用しているリソースをロックします。
    static void LockFrameCommandResource(const std::shared_ptr<void>& p) NN_NOEXCEPT;
    // 完了したコマンドのリソースを解放します。
    static void ReleaseFrameCommandResource() NN_NOEXCEPT;

    // ----------------------------------------

    // フレームバッファの矩形範囲を取得する。
    static framework::Rectangle GetScreenRectangle() NN_NOEXCEPT;
    // マージンも含んだ矩形範囲を取得する。初期化用。
    static framework::Rectangle GetExtendedScreenRectangle() NN_NOEXCEPT;
    // 何も表示するものがない場合のクリップ範囲の矩形を取得する。
    static framework::Rectangle GetIdleCropRectangle() NN_NOEXCEPT;


    static void MakeSetViewportCommand(nn::gfx::CommandBuffer* pCommandBuffer, const framework::Rectangle& rect) NN_NOEXCEPT;
    static void MakeSetScissorCommand(nn::gfx::CommandBuffer* pCommandBuffer, const framework::Rectangle& rect) NN_NOEXCEPT;
    static void MakeSetViewportScissorCommand(nn::gfx::CommandBuffer* pCommandBuffer, const framework::Rectangle& rect) NN_NOEXCEPT;
    static void MakeClearCommand(nn::gfx::CommandBuffer* pCommandBuffer, const nn::util::Color4f& color) NN_NOEXCEPT;

    // ----------------------------------------

    static void* Allocate(size_t size, size_t alignment) NN_NOEXCEPT;
    static void Free(void* p) NN_NOEXCEPT;

    static nn::gfx::Device* GetDevice() NN_NOEXCEPT;
    static int GetCurrentFrameCount() NN_NOEXCEPT;
    static nn::gfx::ColorTargetView* GetCurrentColorTargetView() NN_NOEXCEPT;
    static nn::gfx::Texture* GetCurrentTargetTexture() NN_NOEXCEPT;
    static nn::gfx::ViewportScissorState* GetCurrentViewportScissorState() NN_NOEXCEPT;
    static const nn::gfx::DescriptorSlot& GetDefaultSamplerDescriptorSlot() NN_NOEXCEPT;
    static size_t GetFirmwareMemorySize() NN_NOEXCEPT;

    static nn::gfx::CommandBuffer* GetCommandBuffer() NN_NOEXCEPT;
    static size_t GetRootCommandMemorySize() NN_NOEXCEPT;
    static size_t GetRootControlMemorySize() NN_NOEXCEPT;
    static size_t GetRootCommandMemoryUsedSizeMax() NN_NOEXCEPT;
    static size_t GetRootControlMemoryUsedSizeMax() NN_NOEXCEPT;

    static size_t GetTexturePoolMemorySize() NN_NOEXCEPT;
    static size_t GetCommandBufferPoolMemorySize() NN_NOEXCEPT;
    static size_t GetConstantBufferPoolMemorySize() NN_NOEXCEPT;
    static size_t GetShaderPoolMemorySize() NN_NOEXCEPT;
    static size_t GetDataPoolMemorySize() NN_NOEXCEPT;

    static nn::gfx::MemoryPool* GetTextureMemoryPool() NN_NOEXCEPT;
    static ptrdiff_t AllocateTexturePoolMemory(size_t size, size_t alignment) NN_NOEXCEPT;
    static ptrdiff_t AllocateTexturePoolMemoryNoCheck(size_t size, size_t alignment) NN_NOEXCEPT;
    static void FreeTexturePoolMemory(ptrdiff_t offset) NN_NOEXCEPT;
    static size_t GetTotalAllocatedTexturePoolMemorySize() NN_NOEXCEPT;
    static size_t GetTotalAllocatedTexturePoolMemorySizeMax() NN_NOEXCEPT;

    static nn::gfx::MemoryPool* GetCommandBufferMemoryPool() NN_NOEXCEPT;
    static ptrdiff_t AllocateCommandBufferPoolMemory(size_t size, size_t alignment) NN_NOEXCEPT;
    static ptrdiff_t AllocateCommandBufferPoolMemoryNoCheck(size_t size, size_t alignment) NN_NOEXCEPT;
    static void FreeCommandBufferPoolMemory(ptrdiff_t offset) NN_NOEXCEPT;
    static size_t GetTotalAllocatedCommandBufferPoolMemorySize() NN_NOEXCEPT;
    static size_t GetTotalAllocatedCommandBufferPoolMemorySizeMax() NN_NOEXCEPT;

    static nn::gfx::MemoryPool* GetConstantBufferMemoryPool() NN_NOEXCEPT;
    static ptrdiff_t AllocateConstantBufferPoolMemory(size_t size, size_t alignment) NN_NOEXCEPT;
    static ptrdiff_t AllocateConstantBufferPoolMemoryNoCheck(size_t size, size_t alignment) NN_NOEXCEPT;
    static void FreeConstantBufferPoolMemory(ptrdiff_t offset) NN_NOEXCEPT;
    static size_t GetTotalAllocatedConstantBufferPoolMemorySize() NN_NOEXCEPT;
    static size_t GetTotalAllocatedConstantBufferPoolMemorySizeMax() NN_NOEXCEPT;

    static nn::gfx::MemoryPool* GetShaderMemoryPool() NN_NOEXCEPT;
    static ptrdiff_t AllocateShaderPoolMemory(size_t size, size_t alignment) NN_NOEXCEPT;
    static ptrdiff_t AllocateShaderPoolMemoryNoCheck(size_t size, size_t alignment) NN_NOEXCEPT;
    static void FreeShaderPoolMemory(ptrdiff_t offset) NN_NOEXCEPT;
    static size_t GetTotalAllocatedShaderPoolMemorySize() NN_NOEXCEPT;
    static size_t GetTotalAllocatedShaderPoolMemorySizeMax() NN_NOEXCEPT;

    static nn::gfx::MemoryPool* GetDataMemoryPool() NN_NOEXCEPT;
    static ptrdiff_t AllocateDataPoolMemory(size_t size, size_t alignment) NN_NOEXCEPT;
    static ptrdiff_t AllocateDataPoolMemoryNoCheck(size_t size, size_t alignment) NN_NOEXCEPT;
    static void FreeDataPoolMemory(ptrdiff_t offset) NN_NOEXCEPT;
    static size_t GetTotalAllocatedDataPoolMemorySize() NN_NOEXCEPT;
    static size_t GetTotalAllocatedDataPoolMemorySizeMax() NN_NOEXCEPT;

    static nn::gfx::DescriptorPool* GetTextureDescriptorPool() NN_NOEXCEPT;
    static int AllocateTextureDescriptorSlot() NN_NOEXCEPT;
    static int AllocateTextureDescriptorSlotNoCheck() NN_NOEXCEPT;
    static void FreeTextureDescriptorSlot(int index) NN_NOEXCEPT;

    static nn::gfx::DescriptorPool* GetSamplerDescriptorPool() NN_NOEXCEPT;
    static int AllocateSamplerDescriptorSlot() NN_NOEXCEPT;
    static int AllocateSamplerDescriptorSlotNoCheck() NN_NOEXCEPT;
    static void FreeSamplerDescriptorSlot(int index) NN_NOEXCEPT;

    static std::shared_ptr<framework::PoolTexture> AcquirePoolTexture(int width, int height, nn::gfx::ImageFormat format) NN_NOEXCEPT;
    static std::shared_ptr<framework::PoolTextWriterHolder> AcquirePoolTextWriter() NN_NOEXCEPT;

    // TORIAEZU: ちゃんと各種計算用関数を用意した方がよい気がする。
    static nn::font::ScalableFont* GetTextWriterScalableFont() NN_NOEXCEPT;
    static void CalculateTextRenderingSize(int* pOutWidth, int* pOutHeight, const std::string& text, float fontSize, framework::TextWriterUsage usage) NN_NOEXCEPT;

    static int GetSharedTextureCount() NN_NOEXCEPT;
    static std::shared_ptr<framework::SharedTexture> AcquireSharedTexture(int index) NN_NOEXCEPT;
};
