﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_TimeSpan.h>
#include <nn/util/util_Color.h>

namespace config{

    // 通知の定数
    class NotificationConstant
    {
    public:
        // コンディション通知をキューに積める最大数
        static const int ConditionNotificationQueueSize = 60;

        static const int LowBatteryThresholdPercent1 = 15; // 15%～6%
        static const int LowBatteryThresholdPercent2 = 5;  // 5%～2%
        static const int LowBatteryThresholdPercent3 = 1;  // 1%～0%
    };

    // 通知の表示時間
    class NotificationTtl
    {
    public:
        // 本体バッテリー減少（15%）
        static nn::TimeSpan GetTtlLowBattery1() NN_NOEXCEPT { return nn::TimeSpan::FromMilliSeconds(5000); }
        // 本体バッテリー減少(5%)
        static nn::TimeSpan GetTtlLowBattery2() NN_NOEXCEPT { return nn::TimeSpan::FromMilliSeconds(7000); }
        // 本体バッテリー減少(1%)
        static nn::TimeSpan GetTtlLowBattery3() NN_NOEXCEPT { return nn::TimeSpan::FromMilliSeconds(10000); }

        // 充電開始
        static nn::TimeSpan GetTtlBatteryChargeStart() NN_NOEXCEPT { return nn::TimeSpan::FromMilliSeconds(3000); }
        // 充電停止
        static nn::TimeSpan GetTtlBatteryChargeStop() NN_NOEXCEPT  { return nn::TimeSpan::FromMilliSeconds(3000); }

        // コントローラ登録
        static nn::TimeSpan GetTtlControllerPairing() NN_NOEXCEPT    { return nn::TimeSpan::FromMilliSeconds(3000); }
        // コントローラバッテリー減少
        static nn::TimeSpan GetTtlControllerLowBattery() NN_NOEXCEPT { return nn::TimeSpan::FromMilliSeconds(5000); }
        // コントローラバッテリー切れ
        static nn::TimeSpan GetTtlControllerNoBattery() NN_NOEXCEPT  { return nn::TimeSpan::FromMilliSeconds(5000); }

        // Home ボタン禁止
        static nn::TimeSpan GetTtlHomeButtonProhibited() NN_NOEXCEPT { return nn::TimeSpan::FromMilliSeconds(3000); }

        // 音量変化時
        static nn::TimeSpan GetTtlAudioVolumeChanged() NN_NOEXCEPT    { return nn::TimeSpan::FromMilliSeconds(1200); }
        // ヘッドホンの抜差し
        static nn::TimeSpan GetTtlAudioHeadphonePlugged() NN_NOEXCEPT { return nn::TimeSpan::FromMilliSeconds(3000); }

        // 画面写真成功
        static nn::TimeSpan GetTtlScreenShotSuccess() NN_NOEXCEPT { return nn::TimeSpan::FromMilliSeconds(1800); }
        // 画面写真失敗
        static nn::TimeSpan GetTtlScreenShotFailure() NN_NOEXCEPT { return nn::TimeSpan::FromMilliSeconds(3000); }

        // 動画撮影中（成功か失敗で上書きされるので長めにしておく）
        static nn::TimeSpan GetTtlMovieSaving() NN_NOEXCEPT { return nn::TimeSpan::FromMilliSeconds(60000); }
        // 動画撮影成功
        static nn::TimeSpan GetTtlMovieSuccess() NN_NOEXCEPT { return nn::TimeSpan::FromMilliSeconds(1800); }
        // 動画撮影失敗
        static nn::TimeSpan GetTtlMovieFailure() NN_NOEXCEPT { return nn::TimeSpan::FromMilliSeconds(3000); }

        // プレイタイマー僅少
        static nn::TimeSpan GetTtlPlayTimerLimitComing() NN_NOEXCEPT { return nn::TimeSpan::FromMilliSeconds(5000); }
        // プレイタイマー到達
        static nn::TimeSpan GetTtlPlayTimerLimitReached() NN_NOEXCEPT { return nn::TimeSpan::FromMilliSeconds(10000); }
        // プレイタイマー超過
        static nn::TimeSpan GetTtlPlayTimerLimitOver()   NN_NOEXCEPT { return nn::TimeSpan::FromMilliSeconds(10000); }

        // 見守り設定変更
        static nn::TimeSpan GetTtlParentalControlChanged() NN_NOEXCEPT { return nn::TimeSpan::FromMilliSeconds(3000); }

        // ダウンロード完了
        static nn::TimeSpan GetTtlDownloadCompleted() NN_NOEXCEPT { return nn::TimeSpan::FromMilliSeconds(5000); }

        // フレンド通知
        static nn::TimeSpan GetTtlFriend() NN_NOEXCEPT { return nn::TimeSpan::FromMilliSeconds(7000); }

        // サーバーメンテナンス予告
        static nn::TimeSpan GetTtlServerMaintenanceNotice() NN_NOEXCEPT { return nn::TimeSpan::FromMilliSeconds(10000); }
        // サーバーメンテナンス開始
        static nn::TimeSpan GetTtlServerMaintenanceStart()  NN_NOEXCEPT { return nn::TimeSpan::FromMilliSeconds(8000); }

        // アカウント
        static nn::TimeSpan GetTtlAccount() NN_NOEXCEPT { return nn::TimeSpan::FromMilliSeconds(5000); }

        // SnapShotDomper 進捗
        static nn::TimeSpan GetTtlSnapShotDumperProgress() NN_NOEXCEPT    { return nn::TimeSpan::FromMilliSeconds(5 * 60 * 1000); } // 次の進捗が来たら上書きされる
        static nn::TimeSpan GetTtlSnapShotDumperComplete() NN_NOEXCEPT    { return nn::TimeSpan::FromMilliSeconds(5000); }
        static nn::TimeSpan GetTtlSnapShotDumperFailure() NN_NOEXCEPT     { return nn::TimeSpan::FromMilliSeconds(5000); }
    };


    // コンディション通知の優先度（値が小さい方が高優先度）
    enum ConditionPriority
    {
        ConditionPriority_Highest = 0,
        ConditionPriority_ControllerNoBattery,          // コントローラバッテリー切れ
        ConditionPriority_LowBattery,                   // 本体バッテリー僅少
        ConditionPriority_Friend,                       // フレンド状況通知
        ConditionPriority_Account,                      // アカウント通知
        ConditionPriority_ControllerPairing,            // コントローラ登録完了
        ConditionPriority_ServerMaintenanceStart,       // サーバメンテナンス開始
        ConditionPriority_PlayTimerLimitReachedStop,    // プレイタイマー到達（強制中断）
        ConditionPriority_DownloadCompleted,            // ダウンロード完了
        ConditionPriority_ControllerLowBattery,         // コントローラバッテリー僅少
        ConditionPriority_PlayTimerLimitComing,         // プレイタイマー僅少
        ConditionPriority_PlayTimerLimitReachedAlarm,   // プレイタイマー到達（アラーム）
        ConditionPriority_PlayTimerLimitOver,           // プレイタイマー超過
        ConditionPriority_ParentalControlChanged,       // 見守り設定変更
        ConditionPriority_ServerMaintenanceNotice,      // サーバメンテナンス予告
        ConditionPriority_Lowest,
    };


    // スタイル
    class NotificationStyle
    {
    public:
        static const int Width  = 400;
        static const int Height = 72;

    public:
        // 共通

        static float GetWidthF() NN_NOEXCEPT  { return static_cast<float>(Width); }
        static float GetHeightF() NN_NOEXCEPT { return static_cast<float>(Height); }
        static nn::util::Float2 GetSizeF() NN_NOEXCEPT { return nn::util::MakeFloat2(GetWidthF(), GetHeightF()); }

        static nn::util::Color4f GetBackgroundColor() NN_NOEXCEPT{ return nn::util::Color4f(0, 0, 0, 1); }

        //--------------------------------
        // 本体バッテリー僅少通知
        //--------------------------------

        // 本体バッテリー僅少通知の文字色
        static nn::util::Color4f GetLowBatteryTextColor() NN_NOEXCEPT { return nn::util::Color4f(1, 1, 1, 1); }
        // 本体バッテリー僅少通知の文字の左上座標
        static float GetLowBatteryTextPositionX() NN_NOEXCEPT { return 20; }
        static float GetLowBatteryTextPositionY() NN_NOEXCEPT { return 20; }
        // 本体バッテリー僅少通知の文字サイズ
        static float GetLowBatteryTextSize() NN_NOEXCEPT { return 20; }

        //--------------------------------
        // 充電状態通知
        //--------------------------------

        // 充電状態通知の文字色
        static nn::util::Color4f GetBatteryChargeTextColor() NN_NOEXCEPT { return nn::util::Color4f(1, 1, 1, 1); }
        // 充電状態通知の文字の左上座標
        static float GetBatteryChargeTextPositionX() NN_NOEXCEPT { return 20; }
        static float GetBatteryChargeTextPositionY() NN_NOEXCEPT { return 20; }
        // 充電状態通知の文字サイズ
        static float GetBatteryChargeTextSize() NN_NOEXCEPT { return 20; }

        //--------------------------------
        // コントローラ登録通知
        //--------------------------------

        // コントローラ登録通知の文字色
        static nn::util::Color4f GetControllerPairingTextColor() NN_NOEXCEPT { return nn::util::Color4f(1, 1, 1, 1); }
        // コントローラ登録通知の文字の左上座標
        static float GetControllerPairingTextPositionX() NN_NOEXCEPT { return 20; }
        static float GetControllerPairingTextPositionY() NN_NOEXCEPT { return 20; }
        // コントローラ登録通知の文字サイズ
        static float GetControllerPairingTextSize() NN_NOEXCEPT { return 20; }

        //--------------------------------
        // コントローラバッテリー通知
        //--------------------------------

        // コントローラバッテリー通知の文字色
        static nn::util::Color4f GetControllerBatteryTextColor() NN_NOEXCEPT { return nn::util::Color4f(1, 1, 1, 1); }
        // コントローラバッテリー通知の文字の左上座標
        static float GetControllerBatteryTextPositionX() NN_NOEXCEPT { return 20; }
        static float GetControllerBatteryTextPositionY() NN_NOEXCEPT { return 20; }
        // コントローラバッテリー通知の文字サイズ
        static float GetControllerBatteryTextSize() NN_NOEXCEPT { return 20; }

        //--------------------------------
        // 音量変更通知
        //--------------------------------

        // 音声出力デバイス名の文字色
        static nn::util::Color4f GetAudioDeviceTextColor() NN_NOEXCEPT { return nn::util::Color4f(1, 1, 1, 1); }
        // 音声出力デバイス名の文字位置
        static float GetAudioDeviceTextPositionX() NN_NOEXCEPT { return 10; }
        static float GetAudioDeviceTextPositionY() NN_NOEXCEPT { return 5; }
        // 音声出力デバイス名の文字サイズ
        static float GetAudioDeviceTextSize() NN_NOEXCEPT { return 20; }


        // 音量バーの背景色
        static nn::util::Color4f GetAudioVolumeIndicatorBackgroundColor() NN_NOEXCEPT { return nn::util::Color4f(.2f, .2f, .2f, 1); }
        // 音量バーの前景色
        static nn::util::Color4f GetAudioVolumeIndicatorForegroundColor() NN_NOEXCEPT { return nn::util::Color4f(1, 1, 1, 1); }
        // 音量バーの左上座標（通知領域内のローカル座標系）
        static nn::util::Vector3f GetAudioVolumeIndicatorPosition() NN_NOEXCEPT
        {
            return nn::util::Vector3f(GetWidthF() * 0.1f, GetHeightF() * 0.55f, 0);
        }
        // 音量バーの大きさ
        static nn::util::Float2 GetAudioVolumeIndicatorSize() NN_NOEXCEPT
        {
            return nn::util::MakeFloat2(GetWidthF() * 0.8f, GetHeightF() * 0.1f);
        }

        //--------------------------------
        // 画面写真通知
        //--------------------------------

        // 画面写真通知の文字色
        static nn::util::Color4f GetScreenShotTextColor() NN_NOEXCEPT { return nn::util::Color4f(1, 1, 1, 1); }
        // 画面写真通知の文字の左上座標
        static float GetScreenShotTextPositionX() NN_NOEXCEPT { return 20; }
        static float GetScreenShotTextPositionY() NN_NOEXCEPT { return 20; }
        // 画面写真通知の文字サイズ
        static float GetScreenShotTextSize() NN_NOEXCEPT { return 20; }

        //--------------------------------
        // 動画通知
        //--------------------------------

        // 動画通知の文字色
        static nn::util::Color4f GetMovieTextColor() NN_NOEXCEPT { return nn::util::Color4f(1, 1, 1, 1); }
        // 動画通知の文字の左上座標
        static float GetMovieTextPositionX() NN_NOEXCEPT { return 20; }
        static float GetMovieTextPositionY() NN_NOEXCEPT { return 20; }
        // 動画通知の文字サイズ
        static float GetMovieTextSize() NN_NOEXCEPT { return 20; }

        //--------------------------------
        // プレイタイマー通知
        //--------------------------------

        // プレイタイマー通知の文字色
        static nn::util::Color4f GetPlayTimerTextColor() NN_NOEXCEPT { return nn::util::Color4f(1, 1, 1, 1); }
        // プレイタイマー通知の文字の左上座標
        static float GetPlayTimerTextPositionX() NN_NOEXCEPT { return 20; }
        static float GetPlayTimerTextPositionY() NN_NOEXCEPT { return 20; }
        // プレイタイマー通知の文字サイズ
        static float GetPlayTimerTextSize() NN_NOEXCEPT { return 20; }

        //--------------------------------
        // 見守り設定通知
        //--------------------------------

        // 見守り設定通知の文字色
        static nn::util::Color4f GetParentalControlTextColor() NN_NOEXCEPT { return nn::util::Color4f(1, 1, 1, 1); }
        // 見守り設定通知の文字の左上座標
        static float GetParentalControlTextPositionX() NN_NOEXCEPT { return 20; }
        static float GetParentalControlTextPositionY() NN_NOEXCEPT { return 20; }
        // 見守り設定通知の文字サイズ
        static float GetParentalControlTextSize() NN_NOEXCEPT { return 20; }

        //--------------------------------
        // ダウンロード通知
        //--------------------------------

        // ダウンロード通知の文字色
        static nn::util::Color4f GetDownloadTextColor() NN_NOEXCEPT { return nn::util::Color4f(1, 1, 1, 1); }
        // ダウンロード通知の文字の左上座標
        static float GetDownloadTextPositionX() NN_NOEXCEPT { return 20; }
        static float GetDownloadTextPositionY() NN_NOEXCEPT { return 20; }
        // ダウンロード通知の文字サイズ
        static float GetDownloadTextSize() NN_NOEXCEPT { return 20; }

        //--------------------------------
        // フレンド通知
        //--------------------------------

        // フレンド通知の文字色
        static nn::util::Color4f GetFriendTextColor() NN_NOEXCEPT { return nn::util::Color4f(1, 1, 1, 1); }
        // フレンド通知の文字の左上座標
        static float GetFriendTextPositionX() NN_NOEXCEPT { return 20; }
        static float GetFriendTextPositionY() NN_NOEXCEPT { return 10; }
        // フレンド通知の文字サイズ
        static float GetFriendTextSize() NN_NOEXCEPT { return 20; }

        //--------------------------------
        // Home 禁止通知
        //--------------------------------

        // Home 禁止通知の文字色
        static nn::util::Color4f GetHomeProhibitedTextColor() NN_NOEXCEPT { return nn::util::Color4f(1, 1, 1, 1); }
        // Home 禁止通知の文字の左上座標
        static float GetHomeProhibitedTextPositionX() NN_NOEXCEPT { return 20; }
        static float GetHomeProhibitedTextPositionY() NN_NOEXCEPT { return 20; }
        // Home 禁止通知の文字サイズ
        static float GetHomeProhibitedTextSize() NN_NOEXCEPT { return 20; }

        //--------------------------------
        // サーバーメンテナンス通知
        //--------------------------------

        // サーバーメンテナンス通知の文字色
        static nn::util::Color4f GetServerMaintenanceTextColor() NN_NOEXCEPT { return nn::util::Color4f(1, 1, 1, 1); }
        // サーバーメンテナンス通知の文字の左上座標
        static float GetServerMaintenanceTextPositionX() NN_NOEXCEPT { return 20; }
        static float GetServerMaintenanceTextPositionY() NN_NOEXCEPT { return 20; }
        // サーバーメンテナンス通知の文字サイズ
        static float GetServerMaintenanceTextSize() NN_NOEXCEPT { return 20; }


        //-------------------------------
        // アカウント
        //-------------------------------

        // アカウント通知の文字色
        static nn::util::Color4f GetAccountTextColor() NN_NOEXCEPT { return nn::util::Color4f(1, 1, 1, 1); }
        // アカウント通知の文字の左上座標
        static float GetAccountTextPositionX() NN_NOEXCEPT { return 20; }
        static float GetAccountTextPositionY() NN_NOEXCEPT { return 5; }
        // アカウント通知の文字サイズ
        static float GetAccountTextSize() NN_NOEXCEPT { return 20; }

        //--------------------------------
        // SnapShotDumper の進捗
        //--------------------------------

        // テキスト
        static nn::util::Color4f GetSnapShotDumperProgressTextColor() NN_NOEXCEPT { return nn::util::Color4f(1, 1, 1, 1); }
        static float GetSnapShotDumperProgressTextPositionX() NN_NOEXCEPT { return 10; }
        static float GetSnapShotDumperProgressTextPositionY() NN_NOEXCEPT { return 5; }
        static float GetSnapShotDumperProgressTextSize() NN_NOEXCEPT { return 20; }

        // 進捗バー
        static nn::util::Color4f GetSnapShotDumperProgressIndicatorBackgroundColor() NN_NOEXCEPT { return nn::util::Color4f(.2f, .2f, .2f, 1); }
        static nn::util::Color4f GetSnapShotDumperProgressIndicatorForegroundColor() NN_NOEXCEPT { return nn::util::Color4f(1, 1, 1, 1); }
        static nn::util::Vector3f GetSnapShotDumperProgressIndicatorPosition() NN_NOEXCEPT
        {
            return nn::util::Vector3f(GetWidthF() * 0.1f, GetHeightF() * 0.55f, 0);
        }
        static nn::util::Float2 GetSnapShotDumperProgressIndicatorSize() NN_NOEXCEPT
        {
            return nn::util::MakeFloat2(GetWidthF() * 0.8f, GetHeightF() * 0.1f);
        }

    };

}
