﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Assert.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/os.h>
#include <nn/init.h>
#include <nn/fs.h>
#include <nn/fs/fs_ApiPrivate.h>
#include <nn/ae.h>
#include <nn/ns/ns_InitializationApi.h>
#include <nn/ovln/ovln_ReceiverForOverlay.h>
#include <nn/pcm/pcm.h>
#include <nn/time.h>

#include "Config.h"
#include "ThreadMessageChannel.h"
#include "MemoryManagement.h"
#include "OverlayDisplayThread.h"
#include "AppletMessageThread.h"
#include "OverlayNotificationThread.h"

//#define NN_DEVOVL_LOG_DEBUG(...) NN_DEVOVL_LOG("[debug]" __VA_ARGS__)

#ifndef NN_DEVOVL_LOG_DEBUG
#define NN_DEVOVL_LOG_DEBUG(...)
#endif

namespace {
    NN_ALIGNAS(4096) char g_FileSystemCache[16 * 1024];
    NN_ALIGNAS(4096) char g_OverlayDisplayThreadStack[OverlayDisplayThreadStackSize];
    NN_ALIGNAS(4096) char g_AppletMessageThreadStack[AppletMessageThreadStackSize];
    NN_ALIGNAS(4096) char g_OverlayNotificationThreadStack[OverlayNotificationThreadStackSize];

    nn::os::ThreadType g_OverlayDisplayThread;
    nn::os::ThreadType g_AppletMessageThread;
    nn::os::ThreadType g_OverlayNotificationThread;

    nn::os::SystemEventType g_AppletMessageSystemEvent;
}


static void DevOverlayMain(nn::ae::OverlayAppletParameters* param) NN_NOEXCEPT
{
    NN_UNUSED(param);
    nn::os::SetThreadNamePointer(nn::os::GetCurrentThread(), "DevOverlayDisp>Main");

    nn::fs::InitializeWithMultiSessionForTargetTool();
    nn::account::InitializeForSystemService();
    nn::time::Initialize();

    // ns の初期化
    nn::ns::Initialize();

    // Rom をマウント
    NN_DEVOVL_LOG_DEBUG("MountRom\n");
    nn::fs::MountRom("rom", g_FileSystemCache, sizeof(g_FileSystemCache));
    NN_UTIL_SCOPE_EXIT{ nn::fs::Unmount("rom"); };

    // 動作モードを決定
    RunMode runMode;
    runMode.Initialize();

    // 通知メッセージイベントを初期化
    NN_DEVOVL_LOG_DEBUG("InitializeNotificationMessageEvent\n");
    nn::ae::InitializeNotificationMessageEvent(&g_AppletMessageSystemEvent);
    // スリープの通知を受け取る
    nn::ae::BeginSleepHandling();

    // オーバーレイ通知を初期化
    NN_DEVOVL_LOG_DEBUG("InitializeReceiverLibraryForOverlay\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ovln::InitializeReceiverLibraryForOverlay());
    NN_UTIL_SCOPE_EXIT{ nn::ovln::FinalizeReceicerLibraryForOverlay(); };

    NN_DEVOVL_LOG_DEBUG("Create AppletMessageChannel\n");
    auto appletMessageToOverlayDisplayChannel = ThreadMessageChannel(AppletMessageQueueLength);
    auto appletMessageToOverlayNotificationChannel = ThreadMessageChannel(AppletMessageQueueLength);
    auto overlayNotificationToOverlayDisplayChannel = ThreadMessageChannel(OverlayNotificationQueueLength);

    NN_DEVOVL_LOG_DEBUG("Prepare ThreadParameter\n");
    OverlayDisplayThreadParameter overlayDisplayThreadParam = {};
    overlayDisplayThreadParam.pAppletMessageChannel = &appletMessageToOverlayDisplayChannel;
    overlayDisplayThreadParam.pOverlayNotificationChannel = &overlayNotificationToOverlayDisplayChannel;
    overlayDisplayThreadParam.pRunMode = &runMode;

    AppletMessageThreadParameter appletMessageThreadParam = {};
    appletMessageThreadParam.pMessageReceivedEvent = &g_AppletMessageSystemEvent;
    appletMessageThreadParam.pToOverlayDisplayChannel = &appletMessageToOverlayDisplayChannel;
    appletMessageThreadParam.pToOverlayNotificationChannel = &appletMessageToOverlayNotificationChannel;
    appletMessageThreadParam.pRunMode = &runMode;

    OverlayNotificationThreadParameter overlayNotificationThreadParam;
    overlayNotificationThreadParam.pAppletMessageChannel = &appletMessageToOverlayNotificationChannel;
    overlayNotificationThreadParam.pToOverlayDisplayChannel = &overlayNotificationToOverlayDisplayChannel;

    NN_DEVOVL_LOG_DEBUG("Create OverlayDisplayThread\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        &g_OverlayDisplayThread,
        OverlayDisplayThreadFunction,
        &overlayDisplayThreadParam,
        g_OverlayDisplayThreadStack,
        sizeof(g_OverlayDisplayThreadStack),
        nn::os::DefaultThreadPriority
    ));
    nn::os::SetThreadNamePointer(&g_OverlayDisplayThread, "DevOverlayDisp>OverlayDisplayThread");

    NN_DEVOVL_LOG_DEBUG("Create AppletMessageThread\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        &g_AppletMessageThread,
        AppletMessageThreadFunction,
        &appletMessageThreadParam,
        g_AppletMessageThreadStack,
        sizeof(g_AppletMessageThreadStack),
        nn::os::DefaultThreadPriority
    ));
    nn::os::SetThreadNamePointer(&g_AppletMessageThread, "DevOverlayDisp>AppletMessageThread");

    NN_DEVOVL_LOG_DEBUG("Create OverlayNotificationThread\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        &g_OverlayNotificationThread,
        OverlayNotificationThreadFunction,
        &overlayNotificationThreadParam,
        g_OverlayNotificationThreadStack,
        sizeof(g_OverlayNotificationThreadStack),
        nn::os::DefaultThreadPriority
    ));
    nn::os::SetThreadNamePointer(&g_OverlayNotificationThread, "DevOverlayDisp>OverlayNotificationThread");

    NN_DEVOVL_LOG_DEBUG("Start Threads\n");
    nn::os::StartThread(&g_OverlayDisplayThread);
    nn::os::StartThread(&g_AppletMessageThread);
    nn::os::StartThread(&g_OverlayNotificationThread);

    nn::os::WaitThread(&g_OverlayDisplayThread);
    nn::os::WaitThread(&g_AppletMessageThread);
    nn::os::WaitThread(&g_OverlayNotificationThread);

    nn::os::DestroyThread(&g_OverlayDisplayThread);
    nn::os::DestroyThread(&g_AppletMessageThread);
    nn::os::DestroyThread(&g_OverlayNotificationThread);
    NN_DEVOVL_LOG_DEBUG("Exit\n");
}

extern "C" void nndiagStartup() NN_NOEXCEPT
{
}

extern "C" void nninitStartup() NN_NOEXCEPT
{
    InitializeAllocator();
}

extern "C" void nnMain() NN_NOEXCEPT
{
    nn::ae::InvokeOverlayAppletMain(nn::ae::AppletId_OverlayApplet, DevOverlayMain);
}
