﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <glv_CustomVerticalListView.h>
#include <glv_ScissorBoxView.h>
#include <nn/os/os_ThreadTypes.h>
#include <nn/util/util_ScopeExit.h>

#include "../Common/DevMenu_CommonRadioButtonsConfirmView.h"
#include "../DevMenu_Config.h"
#include "../DevMenu_RootSurface.h"
#include "DevMenu_SaveData.h"

namespace devmenu { namespace savedata {

namespace {

#if defined ( NN_CUSTOMERSUPPORTTOOL )
    const size_t SaveDataPropertyDefaultLength = 80;
#else
    const size_t SaveDataPropertyDefaultLength = 20;
#endif

    const size_t ApplicationIdStrLength = 16;
    const size_t DateStrLength = 14;

#if defined ( NN_CUSTOMERSUPPORTTOOL )
    const size_t FileNameSaveDataUserIdStrLength = 35;
    const size_t ExportedSaveDataDirNameLength = ApplicationIdStrLength + 1 + DateStrLength + 1 + FileNameSaveDataUserIdStrLength; // yyyymmddhhmmss_<ApplicationId>_<SaveFileUserId>
    // yyyymmddhhmmss_<ApplicationId>_<SaveFileUserId>.nsave
    const size_t BackUpPartitionFSFileExtensionLength = 6;
    const size_t BackUpPartitionFSFileNameLength = ApplicationIdStrLength + DateStrLength + 1 + BackUpPartitionFSFileExtensionLength + 1 + FileNameSaveDataUserIdStrLength;
#else
    const size_t ExportedSaveDataDirNameLength = ApplicationIdStrLength + DateStrLength + 1; // yyyymmddhhmmss_<ApplicationId>
#endif

    struct BackupSaveDataXmlInfo
    {
        uint64_t        saveDataUserIdHi;
        uint64_t        saveDataUserIdLow;
        uint64_t        applicationId;
        nn::fs::UserId  userId;
        uint64_t        saveDataSize;
        nn::Bit64       ownerId;
        uint32_t        saveDataFlags;
        int64_t         saveDataAvailableSize;
        int64_t         saveDataJournalSize;
        std::string     saveDataTypeString;
        uint32_t        saveDataSpaceId;
        int32_t         cacheIndex;
    };

    struct SaveDataInfo
    {
        nn::fs::SaveDataType    type;
        nn::ncm::ApplicationId  applicationId;
        nn::fs::UserId          userId;
        nn::Bit64               ownerId;
        uint64_t                dataSize;
        int64_t                 availableSize;
        int64_t                 journalSize;
        uint32_t                flags;
        uint32_t                spaceId;
        int32_t                 cacheIndex;
        std::string             exportedPath; // セーブデータの実体が保存されているパス
    };
}


/*********************************
 * struct ImportSaveDataProperty
 *********************************/
/**
 * @brief インポート対象のセーブデータプロパティ型です。
 */
template< size_t N >
struct ImportSaveDataProperty
{
    bool operator<( const ImportSaveDataProperty& right ) const NN_NOEXCEPT
    {
        return exportedDateWideStr < right.exportedDateWideStr;
    }

    /**
     * @brief バックアップセーブデータから取得した情報をセットします。
     */
    void SetInfo( const BackupSaveDataXmlInfo& info, const std::string& path ) NN_NOEXCEPT;

    /**
     * @brief セーブデータ情報を取得します。
     */
    const SaveDataInfo& GetInfo() const NN_NOEXCEPT;

    glv::WideCharacterType       saveDataTypeWideStr [ N ];
    glv::WideCharacterType       applicationIdWideStr[ N ];
    glv::WideCharacterType       saveDataSizeWideStr [ N ];
    glv::WideCharacterType       exportedDateWideStr [ N ];

private:
    SaveDataInfo            m_SaveDataInfo;
    nn::os::ThreadType      m_ImportThread;
};

/**
 * @brief インポート対象のセーブデータプロパティ( SaveDataPropertyDefaultLenghth 文字上限 )型です。
 */
typedef ImportSaveDataProperty< SaveDataPropertyDefaultLength > ImportSaveDataPropertyType;

/*********************************
 * class ImportSaveDataListView
 *********************************/
/**
 * @brief リストビュー
 */
class ImportSaveDataListView : public glv::CustomVerticalListView< ImportSaveDataPropertyType >
{
public:
    /**
     * @brief コンストラクタです。
     */
    explicit ImportSaveDataListView( const glv::Rect& parentClipRegion ) NN_NOEXCEPT;

protected:
    /**
     * @copydoc CustomVerticalListView<>::OnQueryBounds( const CustomVerticalListView<>::ItemType&, glv::space_t&, glv::space_t& )
     */
    virtual void OnQueryBounds( const ItemType& item, glv::space_t& outWidth, glv::space_t& outHeight ) NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @copydoc CustomVerticalListView<>::OnDrawItem( const CustomVerticalListView<>::ItemType&, const CustomVerticalListView<>::IndexType, const glv::Rect& )
     */
    virtual void OnDrawItem( const ItemType& item, const IndexType index, const glv::Rect& contentRegion ) NN_NOEXCEPT NN_OVERRIDE;

private:

    static const glv::space_t HorizontalLengthInterval;
    static const glv::space_t HorizontalMarginApplicationId;
    static const glv::space_t HorizontalLengthApplicationId;
};

/*********************************
 * class ConfirmView
 *********************************/
/**
 * @brief インポート確認モーダルビュー( キャッシュストレージ用 )
 */
class ConfirmView : public RadioButtonsConfirmView< nn::fs::SaveDataSpaceId >
{
    NN_DISALLOW_COPY( ConfirmView );
    NN_DISALLOW_MOVE( ConfirmView );

public:
    explicit ConfirmView() NN_NOEXCEPT;

    void ExtentSize() NN_NOEXCEPT; // Expected to be called in final step

    nn::fs::SaveDataSpaceId GetSelectedTargetDevice() NN_NOEXCEPT;

private:
    enum TargetDevice : int
    {
        TargetDevice_SystemMemory,
        TargetDevice_SdCard,
    };

private:
    static const RadioButtons< nn::fs::SaveDataSpaceId >::ButtonInfoList RadioButtonInfoList;
};

/*********************************
 * class ImportScene
 *********************************/
class ImportScene : public SaveDataSceneBase
{
    class Header : public glv::Group
    {
    public:
        Header( glv::Label* pAppIdLabel, glv::Label* pTypeLabel, const glv::Rect& rect )
        : glv::Group( rect ), applicationIdLabel( pAppIdLabel ), typeLabel( pTypeLabel ),
        dateTimeLabel( nullptr ), sizeLabel( nullptr )
        { *this << applicationIdLabel; *this << typeLabel; *this << dateTimeLabel, *this << sizeLabel; }

        glv::Label*       applicationIdLabel;
        glv::Label*       typeLabel;
        glv::Label*       dateTimeLabel;
        glv::Label*       sizeLabel;
    };

    enum ImportState
    {
        ImportState_None,
        ImportState_Preparing,
        ImportState_Running,
        ImportState_Completed
    };

public:

    ImportScene( Page* pParentPage, const glv::Rect& rect ) NN_NOEXCEPT;

    ~ImportScene() NN_NOEXCEPT {}

    /**
     * @brief SD カードをマウントしインポート対象のセーブデータ情報を取得します。
     */
    nn::Result Prepare() NN_NOEXCEPT;

    /**
     * @brief glv シーンレンダラ前のループ処理です。glv シーンレンダラへ hid 系イベントが通知される前に呼び出されます。
     */
    virtual void OnLoopBeforeSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT NN_OVERRIDE;

private:

    /**
     * @brief BackupSaveData 情報を取得します。
     */
    const nn::Result GetBackupSaveDataInfo() NN_NOEXCEPT;

    /**
     * @brief SD カードにエクスポートされたセーブデータのプロパティコレクションを登録します。
     * @details 登録内容に応じてリスト or アイテムなしメッセージの切り替えを行います。
     */
    void EntryProperties() NN_NOEXCEPT;

    /**
     * @brief セーブデータを問い合わせます。
     */
    void QueryProperties() NN_NOEXCEPT;

    void QueryPropertiesImpl() NN_NOEXCEPT;

    /**
     * @brief 問い合わせ済プロパティデータを解放します。
     */
    void FinalizeProperties() NN_NOEXCEPT;

    /**
     * @brief セーブデータをソートします。
     */
    void QuickSort() NN_NOEXCEPT;

    /**
     * @brief リスト更新通知ハンドラー
     */
    static void OnPropertyListUpdateNotification( const glv::Notification& notification ) NN_NOEXCEPT;

    /**
     * @brief セーブデータを本体にインポートするか確認するダイアログを表示します。
     */
    void ConfirmImport() NN_NOEXCEPT;

    /**
     * @brief インポート対象のセーブデータについて、ユーザーがインポートを決定した際に呼ばれます。
     */
    const nn::Result Import( const SaveDataInfo& info, nn::fs::SaveDataSpaceId spaceId = nn::fs::SaveDataSpaceId::User ) NN_NOEXCEPT;

    /**
     * @brief インポートを別スレッドで実行します。
     */
    void StartImportThread( const nn::account::Uid& uid, nn::fs::SaveDataSpaceId spaceId ) NN_NOEXCEPT;

    /**
     * @brief SD カードのバックアップデータを本体にコピーして復元します。本体にセーブデータが存在する場合、削除してからコピーします。
     */
    const nn::Result RestoreSaveDataImpl( const nn::account::Uid& uid, nn::fs::SaveDataSpaceId spaceId ) NN_NOEXCEPT;

    /**
     * @brief インポート先の本体に保存されているセーブデータを削除します。インポート対象のセーブデータと同アプリケーション ID, ユーザー ID が対象です。
     */
    const nn::Result DeleteSaveDataInDevice( const nn::fs::UserId& userId ) NN_NOEXCEPT;

    const nn::Result DeleteSaveDataInDeviceImpl( const nn::fs::UserId& userId, nn::fs::SaveDataSpaceId spaceId ) NN_NOEXCEPT;

    /**
     * @brief インポート先の本体にセーブデータを作成します。
     */
    const nn::Result CreateSaveData( const nn::fs::UserId& userId, nn::fs::SaveDataSpaceId spaceId ) NN_NOEXCEPT;

    /**
     * @brief 描画内容をリフレッシュします。
     */
    virtual void Refresh() NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief セーブデータのインポート処理が完了した際に呼ぶコールバック関数です。
    */
    void OnCompleteImportProgress( const nn::Result& result ) NN_NOEXCEPT;

private:
    ImportSaveDataListView::CollectionType*    m_pItems;
    ImportSaveDataListView*                    m_pListView;
    glv::Label*                                m_pLabelNoItem;

    nn::Result                                 m_QueryFsResult;

    SaveDataInfo                               m_TargetSaveDataInfo;
};

}} // ~namespace devmenu::savedata, ~namespace devmenu

