﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <functional>

#include <nn/nn_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_SdCardApi.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_StringUtil.h>

#if defined ( NN_CUSTOMERSUPPORTTOOL )
    #include <nn/fs/fs_MemoryStorage.h>
    #include <nn/fs/fs_SaveDataPrivate.h>
    #include <nn/fs/fsa/fs_Registrar.h>
    #include <nn/fssystem/fs_PartitionFileSystem.h>
#endif

#include "DevMenu_SaveDataConfig.h"
#include "DevMenu_SaveDataImporter.h"
#include "DevMenu_SaveDataUserSelector.h"
#include "DevMenu_SaveData.h"

#if defined ( NN_CUSTOMERSUPPORTTOOL )
    #include "DevMenu_SaveDataCustomerSupportTool.h"
#endif

#include "../DevMenu_Result.h"
#include "../../BackupSaveData/BackupSaveData.h"

namespace devmenu { namespace savedata {

namespace {

    const int MaxPropertyShowCount = 1024;

   /**
    * @brief   指定した長さの文字列が数字を表すかをチェックします。
    * @return  数字を表す場合は true を返します。
    */
    bool HasValueString( const std::string& checkString, int checkLength ) NN_NOEXCEPT
    {
        errno = 0;
        std::strtoull( checkString.c_str(), NULL, checkLength );  // only for check xdigit

        if ( !errno )
        {
            return true;
        }
        return false;
    }

    /**
     * @brief __BackupSaveDataInfo.xml から取得したセーブデータタイプ文字列を元にセーブデータタイプを取得します。
     */
    nn::fs::SaveDataType GetSaveDataType( const std::string& typeString ) NN_NOEXCEPT
    {
        if ( typeString == "System" )
        {
            return nn::fs::SaveDataType::System;
        }
        else if ( typeString == "Account" )
        {
            return nn::fs::SaveDataType::Account;
        }
        else if ( typeString == "Bcat" )
        {
            return nn::fs::SaveDataType::Bcat;
        }
        else if ( typeString == "Device" )
        {
            return nn::fs::SaveDataType::Device;
        }
        else if ( typeString == "Temporary" )
        {
            return nn::fs::SaveDataType::Temporary;
        }
        else if ( typeString == "Cache Storage" || typeString == "Cache" )
        {
            return nn::fs::SaveDataType::Cache;
        }
        else
        {
            return nn::fs::SaveDataType::Account;
        }
    }

   /**
    * @brief  指定した Aplication ID が有効値かをチェックします。
    * @return 有効な値なら true を返します。
    */
    bool IsApplicationIdValid( uint64_t applicationId ) NN_NOEXCEPT
    {
        return nn::fs::InvalidProgramId.value != applicationId;
    }

#if !defined( NN_CUSTOMERSUPPORTTOOL )
    bool CheckParsedSaveDataInfo( nn::fs::SaveDataType saveDataType, int32_t cacheIndex ) NN_NOEXCEPT
    {
        if(saveDataType == nn::fs::SaveDataType::Cache)
        {
            if(cacheIndex < 0)
            {
                return false;
            }
        }
        return true;
    }

    /**
     * @brief 画面に表示するセーブデータサイズの文字列を取得します。
     */
    void GetConvertedDataSizeStr( char* pOutStr, size_t dataSize ) NN_NOEXCEPT
    {
        float dataSizeF = static_cast< float >( dataSize );
        int count = 0;
        while ( dataSizeF > 1024.0f )
        {
            dataSizeF /= 1024;
            count++;
        }
        switch ( count )
        {
        case 0:
            nn::util::SNPrintf( pOutStr, sizeof( pOutStr ), "%lld Byte", dataSize );
            break;
        case 1:
            nn::util::SNPrintf( pOutStr, sizeof( pOutStr ), "%.1f KB", dataSizeF );
            break;
        case 2:
            nn::util::SNPrintf( pOutStr, sizeof( pOutStr ), "%.1f MB", dataSizeF );
            break;
        case 3:
            nn::util::SNPrintf( pOutStr, sizeof( pOutStr ), "%.1f GB", dataSizeF );
            break;
        default:
            nn::util::SNPrintf( pOutStr, sizeof( pOutStr ), "%s", "Over 1TB" );
            break;
        }
    }

   /**
    * @brief  config.xml が存在するかをチェックします。
    * @return 存在する場合は true を返します。
    */
    bool ExistsConfigFile( const std::string& configPath ) NN_NOEXCEPT
    {
        // ToDo: 本当は config.xml の中身までチェックすべきだが、ひとまず存在有無だけチェック
        return ExistsFile( configPath );
    }

   /**
    * @brief  エクスポートしたセーブデータが規定フォーマットの名前であるかをチェックします。
    * @return 規定フォーマットなら true を返します。
    */
    bool IsDirectoryNameValid( const std::string& directoryNameStr ) NN_NOEXCEPT
    {
        // Directory name format is yyyymmddhhmmss_<applicaitonId> and applicationId has 16 charactres.

        const char* directoryName = directoryNameStr.c_str();

        return ( ExportedSaveDataDirNameLength == directoryNameStr.length() ) &&
            ( 0 == nn::util::Strncmp( directoryName + DateStrLength, "_", 1 ) ) &&
            HasValueString( directoryNameStr, DateStrLength ) &&
            HasValueString( std::string( directoryName + DateStrLength + 1 ), ApplicationIdStrLength );
    }

    /**
     * @brief 指定したディレクトリパス以下のデータサイズを取得します。GetExportedSaveDataSize() 内で使用されます。
     */
    const nn::Result GetDirectorySizeRecursively( size_t* pOutSize, const std::string& directoryPath ) NN_NOEXCEPT
    {
        nn::fs::DirectoryHandle directoryHandle;
        int64_t readEntryCount = 0;
        size_t totalFileSize = 0;

        NN_RESULT_DO( nn::fs::OpenDirectory( &directoryHandle, directoryPath.c_str(), nn::fs::OpenDirectoryMode_All ) );
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseDirectory( directoryHandle );
        };

        while ( NN_STATIC_CONDITION( true ) )
        {
            std::unique_ptr< nn::fs::DirectoryEntry > directoryEntryArea;
            NN_RESULT_DO( nn::fs::ReadDirectory( &readEntryCount, directoryEntryArea.get(), directoryHandle, 1 ) );

            if ( 0 == readEntryCount )
            {
                break;
            }

            if ( IsSaveDataInfoFile( directoryEntryArea->name ) )
            {
                continue;
            }

            if ( nn::fs::DirectoryEntryType::DirectoryEntryType_File == directoryEntryArea->directoryEntryType )
            {
                nn::fs::FileHandle handle;
                int64_t fileSize;
                const std::string targetPath = directoryPath + '/' + directoryEntryArea->name;
                NN_RESULT_DO( nn::fs::OpenFile( &handle, targetPath.c_str(), nn::fs::OpenMode_Read ) );
                auto result = nn::fs::GetFileSize( &fileSize, handle );
                nn::fs::CloseFile( handle );
                if ( result.IsFailure() )
                {
                    return result;
                }
                totalFileSize += static_cast< size_t >( fileSize );
            }
            else if ( nn::fs::DirectoryEntryType::DirectoryEntryType_Directory == directoryEntryArea->directoryEntryType )
            {
                size_t dataSize = 0;
                const std::string nextDirPath = directoryPath + '/' + directoryEntryArea->name;
                NN_RESULT_DO( GetDirectorySizeRecursively( &dataSize, nextDirPath ) );
                totalFileSize += dataSize;
            }
        }

        *pOutSize += totalFileSize;

        NN_RESULT_SUCCESS;
    }

    /**
     * @brief 指定したディレクトリ以下のデータサイズを取得します。
     */
    const nn::Result GetExportedSaveDataSize( size_t* pOutSize, const std::string& saveDataDirPath ) NN_NOEXCEPT
    {
        size_t exportedSaveDataSizeSum = 0;

        NN_RESULT_DO( GetDirectorySizeRecursively( &exportedSaveDataSizeSum, saveDataDirPath ) );
        *pOutSize = exportedSaveDataSizeSum;

        NN_RESULT_SUCCESS;
    }

    /**
     * @brief 指定したディレクトリ以下に存在するセーブデータディレクトリから BackupSaveData 情報とディレクトリパスを返します。
     */
    const nn::Result ReadBackupSaveDataInfo( BackupSaveDataXmlInfo* pOutInfo, std::string* pOutPath, const std::string& applicationIdDirPath, int64_t applicationId ) NN_NOEXCEPT
    {
        nn::fs::DirectoryHandle directoryHandle;
        std::unique_ptr< nn::fs::DirectoryEntry> directoryEntryArea( new nn::fs::DirectoryEntry );
        int64_t readEntryCount = int64_t();

        // 現時点では 4 桁の ディレクトリが 1 つ存在するだけのはずだが一応チェックする（複数時はサポートしない）
        NN_RESULT_DO( nn::fs::OpenDirectory( &directoryHandle, applicationIdDirPath.c_str(), nn::fs::OpenDirectoryMode_Directory ) );
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseDirectory( directoryHandle );
        };
        NN_RESULT_DO( nn::fs::GetDirectoryEntryCount( &readEntryCount, directoryHandle ) );

        if ( 1 != readEntryCount )
        {
            pOutInfo->applicationId = 0x0ULL; // TORIAEZU: セーブデータが存在しない場合は無効な ID をセットする
            NN_RESULT_SUCCESS;
        }

        NN_RESULT_DO( nn::fs::ReadDirectory( &readEntryCount, directoryEntryArea.get(), directoryHandle, 1 ) );

        const std::string saveDataBinaryStoredPath = *pOutPath = applicationIdDirPath + '/' + directoryEntryArea->name;
        NN_RESULT_DO( ParseSaveDataInfoFile( saveDataBinaryStoredPath, pOutInfo->saveDataTypeString, pOutInfo->saveDataUserIdHi, pOutInfo->saveDataUserIdLow,
                                             pOutInfo->applicationId, pOutInfo->saveDataSize, pOutInfo->ownerId, pOutInfo->saveDataFlags,
                                             reinterpret_cast< uint64_t& >( pOutInfo->saveDataAvailableSize ),
                                             reinterpret_cast< uint64_t& >( pOutInfo->saveDataJournalSize ),
                                             pOutInfo->saveDataSpaceId, pOutInfo->cacheIndex ) );
        nn::fs::UserId userId = { { pOutInfo->saveDataUserIdHi, pOutInfo->saveDataUserIdLow } };
        pOutInfo->userId = userId;
        nn::fs::SaveDataType saveDataType = GetSaveDataType( pOutInfo->saveDataTypeString );

        if ( 0x00LL == pOutInfo->saveDataSize )
        {
            DEVMENU_LOG( "Invalid save data size is described in xml file. Calculate save data size.\n" );
            NN_RESULT_DO( GetExportedSaveDataSize( reinterpret_cast< size_t* >( pOutInfo->saveDataSize ), applicationIdDirPath + directoryEntryArea->name ) );
        }

    #if defined( SKIP_BACKUP_DEVICE_SAVEDATA )
        if ( nn::fs::SaveDataType::Device == saveDataType )
        {
            DEVMENU_LOG( "Skip backup save data type %d.\n", nn::fs::SaveDataType::Device );
        }
    #endif

        NN_RESULT_SUCCESS;
    }

#else

   /**
    * @brief  nsaveが規定フォーマットの名前であるかをチェックします。
    * @return 規定フォーマットなら true を返します。
    */
    bool IsNsaveFileNameValid( const std::string& directoryNameStr ) NN_NOEXCEPT
    {
        // Directory name format is yyyymmddhhmmss_<applicaitonId> and applicationId has 16 charactres.

        const char* directoryName = directoryNameStr.c_str();

        return ( BackUpPartitionFSFileNameLength == directoryNameStr.length() ) &&
            ( 0 == nn::util::Strncmp( directoryName + DateStrLength, "_", 1 ) ) &&
            ( 0 == nn::util::Strncmp( directoryName + ExportedSaveDataDirNameLength, DataMoveInfo::NsaveFilenameExtension.c_str(), DataMoveInfo::NsaveFilenameExtension.length() ) ) &&
            HasValueString( directoryNameStr, DateStrLength ) &&
            HasValueString( std::string( directoryName + DateStrLength + 1 ), ApplicationIdStrLength );
    }

    const nn::Result ReadBackupSaveDataInfoForNsave( BackupSaveDataXmlInfo* pOutInfo, const std::string& path ) NN_NOEXCEPT
    {
        nn::fs::FileHandle readHandle;
        int64_t fileSize;
        NN_RESULT_DO( MountSdCard( "sd" ) );
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount( "sd" );
        };

        NN_RESULT_DO( nn::fs::OpenFile( &readHandle, path.c_str(), nn::fs::OpenMode_Read ) );
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile( readHandle );
        };

        NN_RESULT_DO(nn::fs::GetFileSize( &fileSize, readHandle ) );
        std::unique_ptr< char[] > buffer( new char[ fileSize ] );
        NN_RESULT_DO(nn::fs::ReadFile( readHandle, 0, buffer.get(), fileSize ) );

        std::unique_ptr< nn::fssystem::PartitionFileSystem > fs( new nn::fssystem::PartitionFileSystem() );
        nn::fs::MemoryStorage fsStorage( buffer.get(), fileSize );

        NN_RESULT_DO( fs->Initialize( &fsStorage ) );
        NN_RESULT_DO( nn::fs::fsa::Register( "nsave", std::move( fs ) ) );
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::fsa::Unregister( "nsave" );
        };

        nn::fs::DirectoryHandle directoryHandle;
        nn::fs::DirectoryEntry* directoryEntryArea = nullptr;
        int64_t readEntryCount = int64_t();

        // 子エントリ情報の取得 (ファイルのみ)
        NN_RESULT_DO( GetFsEntries( &directoryHandle, &directoryEntryArea, &readEntryCount, "nsave:/", nn::fs::OpenDirectoryMode_File ) );

        std::string saveDataBaseDirectoryPath;
        for( auto i = 0; i < readEntryCount; ++i )
        {
            if( strlen( directoryEntryArea[ i ].name ) != SaveDataInfoFileName.size() + DataMoveInfo::DirectoryStrLength
                && 0 != strncmp( &directoryEntryArea[ i ].name[ DataMoveInfo::DirectoryStrLength - 1 ], SaveDataInfoFileName.c_str(), SaveDataInfoFileName.size() ) )
            {
                // __BackupSaveDataInfo.xml ではない
                continue;
            }
            // ～__BackupSaveDataInfo.xml のフルパスを取得
            saveDataBaseDirectoryPath = directoryEntryArea[ i ].name;
            saveDataBaseDirectoryPath = "nsave:/" + saveDataBaseDirectoryPath;

            // pathから /__BackupSaveDataInfo.xml を削除
            saveDataBaseDirectoryPath.erase( saveDataBaseDirectoryPath.end() - SaveDataInfoFileName.size() - 1, saveDataBaseDirectoryPath.end() );
            break;
        }

        if ( true == saveDataBaseDirectoryPath.empty() )
        {
            DEVMENU_LOG( "__BackupSaveDataInfo.xml is not found\n" );
            return ResultApplicationError();
        }

        NN_RESULT_DO( ParseSaveDataInfoFile( saveDataBaseDirectoryPath, pOutInfo->saveDataTypeString, pOutInfo->saveDataUserIdHi, pOutInfo->saveDataUserIdLow,
            pOutInfo->applicationId, pOutInfo->saveDataSize, pOutInfo->ownerId, pOutInfo->saveDataFlags,
            reinterpret_cast< uint64_t& >( pOutInfo->saveDataAvailableSize ),
            reinterpret_cast< uint64_t& >( pOutInfo->saveDataJournalSize ),
            pOutInfo->saveDataSpaceId, pOutInfo->cacheIndex ) );
        nn::fs::UserId userId = { { pOutInfo->saveDataUserIdHi, pOutInfo->saveDataUserIdLow } };
        pOutInfo->userId = userId;

        if ( 0x00LL == pOutInfo->saveDataSize )
        {
            DEVMENU_LOG("Invalid save data size is describe in xml file. Calculate save data size.\n");
            return ResultApplicationError();
        }

        NN_RESULT_SUCCESS;
    }

#endif // !defined ( NN_CUSTOMERSUPPORTTOOL )

} // end of anonymous namespace

/*********************************
 * struct ImportSaveDataProperty
 *********************************/
template< size_t N > void ImportSaveDataProperty< N >::SetInfo( const BackupSaveDataXmlInfo& info, const std::string& path ) NN_NOEXCEPT
{
    m_SaveDataInfo = {
        GetSaveDataType( info.saveDataTypeString ),
        { info.applicationId },
        { { info.saveDataUserIdHi, info.saveDataUserIdLow } },
        info.ownerId,
        info.saveDataSize,
        info.saveDataAvailableSize,
        info.saveDataJournalSize,
        info.saveDataFlags,
        info.saveDataSpaceId,
        info.cacheIndex,
        path
    };
}

template< size_t N > const SaveDataInfo& ImportSaveDataProperty< N >::GetInfo() const NN_NOEXCEPT
{
    return m_SaveDataInfo;
}

/*********************************
 * class ImportSaveDataListView
 *********************************/
const glv::space_t ImportSaveDataListView::HorizontalLengthInterval = 8.f;
const glv::space_t ImportSaveDataListView::HorizontalMarginApplicationId = 4.f;
const glv::space_t ImportSaveDataListView::HorizontalLengthApplicationId = 344.f;

ImportSaveDataListView::ImportSaveDataListView( const glv::Rect& parentClipRegion ) NN_NOEXCEPT
    : CustomVerticalListView( parentClipRegion )
{
    SetTouchAndGo( true );
    glv::Style* pStyle = new glv::Style();
    pStyle->color = glv::Style::standard().color;
    pStyle->color.selection.set( 0.1f, 0.85f, 0.2f );
    style( pStyle );
    font().size( 25.0f );
}

void ImportSaveDataListView::OnQueryBounds( const ItemType& item, glv::space_t& outWidth, glv::space_t& outHeight ) NN_NOEXCEPT
{
    this->font().getBounds( outWidth, outHeight, item.applicationIdWideStr );
    outWidth = this->width();
}

void ImportSaveDataListView::OnDrawItem( const ItemType& item, const IndexType index, const glv::Rect& contentRegion ) NN_NOEXCEPT
{
    NN_UNUSED( index );
    glv::Font& font = this->font();

    const glv::space_t posId = contentRegion.left() + ImportSaveDataListView::HorizontalMarginApplicationId;
    font.render( item.applicationIdWideStr, posId, contentRegion.top() );

#if !defined ( NN_CUSTOMERSUPPORTTOOL )
    const glv::space_t posType = posId + ImportSaveDataListView::HorizontalLengthApplicationId;
    font.render( item.saveDataTypeWideStr, posType, contentRegion.top() );

    const glv::space_t posSize = posType + 220.f + ImportSaveDataListView::HorizontalLengthInterval;
    font.render( item.saveDataSizeWideStr, posSize, contentRegion.top() );

    glv::space_t outWidth, outHeight;
    font.getBounds( outWidth, outHeight, item.exportedDateWideStr );
    const glv::space_t ownerExpect = contentRegion.right() - ( outWidth + ( this->paddingX() * 2 ) );
    const glv::space_t ownerLimit = posId + ImportSaveDataListView::HorizontalMarginApplicationId + ImportSaveDataListView::HorizontalLengthApplicationId + 8.f;

    font.render( item.exportedDateWideStr, ( ownerExpect < ownerLimit ) ? ownerLimit : ownerExpect, contentRegion.top() );
#endif
}

/**************************************
 * class ConfirmView
 **************************************/

// Note:  You must ensure that the labels are in the same order as in the TargetInstallLocation enum
const RadioButtons< nn::fs::SaveDataSpaceId >::ButtonInfoList ConfirmView::RadioButtonInfoList = {
    { TargetDevice_SystemMemory, nn::fs::SaveDataSpaceId::User, "System Memory" },
    { TargetDevice_SdCard, nn::fs::SaveDataSpaceId::SdUser, "SD Card" },
};

ConfirmView::ConfirmView() NN_NOEXCEPT
    : RadioButtonsConfirmView< nn::fs::SaveDataSpaceId >( RadioButtonInfoList, TargetDevice_SystemMemory, true )
{
    Remove();

    // Radio Button Selection
    auto pCaptionMessageTable = new glv::Table( "< <" ); // 何故か余計に一段 Table を挟まないとレイアウトが期待どおりにならない
    auto pLabel = new glv::Label( "Target Location:" );
    pLabel->size( CommonValue::InitialFontSize ).pos( glv::Place::CL ).anchor( glv::Place::CL );
    *pCaptionMessageTable << new Spacer( 20.0f, 0.0f ) << pLabel;
    pCaptionMessageTable->anchor( glv::Place::CL ).pos( glv::Place::CL );
    pCaptionMessageTable->arrange();
    pCaptionMessageTable->enable( glv::Property::KeepWithinParent );

    auto pRadioButtonSelectionTable = new glv::Table( "<" );
    GetRadioButtons().SetPosition( glv::Place::CL, glv::Place::CL );
    *pRadioButtonSelectionTable << pCaptionMessageTable << GetRadioButtons();
    pRadioButtonSelectionTable->arrange().fit( false );
    pRadioButtonSelectionTable->anchor( glv::Place::CL ).pos( glv::Place::CL );
    pRadioButtonSelectionTable->enable( glv::Property::KeepWithinParent );

    // ModalView Tables
    m_ViewTable.arrangement( "<, <, x" );
    m_ViewTable << m_MessagesTable << pRadioButtonSelectionTable << m_ButtonsTable;
    m_ViewTable.arrange();
    *this << m_ViewTable;
}

/*********************************
 * class ImportScene
 *********************************/
ImportScene::ImportScene( Page* pParentPage, const glv::Rect& rect ) NN_NOEXCEPT
    : SaveDataSceneBase( pParentPage, rect, true )
    , m_pItems( nullptr )
    , m_pListView( nullptr )
    , m_pLabelNoItem( nullptr)
    , m_QueryFsResult( nn::ResultSuccess() )
{
    // シザーボックスの領域設定
    const glv::space_t headerRegion = 48.0f;
    const glv::space_t footerRegion = 64.0f;
    glv::ScissorBoxView* pListContainer = new glv::ScissorBoxView( 0, headerRegion, width(), height() - ( headerRegion + footerRegion ) );

    // リスト要素のパディング ( ボーダーを表示する場合は 2 以上必要 )
    const glv::space_t listMarginLeft = 16.0f;
    const glv::space_t listMarginRight = 16.0f;
    const glv::space_t listMarginVertical = 12.0f;

    // ヘッダ
    const glv::space_t headerFontSize = 25.f;
    const glv::space_t applicationIdHorizontalPos = listMarginLeft + 8.f;

#if !defined ( NN_CUSTOMERSUPPORTTOOL )
    auto pSaveDataLabel = new glv::Label( "Application ID", glv::Label::Spec( glv::Place::TL, applicationIdHorizontalPos     , listMarginVertical, headerFontSize ) );
    auto pTypeLabel     = new glv::Label( "Type",           glv::Label::Spec( glv::Place::TL, pSaveDataLabel->right() + 180.f, listMarginVertical, headerFontSize ) );
    auto pSizeLabel     = new glv::Label( "Size",           glv::Label::Spec( glv::Place::TL, pTypeLabel->right() + 170.f    , listMarginVertical, headerFontSize ) );
    auto pDateLabel     = new glv::Label( "Date",           glv::Label::Spec( glv::Place::TL, pSizeLabel->right() + 150.f    , listMarginVertical, headerFontSize ) );

    *this << pSaveDataLabel << pTypeLabel << pSizeLabel << pDateLabel;
#else
    auto pSaveDataLabel = new glv::Label( "nsave FileName", glv::Label::Spec( glv::Place::TL, applicationIdHorizontalPos     , listMarginVertical, headerFontSize ) );
    *this << pSaveDataLabel;
#endif

    // フッタ
    auto footer = new glv::Label( " A:Import", glv::Label::Spec( glv::Place::BL, listMarginLeft, - 8.f - 15.f - 8.f, 25.f ) );

    *this << footer;

    // リストビュー
    glv::Rect clipRegion( listMarginLeft, listMarginVertical, pListContainer->width() - ( listMarginLeft + listMarginRight ), pListContainer->height() - ( listMarginVertical * 2 ) );
    ImportSaveDataListView* pListView = new ImportSaveDataListView( clipRegion );
    pListView->attach( OnPropertyListUpdateNotification, glv::Update::Action, this );
    // pListView->attach( OnPropertyListSelectionNotification, glv::Update::Selection, this );

    *pListContainer << pListView;
    m_pListView = pListView;

    // アイテムなしメッセージ
    glv::Label* pLabelNoItem = new glv::Label( "No save data is exported.", glv::Label::Spec( glv::Place::CC, 0, 0, 32.f ) );
    *pListContainer << pLabelNoItem;
    m_pLabelNoItem = pLabelNoItem;

    *this << pListContainer;
    disable( glv::Visible );
}

const nn::Result ImportScene::GetBackupSaveDataInfo() NN_NOEXCEPT
{
    nn::fs::DirectoryHandle directoryHandle;

    NN_RESULT_TRY( nn::fs::OpenDirectory( &directoryHandle, DataMoveInfo::ExportPathRoot.c_str(), nn::fs::OpenDirectoryMode_All ) );
        NN_RESULT_CATCH( nn::fs::ResultPathNotFound )
        {
            // セーブデータディレクトリが存在しない場合は SUCCESS を返す
            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY;

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseDirectory( directoryHandle );
    };

#if defined ( NN_CUSTOMERSUPPORTTOOL )
    int itemsCount = 0;
#endif

    while ( NN_STATIC_CONDITION( true ) )
    {
        std::unique_ptr< nn::fs::DirectoryEntry > directoryEntryArea( new nn::fs::DirectoryEntry );
        int64_t readEntryCount = int64_t();

        if ( nn::fs::ReadDirectory( &readEntryCount, directoryEntryArea.get(), directoryHandle, 1 ).IsSuccess() )
        {
            if ( 0 == readEntryCount )
            {
                break;
            }

#if !defined ( NN_CUSTOMERSUPPORTTOOL )
            if ( !IsDirectoryNameValid( directoryEntryArea->name ) )
            {
                continue;
            }
#else
            if ( IsNsaveFileNameValid( directoryEntryArea->name ) )
            {
                NN_ASSERT_NOT_NULL( m_pItems );
                m_pItems->resize( itemsCount + 1 );
                BackupSaveDataXmlInfo info;
                BuildUtf16< BackUpPartitionFSFileNameLength + 1 >( m_pItems->at( itemsCount ).applicationIdWideStr, "%s", directoryEntryArea->name );
                m_pItems->at( itemsCount ).SetInfo( info, directoryEntryArea->name );
                itemsCount++;
            }
#endif
        }
        else
        {
            continue;
        }

#if !defined ( NN_CUSTOMERSUPPORTTOOL )

        // 日付
        std::string dateStr;
        char tempDateStr[ DateStrLength ];
        const size_t inputLength = static_cast< size_t >( nn::util::Strlcpy( tempDateStr, directoryEntryArea->name, 9 ) ); // 9 -> yyyymmdd + 終端文字
        if ( ExportedSaveDataDirNameLength != inputLength )
        {
            continue;
        }
        dateStr = tempDateStr;
        nn::util::Strlcpy( tempDateStr, directoryEntryArea->name + 8, 7 ); // 7 -> hhmmdd + 終端文字
        dateStr += "_" + std::string( tempDateStr );

        // Application ID
        const char* applcationIdStr = directoryEntryArea->name + DateStrLength + 1;
        const uint64_t applicationId = std::stoull( directoryEntryArea->name + DateStrLength + 1, nullptr, ApplicationIdStrLength );
        if ( !IsApplicationIdValid( applicationId ) )
        {
            continue;
        }

        // config.xml のチェック（中身は読み込まないので存在だけチェック）
        const std::string configFilePath = DataMoveInfo::ExportPathRoot + '/' + directoryEntryArea->name + '/' + DataMoveInfo::ConfigFileName;
        if ( !ExistsConfigFile( configFilePath ) )
        {
            continue;
        }

        // __BackupSaveDataInfo.xml の読み込み
        BackupSaveDataXmlInfo info;
        std::string saveDataStoredPath;
        int foundCounts = 0;

        for ( const auto& iter : DataMoveInfo::TypeDirectoryInfo )
        {
            const std::string applicationIdDirectoryPath = DataMoveInfo::ExportPathRoot + '/' + directoryEntryArea->name + iter.name + applcationIdStr;

            if ( ReadBackupSaveDataInfo( &info, &saveDataStoredPath, applicationIdDirectoryPath, applicationId ).IsSuccess() )
            {
                //  アプリケーション ID が無効 or アプリケーション ID が不一致
                if ( nn::fs::InvalidApplicationId.value == info.applicationId || applicationId != info.applicationId )
                {
                    continue;
                }
                // Export の仕様上、最大で 1 つしかヒットしないはず
                foundCounts++;
                break;
            }
            else
            {
                // 読み込み失敗
                continue;
            }
        }

        if ( 0 == foundCounts )
        {
            continue;
        }

        char saveDataSizeStr[ 10 ];
        GetConvertedDataSizeStr( saveDataSizeStr, info.saveDataSize );
        auto saveDataType = GetSaveDataType( info.saveDataTypeString );

        NN_ASSERT_NOT_NULL( m_pItems );

        // ListView への登録
        const size_t itemSize = m_pItems->size();
        m_pItems->resize( itemSize + 1 );
        BuildUtf16< DateStrLength + 2 >( m_pItems->at( itemSize ).exportedDateWideStr, "%s", dateStr.c_str() ); // +2 -> 区切り文字 & 終端文字
        BuildUtf16< ApplicationIdStrLength + 1 >( m_pItems->at( itemSize ).applicationIdWideStr, "0x%s", applcationIdStr );
        if ( nn::fs::SaveDataType::Cache == saveDataType )
        {
            BuildUtf16< SaveDataPropertyDefaultLength >( m_pItems->at( itemSize ).saveDataTypeWideStr, "%s (index %d)", info.saveDataTypeString.c_str(), info.cacheIndex );
        }
        else
        {
            BuildUtf16< SaveDataPropertyDefaultLength >( m_pItems->at( itemSize ).saveDataTypeWideStr, "%s", info.saveDataTypeString.c_str() );
        }
        BuildUtf16< SaveDataPropertyDefaultLength >( m_pItems->at( itemSize ).saveDataSizeWideStr, "%s", saveDataSizeStr );
        m_pItems->at( itemSize ).SetInfo( info, saveDataStoredPath );
#endif
    }

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

/**
 * @brief SD カードをマウントしインポート対象のセーブデータ情報を取得します。
 */
nn::Result ImportScene::Prepare() NN_NOEXCEPT
{
#if !defined ( NN_BUILD_CONFIG_OS_WIN )
    NN_RESULT_DO( MountSdCard( "sd" ) );
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount( "sd" );
    };

    NN_RESULT_DO( GetBackupSaveDataInfo() );
#endif

    NN_RESULT_SUCCESS;
}

void ImportScene::OnLoopBeforeSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT
{
    // TORIAEZU: Refresh の段階で StartModal すると InvisibleWall が表示されないので遅らせて StatrModal を実行する
    if ( m_QueryFsResult.IsFailure() )
    {
        auto pView = new MessageView( false );
        if (nn::fs::ResultInvalidFatFormatSd::Includes(m_QueryFsResult))
        {
            CreateErrorMessageView( pView, "Fat format of SD card is invalid.", m_QueryFsResult );
        }
        else
        {
            CreateErrorMessageView( pView, "Failed to mount the SD card or read the save data.", m_QueryFsResult );
        }
        this->StartModal( pView );
        m_QueryFsResult = nn::ResultSuccess();
    }

    this->UpdateExecutionProgress();
}

void ImportScene::EntryProperties() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL( m_pItems );

    const glv::Property::t focusableProperty = glv::Property::t::Controllable | glv::Property::t::HitTest;
    auto* const pLabelNoItem = m_pLabelNoItem;
    if ( nullptr != m_pListView )
    {
        m_pListView->EntryCollection( *m_pItems );
        if ( true == m_pItems->empty() )
        {
            m_pListView->disable( focusableProperty );
            pLabelNoItem->enable( glv::Property::t::Visible );
            pLabelNoItem->bringToFront();
        }
        else
        {
            m_pListView->enable( focusableProperty );
            m_pLabelNoItem->disable( glv::Property::t::Visible );
        }
    }
    enable( glv::Visible );
}

void ImportScene::QueryProperties() NN_NOEXCEPT
{
    FinalizeProperties();

    m_pItems = new ImportSaveDataListView::CollectionType();
    m_pItems->reserve( MaxPropertyShowCount );
    m_QueryFsResult = Prepare();
    QueryPropertiesImpl();
    QuickSort();

    // プロパティリスト登録
    EntryProperties();
}

void ImportScene::QueryPropertiesImpl() NN_NOEXCEPT
{
}

void ImportScene::QuickSort() NN_NOEXCEPT
{
    std::sort( m_pItems->begin(), m_pItems->end() );
}

void ImportScene::FinalizeProperties() NN_NOEXCEPT
{
    if ( nullptr != m_pItems )
    {
        delete m_pItems;
        m_pItems = nullptr;
    }
}

void ImportScene::OnPropertyListUpdateNotification( const glv::Notification& notification ) NN_NOEXCEPT
{
    notification.receiver< ImportScene >()->ConfirmImport();
}

void ImportScene::ConfirmImport() NN_NOEXCEPT
{
    const auto pSelectedItem = m_pListView->GetSelectedValue();
    const auto saveDataType = pSelectedItem->GetInfo().type;

    auto addMessages = [ pSelectedItem ]( MessageView* pView )-> void
    {
        // Set up the Title and Message
        pView->AddMessage( "Import Save Data" );

#if !defined ( NN_CUSTOMERSUPPORTTOOL )
        pView->AddMessage( "Import the selected save data?" );
        glv::WideString idString = glv::WideString( GLV_TEXT_API_WIDE_STRING( "[Application ID]\n" ) ) + pSelectedItem->applicationIdWideStr;
#else
        pView->AddMessage( "Import the selected nsave data?" );
        glv::WideString idString = glv::WideString( GLV_TEXT_API_WIDE_STRING( "[nsave FileName]\n" ) ) + pSelectedItem->applicationIdWideStr;
#endif
        pView->AddMessage( idString );
    };

    auto addButtons = [ this ]( MessageView* pView, const std::function< nn::fs::SaveDataSpaceId() >& getTargetDevice )-> void
    {
        pView->AddButton( "Cancel" );

        // Add the button with the Lambda that will import the Save Data
        pView->AddButton(
            "Import",
            [ this, getTargetDevice ]( void* pParam, nn::TimeSpan& timespan )
            {
                const ImportSaveDataPropertyType* pCurrentProperty = m_pListView->GetSelectedValue();
                if ( nullptr == pCurrentProperty )
                {
                    DEVMENU_LOG( "Could not delete Save Data - Selected Value was null.\n" );
                }
                else
                {
                    const auto& info = pCurrentProperty->GetInfo();
#if !defined ( NN_CUSTOMERSUPPORTTOOL )
                    if ( false == CheckParsedSaveDataInfo( info.type, info.cacheIndex ) )
                    {
                        auto pErrorView = new MessageView( false );
                        CreateErrorMessageView( pErrorView, "The cache index is not set in exported cache storage data.\n"
                            "The Specified exported data doesn't support cache storage yet.\n"
                            "If you need to import the data, please see the DevMenu manual and edit the exported xml file." );
                        this->StartModal( pErrorView, true, true );
                    }
                    else
#endif
                    {
                        const auto spaceId = getTargetDevice();
                        const auto result = Import( info, spaceId );
                        if ( result.IsFailure() )
                        {
                            auto pErrorView = new MessageView( false );
                            CreateErrorMessageView( pErrorView, "Failed to restore savedata. Save data size is invalid.\n", result );
                            StartModal( pErrorView, true, true );
                        }
                    }
                }
            },
            nullptr,
            MessageView::ButtonTextColor::Green
        );
    };

    if ( nn::fs::SaveDataType::Cache == saveDataType )
    {
        auto pView = new ConfirmView();
        addMessages( pView );
        addButtons( pView, [ pView ]()-> nn::fs::SaveDataSpaceId{ return pView->GetSelectedValue(); } );
        this->StartModal( pView );
    }
    else if ( nn::fs::SaveDataType::Bcat == saveDataType )
    {
        auto pView = new MessageView( false );

        // 一度エクスポートしたデータ配信キャッシュストレージを本体に書き戻すことは、システムのステータス不整合を招く
        pView->AddMessage( "Unable to restore BCAT save data." );
        pView->AddButton( "Close" );
        this->StartModal( pView );
    }
    else
    {
        auto pView = new MessageView( true );
        addMessages( pView );
        addButtons( pView, []()-> nn::fs::SaveDataSpaceId{ return nn::fs::SaveDataSpaceId::User; } );
        this->StartModal( pView );
    }
}

const nn::Result ImportScene::Import( const SaveDataInfo& info, nn::fs::SaveDataSpaceId spaceId ) NN_NOEXCEPT
{
#if !defined( NN_CUSTOMERSUPPORTTOOL )

    m_TargetSaveDataInfo = info;

    if ( 0x0ULL == info.dataSize && 0x0ULL == info.availableSize )
    {
        return ResultApplicationError();
    }

    if ( nn::fs::SaveDataType::Account == info.type )
    {
        // アカウントセレクタを表示する
        auto pView = new UserAccountSelector(
            [ this, spaceId ]( nn::account::Uid uid )
            {
                // セーブデータの書き戻し
                StartImportThread( uid, spaceId );
            }
        );
        this->StartModal( pView, true, true );
    }
    else
    {
        StartImportThread( nn::account::InvalidUid, spaceId );
    }

#else
    std::string path = DataMoveInfo::ExportPathRoot + "/" + info.exportedPath;

    const uint64_t applicationId = std::stoull( info.exportedPath.c_str() + DateStrLength + 1, nullptr, ApplicationIdStrLength );
    if ( !IsApplicationIdValid( applicationId ) )
    {
        return ResultApplicationError();
    }

    // __BackupSaveDataInfo.xml の読み込み
    BackupSaveDataXmlInfo backupSaveDataInfo;
    std::string saveDataStoredPath;
    auto result = ReadBackupSaveDataInfoForNsave( &backupSaveDataInfo, path );
    if ( result.IsFailure() || 0x0ULL == backupSaveDataInfo.applicationId || applicationId != backupSaveDataInfo.applicationId )
    {
        return ResultApplicationError();
    }

    m_TargetSaveDataInfo = {
        GetSaveDataType(backupSaveDataInfo.saveDataTypeString ),
        { backupSaveDataInfo.applicationId },
        { { backupSaveDataInfo.saveDataUserIdHi, backupSaveDataInfo.saveDataUserIdLow } },
        backupSaveDataInfo.ownerId,
        backupSaveDataInfo.saveDataSize,
        backupSaveDataInfo.saveDataAvailableSize,
        backupSaveDataInfo.saveDataJournalSize,
        backupSaveDataInfo.saveDataFlags,
        backupSaveDataInfo.saveDataSpaceId,
        backupSaveDataInfo.cacheIndex,
        path
    };

    if ( 0x0ULL == m_TargetSaveDataInfo.dataSize && 0x0ULL == m_TargetSaveDataInfo.availableSize )
    {
        return ResultApplicationError();
    }

    // アカウントセレクタを表示する
    const nn::account::Uid uid = { { backupSaveDataInfo.saveDataUserIdHi ,backupSaveDataInfo.saveDataUserIdLow } };
    StartImportThread( uid, spaceId );

#endif
    NN_RESULT_SUCCESS;
}

void ImportScene::StartImportThread( const nn::account::Uid& uid, nn::fs::SaveDataSpaceId spaceId ) NN_NOEXCEPT
{
    StartProgressModalThread(
        "Import save data",
        [ this, uid, spaceId ]( bool* pOutIsSuccess ) -> nn::Result
            {
                NN_UNUSED( pOutIsSuccess );
                return RestoreSaveDataImpl( uid, spaceId );
            },
        [&]( const nn::Result& result, bool isSuccess )
            {
                NN_UNUSED( isSuccess );
                OnCompleteImportProgress( result );
            }
    );
}

void ImportScene::OnCompleteImportProgress( const nn::Result& result ) NN_NOEXCEPT
{
    if ( result.IsSuccess() )
    {
        this->SetModalViewMessage( std::string( "Save data has been restored." ) );
    }
    else
    {
        std::string errorMessage;

#if !defined ( NN_CUSTOMERSUPPORTTOOL )

        if ( nn::fs::ResultUsableSpaceNotEnoughForSaveData::Includes( result ) || nn::fs::ResultUsableSpaceNotEnoughForCacheStorage::Includes( result ) )
        {
            CreateErrorMessage( &errorMessage, std::string( "There is no enouch space to restore the save data." ), result );
        }
        else if ( nn::ns::ResultSdCardNoOwnership::Includes( result ) || nn::ns::ResultSdCardDatabaseCorrupted::Includes( result ) )
        {
            CreateErrorMessage( &errorMessage, std::string( "SD card cannot be used for reading or writing NX contents on the device.\nReboot the device and clean up the SD card." ), result );
        }
        else if ( nn::ns::ResultSdCardNotMounted::Includes( result ) )
        {
            CreateErrorMessage( &errorMessage, std::string( "SD card cannot be used. Device reboot is required." ), result );
        }
        else if ( nn::ns::ResultSdCardNotInserted::Includes( result ) )
        {
            CreateErrorMessage( &errorMessage, std::string( "SD card is not inserted or in unusable state.\nReboot the device and insert the SD card correctly." ), result );
        }
        else if ( nn::ns::ResultSdCardAccessFailed::Includes( result ) )
        {
            CreateErrorMessage( &errorMessage, std::string( "Failed to access the SD card." ), result );
        }
        else
        {
            CreateErrorMessage( &errorMessage, std::string( "Failed to restore the save data." ), result );
        }

#else

        if ( ResultCmacError::Includes( result ) )
        {
            CreateErrorMessage( &errorMessage, std::string( "Failed to restore savedata. cmac is invalid." ), result );
        }
        else if ( ResultSizeOverError::Includes( result ) )
        {
            CreateErrorMessage( &errorMessage, std::string( "Save data beyond 2GB size can not be imported." ), result );
        }
        else
        {
            CreateErrorMessage( &errorMessage, std::string( "Failed to restore save data." ), result );
        }

#endif
        this->SetModalViewMessage( errorMessage );
    }
}

const nn::Result ImportScene::RestoreSaveDataImpl( const nn::account::Uid& uid, nn::fs::SaveDataSpaceId spaceId )  NN_NOEXCEPT
{
    const auto& info = m_TargetSaveDataInfo;

    nn::fs::UserId userId;
    userId._data[ 0 ] = uid._data[ 0 ];
    userId._data[ 1 ] = uid._data[ 1 ];

#if !defined( NN_CUSTOMERSUPPORTTOOL )

    // CacheStorage のリストア時は SD カードをチェックする
    // ToDo: このタイミングだけでは不十分なので要改善。nn::fs::SetEnabledAutoAbort( false ) にしてエラーハンドリングに任せたい
    if ( nn::fs::SaveDataType::Cache == info.type )
    {
        NN_RESULT_DO( nn::ns::CheckSdCardMountStatus() );
    }

    NN_RESULT_DO( DeleteSaveDataInDevice( userId ) );
    NN_RESULT_DO( CreateSaveData( userId, spaceId ) );

#else

    if ( !nn::fs::IsSaveDataExisting( info.applicationId, userId ) )
    {
        return ResultApplicationError();
    }

#endif

    NN_RESULT_DO( MountSaveData( info.type, "save", info.applicationId, userId, info.cacheIndex ) );
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount( "save" );
    };
    NN_RESULT_DO( MountSdCard( "sd" ) );
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount( "sd" );
    };

#if !defined( NN_CUSTOMERSUPPORTTOOL )
    NN_RESULT_DO( CopySaveData( "save:/", info.exportedPath, true, info.journalSize ) );
#else
    NN_RESULT_DO( CopySaveDataDirectoryForNsave( "save:/", info.exportedPath, true, info.journalSize ) );
#endif

    NN_RESULT_DO( nn::fs::CommitSaveData( "save" ) );

    NN_RESULT_SUCCESS;
}

const nn::Result ImportScene::DeleteSaveDataInDevice( const nn::fs::UserId& userId ) NN_NOEXCEPT
{
    NN_RESULT_DO( DeleteSaveDataInDeviceImpl( userId, nn::fs::SaveDataSpaceId::User ) );

    if ( nn::fs::SaveDataType::Cache == m_TargetSaveDataInfo.type )
    {
        NN_RESULT_DO( DeleteSaveDataInDeviceImpl( userId, nn::fs::SaveDataSpaceId::SdUser ) );
    }

    NN_RESULT_SUCCESS;
}

const nn::Result ImportScene::DeleteSaveDataInDeviceImpl( const nn::fs::UserId& userId, nn::fs::SaveDataSpaceId spaceId ) NN_NOEXCEPT
{
    NN_ASSERT( nn::fs::SaveDataSpaceId::User == spaceId || nn::fs::SaveDataSpaceId::SdUser == spaceId );

    std::unique_ptr< nn::fs::SaveDataIterator > iter;
    NN_RESULT_DO( nn::fs::OpenSaveDataIterator( &iter, spaceId ) );

    while ( NN_STATIC_CONDITION( true ) )
    {
        nn::fs::SaveDataInfo info;
        int64_t count = int64_t();

        NN_RESULT_DO( iter->ReadSaveDataInfo( &count, &info, 1 ) );

        if ( 0 == count )
        {
            break;
        }

        // ApplicationId と SaveDataType が一致
        if ( m_TargetSaveDataInfo.applicationId.value == info.applicationId.value && m_TargetSaveDataInfo.type == info.saveDataType )
        {
            if ( nn::fs::SaveDataType::Account == info.saveDataType )
            {
                if ( userId != info.saveDataUserId )
                {
                    continue;
                }
            }
            else if ( nn::fs::SaveDataType::Cache == info.saveDataType )
            {
                if ( m_TargetSaveDataInfo.cacheIndex != info.index )
                {
                    // インデックスが異なるキャッシュストレージは削除対象外
                    continue;
                }
            }
            else
            {
                // 何もしないでスルー
                // System のみ未確認だがスルーでよいはず. DevMenuSystem でのみ考慮すればよいので後回しにする
            }

            const auto result = nn::fs::DeleteSaveData( spaceId, info.saveDataId );
            if ( result.IsFailure() )
            {
                DEVMENU_LOG( "Failed to delete save data info (0x%08x)\n", result.GetInnerValueForDebug() );
            }
            return result;
        }
    }
    NN_RESULT_SUCCESS;
}

const nn::Result ImportScene::CreateSaveData( const nn::fs::UserId& userId, nn::fs::SaveDataSpaceId spaceId ) NN_NOEXCEPT
{
    const auto& info = m_TargetSaveDataInfo;

    // Use save data size from SaveDataInfo.saveDataSize if available size is invalid.
    const auto availableSize = ( 0x0ULL != info.availableSize ) ? info.availableSize : info.dataSize;
    const auto journalSize = ( 0x0ULL != info.availableSize && 0x0ULL != info.journalSize ) ? info.journalSize : info.dataSize;

    switch ( info.type )
    {
    case nn::fs::SaveDataType::System:
        DEVMENU_LOG( "Error: Cannot create System save data.\n" );
        return ResultApplicationError();
    case nn::fs::SaveDataType::Account:
        return nn::fs::CreateSaveData( info.applicationId, userId, info.ownerId, availableSize, journalSize, info.flags );
    case nn::fs::SaveDataType::Bcat:
        return nn::fs::CreateBcatSaveData( info.applicationId, availableSize );
    case nn::fs::SaveDataType::Device:
        return nn::fs::CreateDeviceSaveData( info.applicationId, info.ownerId, availableSize, journalSize, info.flags );
    case nn::fs::SaveDataType::Cache:
        return nn::fs::CreateCacheStorage( info.applicationId, spaceId, info.ownerId, info.cacheIndex, availableSize, journalSize, info.flags );
    case nn::fs::SaveDataType::Temporary:
        DEVMENU_LOG( "Error: Cannot create Temporary save data.\n" );
        return ResultApplicationError();
    default:
        DEVMENU_LOG( "Error: Cannot create save data of unknown type.\n" );
        return ResultApplicationError();
    }
}

void ImportScene::Refresh() NN_NOEXCEPT
{
    // ToDo: 別スレッドで実行するのがベター
    QueryProperties();
}

}} // ~namespace devmenu::savedata, ~namespace devmenu
