﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "DevMenu_SaveDataDetailScene.h"
#include "DevMenu_SaveDataProperty.h"
#include "../DevMenu_Config.h"
#include "../DevMenu_Common.h"
#include "../Applications/DevMenu_ApplicationsCommon.h"

//#define DEBUG_SAVE_DATA

namespace devmenu { namespace savedata {

/* DetailScene::ButtonsTop ---------------------------------------------------------------------------------------
 */

const glv::Rect DetailScene::ButtonsTop::CloseButtonRect( Default::ButtonWidthBack, Default::ButtonHeight );

DetailScene::ButtonsTop::ButtonsTop( const AbstractOperators& op, DetailScene& parent, const glv::Rect& rect ) NN_NOEXCEPT
    : glv::View( glv::Rect( rect.width(), Default::ButtonHeight), glv::Place::TL )
    , m_Op( op )
    , m_Parent( parent )
    , m_Table( "p," "|p,", 30.0f )
    , m_BackButton( "< Back", [&]{ m_Op.CloseSaveDataDetails( m_Parent.m_ParentScene ); }, ButtonsTop::CloseButtonRect, glv::Place::TL )
{
    disable( glv::Property::Controllable | glv::Property::DrawBack | glv::Property::DrawBorder | glv::Property::HitTest );
    const auto pSpacerBack = new Spacer( 1.0f, 10.0f );
    m_Table << pSpacerBack << m_BackButton;
    m_Table.arrange();
    *this << m_Table;
}

void DetailScene::ButtonsTop::Clear() NN_NOEXCEPT
{
    auto& glvRoot = reinterpret_cast< glv::GLV& > ( this->root() );
    NN_ASSERT( !glv::GLV::valid( &glvRoot ) );
    NN_UNUSED( glvRoot ); // For Release build
}

void DetailScene::ButtonsTop::Update() NN_NOEXCEPT
{
}

/* DetailScene::IconButtons ---------------------------------------------------------------------------------------
 */

const glv::Rect DetailScene::IconButtons::ExportButtonRect( Default::ButtonWidthExport, Default::ButtonHeight );
const glv::Rect DetailScene::IconButtons::VerifyButtonRect( Default::ButtonWidthVerify, Default::ButtonHeight );
const glv::Rect DetailScene::IconButtons::DeleteButtonRect( Default::ButtonWidthDelete, Default::ButtonHeight );
const glv::Rect DetailScene::IconButtons::CorruptButtonRect( Default::ButtonWidthCorrupt, Default::ButtonHeight );
const glv::Rect DetailScene::IconButtons::ExportRawButtonRect( Default::ButtonWidthExportRaw, Default::ButtonHeight );

DetailScene::IconButtons::IconButtons( const AbstractOperators& op, DetailScene& parent, const glv::Rect& rect ) NN_NOEXCEPT
    : glv::View( glv::Rect( rect.width() / 3, rect.height() - Default::ButtonHeight ), glv::Place::TL )
    , m_Op( op )
    , m_Parent( parent )
    , m_ExportButton( "Export", [&]{ m_Op.ExportSaveData( m_Parent.m_ParentScene ); }, IconButtons::ExportButtonRect, glv::Place::CL )
    , m_VerifyButton( "Verify", [&]{ m_Op.VerifySaveData( m_Parent.m_ParentScene ); }, IconButtons::VerifyButtonRect, glv::Place::CL )
    , m_DeleteButton( "Delete", [&]{ m_Op.DeleteSaveData( m_Parent.m_ParentScene ); }, IconButtons::DeleteButtonRect, glv::Place::CL )
    , m_CorruptButton( "Corrupt", [&]{ m_Op.CorruptSaveData( m_Parent.m_ParentScene ); }, IconButtons::CorruptButtonRect, glv::Place::CL )
    , m_ExportRawButton( "Export(+Raw)", [&]{ m_Op.ExportRawSaveData( m_Parent.m_ParentScene ); }, IconButtons::ExportRawButtonRect, glv::Place::CL )
    , m_Table( "p|", 50.0f, 10.0f )
    , m_Icon( 256, 256, 0.0f, 15.0f )
{
    NN_UNUSED( parent );

    disable( glv::Property::Controllable | glv::Property::DrawBack | glv::Property::DrawBorder | glv::Property::HitTest );
    m_Table << m_Icon << m_ExportButton << m_VerifyButton << m_DeleteButton;

#if defined ( NN_DEVMENUSYSTEM )
    m_Table << m_ExportRawButton << m_CorruptButton;
#endif

    m_Table.arrange();
    *this << m_Table;
}

void DetailScene::IconButtons::Clear() NN_NOEXCEPT
{
    auto& glvRoot = reinterpret_cast< glv::GLV& >( this->root() );
    NN_ASSERT( !glv::GLV::valid( &glvRoot ) );
    NN_UNUSED( glvRoot ); // For Release build
}

void DetailScene::IconButtons::Update() NN_NOEXCEPT
{
}

void DetailScene::IconButtons::UpdateApplicationIcon( const devmenu::Buffer& iconBuffer ) NN_NOEXCEPT
{
    // Update the icon texture
    m_Icon.SetTexture( iconBuffer.GetPointer(), iconBuffer.GetSize() );
}

void DetailScene::IconButtons::ClearApplicationIcon() NN_NOEXCEPT
{
    m_Icon.ClearTexture();
}

/* DetailScene::Detail ---------------------------------------------------------------------------------------
 */

const glv::Label::Spec DetailScene::Detail::DefalutLabelSpec( glv::Place::TL, 0.0f, 0.0f, CommonValue::InitialFontSize );
const glv::Label::Spec DetailScene::Detail::InfoLabelSpecDefault( glv::Place::TL, 0.0f, 0.0f, Default::FontSize );
const glv::Label::Spec DetailScene::Detail::InfoLabelSpecMinimum( glv::Place::TL, 0.0f, 0.0f, Default::FontSizeMedium );

DetailScene::Detail::Detail( DetailScene& parent, const glv::Rect& rect ) NN_NOEXCEPT
    : glv::View( glv::Rect( rect.width() * 2 / 3, rect.height() - Default::ButtonHeight ), glv::Place::CC )
    , m_Parent( parent )
    , m_ScrollableContainer( "<", glv::Rect( 0.0f, 0.0f, width() - 30.0f, height() - 16.0f ), 0.0f, 0.0f )
    , m_BaseTable( "<", 0.0f, 0.0f )
    , m_SaveDataHeader( "p p", 0.0f, 0.0f )
    , m_SaveDataInfo( "p p", 32.0f, 16.0f )
    , m_ApplicationInfo( "p p", 32.0f, 16.0f )
    , m_LabelApplicationInfoHeader( "Application Information", Detail::DefalutLabelSpec )
    , m_LabelSaveDataInfoHeader( "Save Data Information", Detail::DefalutLabelSpec )
    , m_LabelCorruptedSaveData( " - Corrupted Save Data", Detail::DefalutLabelSpec )
    , m_DividerMetadataInfoHeader( 10.0f, 2.0f )
    , m_DividerApplicationInfoHeader( 10.0f, 2.0f )
{
    NN_UNUSED( m_Parent );

    this->disable( glv::Property::DrawBack | glv::Property::DrawBorder | glv::Property::HitTest );

    // Disable the properties to prevent the tables from receiving input.  If this isn't done, then items in the layout can potentially be moved when the tables are arranged.
    const auto viewProperty = glv::Property::Controllable | glv::Property::HitTest | glv::Property::Focused;

    m_SaveDataHeader.disable( viewProperty );
    m_LabelSaveDataInfoHeader.disable( viewProperty );
    m_DividerMetadataInfoHeader.disable( viewProperty );
    m_SaveDataInfo.disable( viewProperty );
    m_LabelApplicationInfoHeader.disable( viewProperty );
    m_DividerApplicationInfoHeader.disable( viewProperty );
    m_ApplicationInfo.disable( viewProperty );

    // Disable the focused/focusHighlight flags to ensure that the highlight border is not shown, and also hide the background and standard border.
    m_ScrollableContainer.disable( glv::Property::DrawBack | glv::Property::DrawBorder );
    m_ScrollableContainer.enable( glv::Property::HitTest );

    m_pLabelApplicationInfoTitle[ LabelTypesApplicationInfo_ApplicationTitle ] = new glv::Label( "Application Name", Detail::InfoLabelSpecDefault );
    m_pLabelApplicationInfoTitle[ LabelTypesApplicationInfo_ApplicationId ]    = new glv::Label( "Application ID",   Detail::InfoLabelSpecDefault );

    m_pLabelSaveDataInfoTitle[ LabelTypesSaveDataInfo_SaveDataId ]      = new glv::Label( "Save Data ID",   Detail::InfoLabelSpecDefault );
    m_pLabelSaveDataInfoTitle[ LabelTypesSaveDataInfo_SaveDataOwnerId ] = new glv::Label( "Owner ID",       Detail::InfoLabelSpecDefault );
    m_pLabelSaveDataInfoTitle[ LabelTypesSaveDataInfo_SaveDataType ]    = new glv::Label( "Type",           Detail::InfoLabelSpecDefault );
    m_pLabelSaveDataInfoTitle[ LabelTypesSaveDataInfo_AvailableSize ]   = new glv::Label( "Available Size", Detail::InfoLabelSpecDefault );
    m_pLabelSaveDataInfoTitle[ LabelTypesSaveDataInfo_JournalSize ]     = new glv::Label( "Journal Size",   Detail::InfoLabelSpecDefault );

#if defined ( NN_DEVMENUSYSTEM )
    m_pLabelSaveDataInfoTitle[ LabelTypesSaveDataInfo_SystemSaveDataId ] = new glv::Label( "System Save Data ID", Detail::InfoLabelSpecDefault );
    m_pLabelSaveDataInfoTitle[ LabelTypesSaveDataInfo_Flag ]             = new glv::Label( "Flag"               , Detail::InfoLabelSpecDefault );
#endif

    // Header labels & styles
    m_StyleHeader.color.text.set( 0.6f, 0.6f, 0.6f );
    m_LabelSaveDataInfoHeader.style( &m_StyleHeader );
    m_LabelApplicationInfoHeader.style( &m_StyleHeader );
    static devmenu::Spacer s_TableSpacer( 0.0f, 16.0f );

    // Update the Corrupted Save Data warning label
    // Disable warning by default
    m_LabelCorruptedSaveData.disable( glv::Property::Visible );

    // Set the corrupted warning style
    glv::Style* pStyle = new glv::Style();
    pStyle->color.set( glv::StyleColor::BlackOnWhite );
    pStyle->color.text.set( 1.0f, 0.0f, 0.0f );
    m_LabelCorruptedSaveData.style( pStyle );


    // Initialize the labels and strings
    for ( int i = 0; i < LabelTypesApplicationInfo_NumElements; ++i )
    {
        m_pLabelApplicationInfoField[ i ] = new glv::Label( GLV_TEXT_API_WIDE_STRING( "" ), Detail::InfoLabelSpecDefault );
        stringsApplicationInfoField[ i ] = GLV_TEXT_API_WIDE_STRING( "" );

        m_ApplicationInfo
            << m_pLabelApplicationInfoTitle[ i ]
            << m_pLabelApplicationInfoField[ i ];
    }

    for ( int i = 0; i < LabelTypesSaveDataInfo_NumElements; ++i )
    {
        switch ( i )
        {
        case LabelTypesSaveDataInfo_SaveDataOwnerId:
            m_pLabelSaveDataInfoField[ i ] = new glv::Label( GLV_TEXT_API_WIDE_STRING( "" ), Detail::InfoLabelSpecMinimum );
            break;
        default:
            m_pLabelSaveDataInfoField[ i ] = new glv::Label( GLV_TEXT_API_WIDE_STRING( "" ), Detail::InfoLabelSpecDefault );
            break;
        }

        stringsSaveDataInfoField[ i ] = GLV_TEXT_API_WIDE_STRING( "" );

        m_SaveDataInfo
            << m_pLabelSaveDataInfoTitle[ i ]
            << m_pLabelSaveDataInfoField[ i ];
    }

    // Add Save Data info header and corrupted warning label to Save Data header table
    m_SaveDataHeader << m_LabelSaveDataInfoHeader << m_LabelCorruptedSaveData;

    m_BaseTable
        << m_SaveDataHeader
        << m_DividerMetadataInfoHeader
        << m_SaveDataInfo
        << s_TableSpacer
        << m_LabelApplicationInfoHeader
        << m_DividerApplicationInfoHeader
        << m_ApplicationInfo;

    m_ScrollableContainer << m_BaseTable;
    *this << m_ScrollableContainer;
}

DetailScene::Detail::~Detail() NN_NOEXCEPT
{
    Clear();
}

void DetailScene::Detail::Update() NN_NOEXCEPT
{
    Clear();

    SetLabelInfo();

    // Set the Scroll view to the top of list to ensure that we're always at the top when switching between different info pages
    m_ScrollableContainer.ScrollToTop();

    // Arrange the sub-tables (m_AppInfoTable and m_MetaInfoTable) first, then arrange the main table
    m_SaveDataHeader.arrange();
    m_SaveDataInfo.arrange();
    m_ApplicationInfo.arrange();
    m_BaseTable.arrange();

    m_ScrollableContainer.ArrangeTable();
}

void DetailScene::Detail::SetLabelInfo() NN_NOEXCEPT
{
    // Find the Width of the widest title - this is useful in wrapping text and ensuring tables have the same column sizes
    float titleLabelWidth = 0.0f;
    for ( int i = 0; i < LabelTypesApplicationInfo_NumElements; ++i )
    {
        if ( m_pLabelApplicationInfoTitle[ i ]->width() > titleLabelWidth )
        {
            titleLabelWidth = m_pLabelApplicationInfoTitle[ i ]->width();
        }
    }
    for (int i = 0; i < LabelTypesSaveDataInfo_NumElements; ++i)
    {
        if ( m_pLabelSaveDataInfoTitle[i]->width() > titleLabelWidth )
        {
            titleLabelWidth = m_pLabelSaveDataInfoTitle[i]->width();
        }
    }

    // Initialize the Application Info Table
    for ( int i = 0; i < LabelTypesApplicationInfo_NumElements; ++i )
    {
#if !defined( DEBUG_SAVE_DATA )
        m_pLabelApplicationInfoField[ i ]->setValue( stringsApplicationInfoField[ i ] );
#else
        switch ( i )
        {
        case LabelTypesApplicationInfo_ApplicationTitle:
            m_pLabelApplicationInfoField[ LabelTypesApplicationInfo_ApplicationTitle ]->setValue( "DevMenu Dummy Title" );
            break;
        case LabelTypesApplicationInfo_ApplicationID:
            m_pLabelApplicationInfoField[ LabelTypesApplicationInfo_ApplicationID ]->setValue( "0x100000000002065" );
            break;
        default:
            break;
        }
#endif

        switch ( i )
        {
        case LabelTypesApplicationInfo_ApplicationTitle:
            {
                // The maximum width available is calculated from the width of the scroll window, minus the width of the entry title label, minus three times the table padding (32.0f), and then a small padding to separate the
                // text from the scroll bar (8.0f)
                const float maxWidth = m_ScrollableContainer.w - titleLabelWidth - 3 * 32.0f - 8.0f;
                m_pLabelApplicationInfoField[i]->WrapText( maxWidth );
            }
            break;
        default:
            break;
        }

        if ( m_pLabelApplicationInfoTitle[ i ]->w < titleLabelWidth )
        {
            m_pLabelApplicationInfoTitle[ i ]->w = titleLabelWidth;
        }
    }

    // Initialize the Save Data Info Table
    for ( int i = 0; i < LabelTypesSaveDataInfo_NumElements; ++i )
    {
#if !defined( DEBUG_SAVE_DATA )
        m_pLabelSaveDataInfoField[ i ]->setValue( stringsSaveDataInfoField[ i ] );
#else
        switch ( i )
        {
        case LabelTypesSaveDataInfo_SaveDataID:
            m_pLabelSaveDataInfoField[ LabelTypesSaveDataInfo_SaveDataID ]->setValue( "0x1000_0000_0000_000A" );
            break;
        case LabelTypesSaveDataInfo_SaveDataOwnerID:
            m_pLabelSaveDataInfoField[ LabelTypesSaveDataInfo_SaveDataOwnerID ]->setValue( "10000010_8f3eddbd_799086ed_5d35c6be" );
            break;
        case LabelTypesSaveDataInfo_SaveDataType:
            m_pLabelSaveDataInfoField[ LabelTypesSaveDataInfo_SaveDataType ]->setValue( "0x1000_0000_0000_000A" );
            break;
        case LabelTypesSaveDataInfo_AvailableSize:
            m_pLabelSaveDataInfoField[ LabelTypesSaveDataInfo_AvailableSize ]->setValue( "10.4 MB" );
            break;
        case LabelTypesSaveDataInfo_JournalSize:
            m_pLabelSaveDataInfoField[ LabelTypesSaveDataInfo_JournalSize ]->setValue( "5.2 KB" );
            break;
#if defined( NN_DEVMENUSYSTEM )
        case LabelTypesSaveDataInfo_SystemSaveDataId:
            m_pLabelSaveDataInfoField[ LabelTypesSaveDataInfo_SystemSaveDataId ]->setValue( "0x8000_0000_0000_0001" );
            break;
        case LabelTypesSaveDataInfo_Flag:
            m_pLabelSaveDataInfoField[ LabelTypesSaveDataInfo_Flag ]->setValue( "0x0000_0000" );
            break;
#endif
        default:
            break;
        }
#endif

        if ( m_pLabelSaveDataInfoTitle[ i ]->w < titleLabelWidth )
        {
            m_pLabelSaveDataInfoTitle[ i ]->w = titleLabelWidth;
        }
    }
}

void DetailScene::Detail::EnableCorruptWarning( bool isEnabled ) NN_NOEXCEPT
{
    if ( isEnabled )
    {
        m_LabelCorruptedSaveData.enable( glv::Property::Visible );
    }
    else
    {
        m_LabelCorruptedSaveData.disable( glv::Property::Visible );
    }
}

void DetailScene::Detail::Clear() NN_NOEXCEPT
{
    for ( int i = 0; i < LabelTypesApplicationInfo_NumElements; ++i )
    {
        m_pLabelApplicationInfoField[ i ]->setValue( "" );
    }

    for ( int i = 0; i < LabelTypesSaveDataInfo_NumElements; ++i )
    {
        m_pLabelSaveDataInfoField[ i ]->setValue( "" );
    }
}

/* DetailScene ----------------------------------------------------------------------------------------
 */

DetailScene::DetailScene( const AbstractOperators& op, Page* pParentPage, const glv::Rect& rect, const std::function< void() >& backButtonCallback ) NN_NOEXCEPT
    : SaveDataSceneBase( pParentPage, rect, false, backButtonCallback )
    , m_Base( "<-," "<>" )
    , m_Detail( *this, rect )
    , m_ButtonsTop( op, *this, rect )
    , m_IconButtons( op, *this, rect )
{
    this->SetFirstFocusTargetView( GetPrimaryView() );
    *this << m_Base;
    disable( glv::Visible );
}

DetailScene::~DetailScene() NN_NOEXCEPT
{
    DetailScene::Clear();
}

glv::View* DetailScene::GetPrimaryView() NN_NOEXCEPT
{
    return &m_ButtonsTop.GetBackButton();
}

SceneType DetailScene::GetParentScene() NN_NOEXCEPT
{
    return m_ParentScene;
}

void DetailScene::Clear() NN_NOEXCEPT
{
    auto& glvRoot = reinterpret_cast< glv::GLV& >( this->root() );
    NN_ASSERT( glv::GLV::valid( &glvRoot ) );
    glvRoot.setFocus( nullptr );
}

void DetailScene::UpdateDetails( const SaveDataPropertyType* const pSaveDataProperty ) NN_NOEXCEPT
{
    // Set corrupt warning
    m_Detail.EnableCorruptWarning( !pSaveDataProperty->IsValid() );

    // Update SaveDataInfoField information
    glv::CopyWithConvertWideString( m_Detail.stringsSaveDataInfoField[ Detail::LabelTypesSaveDataInfo_SaveDataId ], pSaveDataProperty->saveDataIdStr );
    glv::CopyWithConvertWideString( m_Detail.stringsSaveDataInfoField[ Detail::LabelTypesSaveDataInfo_AvailableSize ], pSaveDataProperty->availableSizeStr );
    glv::CopyWithConvertWideString( m_Detail.stringsSaveDataInfoField[ Detail::LabelTypesSaveDataInfo_JournalSize ], pSaveDataProperty->journalSizeStr );
    glv::CopyWithConvertWideString( m_Detail.stringsSaveDataInfoField[ Detail::LabelTypesSaveDataInfo_SaveDataOwnerId ], pSaveDataProperty->saveDataUserIdStr );
    m_Detail.stringsSaveDataInfoField[ Detail::LabelTypesSaveDataInfo_SaveDataType ] = glv::WideString( pSaveDataProperty->saveDataTypeWideStr );

#if defined( NN_DEVMENUSYSTEM )
    // System Save Data ID & Flag
    glv::CopyWithConvertWideString( m_Detail.stringsSaveDataInfoField[ Detail::LabelTypesSaveDataInfo_SystemSaveDataId ], pSaveDataProperty->systemSaveDataIdStr );
    glv::CopyWithConvertWideString( m_Detail.stringsSaveDataInfoField[ Detail::LabelTypesSaveDataInfo_Flag ], pSaveDataProperty->flagStr );
#endif

    // Update Application ID
    m_Detail.stringsApplicationInfoField[ Detail::LabelTypesApplicationInfo_ApplicationId ] = glv::WideString( pSaveDataProperty->applicationIdWideStr );

    // Other Application Information (if available)
    std::unique_ptr< char[] > buffer;
    size_t dataSize;
    const auto result = application::GetApplicationControlData( &dataSize, &buffer, nn::ns::ApplicationControlSource::Storage, pSaveDataProperty->GetApplicationId() );

    if ( result.IsSuccess() )
    {
        const nn::ns::ApplicationControlDataAccessor accessor( buffer.get(), dataSize );
        const nn::ns::ApplicationControlProperty& property = accessor.GetProperty();
        const nn::ns::ApplicationTitle& title = property.GetDefaultTitle();

        // Title
        glv::CopyWithConvertWideString( m_Detail.stringsApplicationInfoField[ Detail::LabelTypesApplicationInfo_ApplicationTitle ], title.name );

        // Get the icon data size to determine if the icon exists, and if so, load it.  Else, clear the texture.
        const size_t iconDataSize = accessor.GetIconSize();
        if ( iconDataSize > 0 )
        {
            const void* iconData = accessor.GetIconData();

            devmenu::Buffer iconBuffer( iconDataSize );
            memcpy( iconBuffer.GetPointer(), iconData, iconDataSize );
            m_IconButtons.UpdateApplicationIcon( iconBuffer );
        }
        else
        {
            m_IconButtons.ClearApplicationIcon();
        }
    }
    else
    {
        m_Detail.stringsApplicationInfoField[ Detail::LabelTypesApplicationInfo_ApplicationTitle ] = glv::WideString( GLV_TEXT_API_WIDE_STRING( "Unknown" ) );
        m_IconButtons.ClearApplicationIcon();
    }

    Refresh();
}

void DetailScene::Refresh() NN_NOEXCEPT
{
    Clear();
    ClearRefreshRequest();

    m_Detail.Update();
    m_ButtonsTop.Update();
    m_IconButtons.Update();

    m_Base << m_ButtonsTop;
    m_Base << m_IconButtons;
    m_Base << m_Detail;
    m_Base.arrange();

    auto& glvRoot = reinterpret_cast< glv::GLV& >( this->root() );
    NN_ASSERT( glv::GLV::valid( &glvRoot ) );
    glvRoot.setFocus( GetPrimaryView() );
}

}} // ~namespace devmenu::savedata, ~namespace devmenu
