﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs/fs_BcatSaveData.h>
#include <nn/fs/fs_CacheStoragePrivate.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/result/result_HandlingUtility.h>

#if defined ( NN_CUSTOMERSUPPORTTOOL )
    #include "DevMenu_SaveDataCustomerSupportTool.h"
#endif

#include "DevMenu_SaveData.h"
#include "DevMenu_SaveDataConfig.h"

#include "../../BackupSaveData/BackupSaveData.h"

namespace devmenu { namespace savedata {

namespace {

    const std::string GetFilteredName( const std::string& name, bool isImport ) NN_NOEXCEPT
    {
        return ( isImport ? DecodeFromBackupFileName( name ) : EncodeToBackupFileName( name ) );
    }

    const nn::Result CopyFile( const std::string& writePath, const std::string& readPath, bool isJournalingEnabled, int64_t journalSize ) NN_NOEXCEPT
    {
        nn::fs::FileHandle readHandle;
        nn::fs::FileHandle writeHandle;
        int64_t fileSize;

        NN_RESULT_DO( nn::fs::OpenFile( &readHandle, readPath.c_str(), nn::fs::OpenMode_Read ) );
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile( readHandle );
        };

        NN_RESULT_DO( nn::fs::GetFileSize( &fileSize, readHandle ) );

        NN_RESULT_DO( nn::fs::CreateFile( writePath.c_str(), fileSize ) );

        const std::string saveDataStorageName = GetStorageName( writePath );
        if ( isJournalingEnabled )
        {
            NN_RESULT_DO( nn::fs::CommitSaveData( saveDataStorageName.c_str() ) );
        }

        const int64_t maxBufferSize = 1024 * 1024;
        const int64_t writeBufferSize = isJournalingEnabled ? ( journalSize > maxBufferSize ? maxBufferSize : journalSize ) : maxBufferSize;
        std::unique_ptr< char[] > workBuffer( new char[ writeBufferSize ] );
        const int64_t journalSizeToUse = isJournalingEnabled ? journalSize : fileSize;

        for ( int64_t journalingOffset = 0; journalingOffset < fileSize; journalingOffset += journalSizeToUse )
        {
            const int64_t workJounalSize = std::min( fileSize - journalingOffset, journalSizeToUse );

            // ファイル Open から Close の 1 ループスコープ
            {
                NN_RESULT_DO( nn::fs::OpenFile( &writeHandle, writePath.c_str(), nn::fs::OpenMode_Write ) );
                NN_UTIL_SCOPE_EXIT
                {
                    nn::fs::CloseFile( writeHandle );
                };

                for ( int64_t bufferOffset = 0; bufferOffset < workJounalSize; bufferOffset += writeBufferSize )
                {
                    const int64_t offset = journalingOffset + bufferOffset;
                    const int64_t writeSize = std::min( workJounalSize - bufferOffset, writeBufferSize );
                    NN_RESULT_DO( nn::fs::ReadFile( readHandle, offset, workBuffer.get(), static_cast< int >( writeSize ) ) ) ;
                    NN_RESULT_DO( nn::fs::WriteFile( writeHandle, offset, workBuffer.get(), static_cast< int >( writeSize ), nn::fs::WriteOption() ) );
                }
                NN_RESULT_DO( nn::fs::FlushFile( writeHandle ) );
            }
            if ( isJournalingEnabled )
            {
                NN_RESULT_DO( nn::fs::CommitSaveData( saveDataStorageName.c_str() ) );
            }
        }

        if ( isJournalingEnabled )
        {
            NN_RESULT_DO( nn::fs::CommitSaveData( saveDataStorageName.c_str() ) );
        }
        NN_RESULT_SUCCESS;
    }

    const nn::Result CopyFiles( const std::string& destPath, const std::string& srcPath, bool isJournalingEnabled, int64_t journalSize ) NN_NOEXCEPT
    {
        nn::fs::DirectoryHandle directoryHandle;

        NN_RESULT_DO( nn::fs::OpenDirectory( &directoryHandle, srcPath.c_str(), nn::fs::OpenDirectoryMode_File ) );
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseDirectory( directoryHandle );
        };

        while ( NN_STATIC_CONDITION( true ) )
        {
            std::unique_ptr< nn::fs::DirectoryEntry > directoryEntryArea( new nn::fs::DirectoryEntry );
            int64_t readEntryCount = int64_t();

            NN_RESULT_DO( nn::fs::ReadDirectory( &readEntryCount, directoryEntryArea.get(), directoryHandle, 1 ) );

            if ( 0 == readEntryCount )
            {
                break;
            }

            const bool isImport = ( "save" ==  GetStorageName( destPath ) );

            // Import 時は BackupSaveDataInfo ファイルか否かをチェック
            // ToDo: "save", "sd" のマウント名を静的定数で定義する
            if ( isImport )
            {
                if ( IsSaveDataInfoFile( directoryEntryArea->name ) )
                {
                    DEVMENU_LOG( "Copy of %s has been skipped.\n", directoryEntryArea->name );
                    continue;
                }
            }

            const std::string readPath = srcPath + '/' + directoryEntryArea->name;
            const std::string writePath = destPath + '/' + GetFilteredName( directoryEntryArea->name, isImport );

            NN_RESULT_TRY( CopyFile(writePath, readPath, isJournalingEnabled, journalSize ) )
                NN_RESULT_CATCH( nn::fs::ResultAlreadyExists )
                {
                    continue;
                }
            NN_RESULT_END_TRY;
        }
        NN_RESULT_SUCCESS;
    }

    const nn::Result CopyDirectories( const std::string& destPath, const std::string& srcPath, bool isJournalingEnabled, int64_t journalSize ) NN_NOEXCEPT
    {
        nn::fs::DirectoryHandle directoryHandle;

        NN_RESULT_DO( nn::fs::OpenDirectory( &directoryHandle, srcPath.c_str(), nn::fs::OpenDirectoryMode_Directory ) );
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseDirectory( directoryHandle );
        };

        while ( NN_STATIC_CONDITION( true ) )
        {
            std::string workSrcPath, workDestPath;

            // 再帰的にディレクトリをコピーするので unique_ptr のスコープを限定する（DirectoryEntry が 800 バイト弱ある）
            {
                std::unique_ptr< nn::fs::DirectoryEntry> directoryEntryArea( new nn::fs::DirectoryEntry );
                int64_t readEntryCount = int64_t();

                NN_RESULT_DO( nn::fs::ReadDirectory( &readEntryCount, directoryEntryArea.get(), directoryHandle, 1 ) );

                if ( 0 == readEntryCount )
                {
                    break;
                }

                const bool isImport = ( "save" ==  GetStorageName( destPath ) );

                workSrcPath = srcPath + '/' + directoryEntryArea->name;
                workDestPath = destPath + '/' + GetFilteredName( directoryEntryArea->name, isImport );

                NN_RESULT_TRY( nn::fs::CreateDirectory( workDestPath.c_str() ) );
                    NN_RESULT_CATCH( nn::fs::ResultPathAlreadyExists )
                    {
                    }
                NN_RESULT_END_TRY;
            }

            NN_RESULT_DO( CopyFiles( workDestPath, workSrcPath, isJournalingEnabled, journalSize ) );
            NN_RESULT_DO( CopyDirectories( workDestPath, workSrcPath, isJournalingEnabled, journalSize ) );
        }
        NN_RESULT_SUCCESS;
    }

} // end of anonymous namespace

const nn::Result MountSdCard( const char* name ) NN_NOEXCEPT
{
#if !defined ( NN_BUILD_CONFIG_OS_WIN )
    NN_RESULT_DO( nn::fs::MountSdCardForDebug( name ) );
#endif
    NN_RESULT_SUCCESS;
}

bool ExistsFile( const std::string& path ) NN_NOEXCEPT
{
    // mount していることが前提
    nn::fs::DirectoryEntryType dummyEntry;

    return nn::fs::GetEntryType( &dummyEntry, path.c_str() ).IsSuccess();
}

bool IsSaveDataInfoFile( const std::string& fileName ) NN_NOEXCEPT
{
    return DataMoveInfo::ExportedSaveDataInfoFileName == fileName;
}

const nn::Result MountSaveData( nn::fs::SaveDataType saveDataType, const std::string& mountName, nn::ncm::ApplicationId applicationId, nn::fs::UserId userId, int cacheIndex ) NN_NOEXCEPT
{
    switch( saveDataType )
    {
    case nn::fs::SaveDataType::System:
        DEVMENU_LOG( "Error: System save data cannot be mounted.\n" );
        return ResultApplicationError();
    case nn::fs::SaveDataType::Account:
        return nn::fs::MountSaveData( mountName.c_str(), applicationId, userId );
    case nn::fs::SaveDataType::Bcat:
        return nn::fs::MountBcatSaveData( mountName.c_str(), applicationId );
    case nn::fs::SaveDataType::Device:
        return nn::fs::MountDeviceSaveData( mountName.c_str(), applicationId );
    case nn::fs::SaveDataType::Cache:
        return nn::fs::MountCacheStorage( mountName.c_str(), applicationId, cacheIndex );
    case nn::fs::SaveDataType::Temporary:
        DEVMENU_LOG( "Error: Temporary save data cannot be mounted.\n" );
        return ResultApplicationError();
    default:
        DEVMENU_LOG( "Error: Unknown save data type. Cannot be mounted.\n" );
        return ResultApplicationError();
    }
}

const std::string GetStorageName( const std::string& path ) NN_NOEXCEPT
{
    size_t pos = path.find( ":" );
    if ( std::string::npos == pos )
    {
        return "";
    }
    return path.substr( 0, pos );
}

// TORIAEZU: 未使用, 何かに使うこともあるかもしれないのでとりあえず残しておく
const nn::Result GetFsEntries( nn::fs::DirectoryHandle* pOutDirectoryHandle, std::vector< std::unique_ptr< nn::fs::DirectoryEntry > >* pOutDirectoryEntryAreaList, int64_t* outReadCount,
                         const std::string& direcotryPath, nn::fs::OpenDirectoryMode mode ) NN_NOEXCEPT
{
    NN_RESULT_DO( nn::fs::OpenDirectory( pOutDirectoryHandle, direcotryPath.c_str(), mode ) );
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseDirectory( *pOutDirectoryHandle );
    };

    int64_t directoryEntryCount = int64_t();
    int64_t readEntryCount = int64_t();

    NN_RESULT_DO( nn::fs::GetDirectoryEntryCount( &directoryEntryCount, *pOutDirectoryHandle ) );
    pOutDirectoryEntryAreaList->reserve( directoryEntryCount );

    // ToDo: メモリ不足エラーへの対処 ( DirectoryEntry 1 つあたり 800 byte あるので Entry 数がかなり多いとメモリを確保できない )
    auto buffer = new nn::fs::DirectoryEntry[ directoryEntryCount ];

    NN_RESULT_TRY( nn::fs::ReadDirectory( &readEntryCount, buffer, *pOutDirectoryHandle, directoryEntryCount ) )
        NN_RESULT_CATCH_ALL
        {
            if ( nullptr != buffer )
            {
                delete[] buffer;
            }
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY;

    // i > 1 だと、delete 時に NotNMallocPtrError が発生するが未調査
    for ( int64_t i = 0; i < directoryEntryCount; ++i )
    {
        pOutDirectoryEntryAreaList->push_back( std::unique_ptr< nn::fs::DirectoryEntry >( &buffer[ i ] ) );
    }

    *outReadCount = readEntryCount;

    NN_RESULT_SUCCESS;
}

const nn::Result CopySaveData( const std::string& destPath, const std::string& srcPath, bool isJournalingEnabled, int64_t journalSize ) NN_NOEXCEPT
{
    DEVMENU_LOG( "Copied %s to %s\n", srcPath.c_str(), destPath.c_str() );

    if ( isJournalingEnabled )
    {
        if ( journalSize == 0 )
        {
            DEVMENU_LOG( "[Error] To copy savedata, must set destSaveDataJournalSize." );
            return ResultApplicationError();
        }
        if ( journalSize < 0x4000 ) // 16 KB
        {
            DEVMENU_LOG( "[Warning] JournalSize is too small. (size: %llx)", journalSize );
        }
    }

    nn::fs::DirectoryEntryType outValue;
    NN_RESULT_DO( nn::fs::GetEntryType( &outValue, srcPath.c_str() ) );
    if ( outValue == nn::fs::DirectoryEntryType_File )
    {
        // srcPath のファイルをコピー
        NN_RESULT_DO( CopyFile( destPath, srcPath, isJournalingEnabled, journalSize ) );
    }
    else
    {
        // srcPath 直下のファイルコピー
        NN_RESULT_DO( CopyFiles( destPath, srcPath, isJournalingEnabled, journalSize ) );

        // srcPath 直下のディレクトリコピー
        NN_RESULT_DO( CopyDirectories( destPath, srcPath, isJournalingEnabled, journalSize ) );
    }

    NN_RESULT_SUCCESS;
}

const char* GetTypeString( nn::fs::SaveDataType saveDataType, bool isEnumName ) NN_NOEXCEPT
{
    switch ( saveDataType )
    {
    case nn::fs::SaveDataType::System:    return "System";
    case nn::fs::SaveDataType::Account:   return "Account";
    case nn::fs::SaveDataType::Device:    return "Device";
    case nn::fs::SaveDataType::Bcat:      return "Bcat";
    case nn::fs::SaveDataType::Temporary: return "Temporary";
    case nn::fs::SaveDataType::Cache:     return  isEnumName ? "Cache" : "Cache Storage";
    default:                              return "Invalid";
    }
}

const char* GetTypeString( nn::fs::SaveDataType saveDataType, nn::fs::SaveDataSpaceId spaceId, int cacheIndex ) NN_NOEXCEPT
{
    if ( nn::fs::SaveDataType::Cache != saveDataType )
    {
        return GetTypeString( saveDataType );
    }
    else
    {
        std::string spaceString;
        static char s_TypeString[ 48 ];

        switch ( spaceId )
        {
        case nn::fs::SaveDataSpaceId::User:
            spaceString = "System Memory";
            break;
        case nn::fs::SaveDataSpaceId::SdUser:
            spaceString = "SD Card";
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
        nn::util::SNPrintf( s_TypeString, sizeof( s_TypeString ), "%s (index %d, %s)", GetTypeString( saveDataType ), cacheIndex, spaceString.c_str() );
        return s_TypeString;
    }
}

}} // ~namespace devmenu::savedata, ~namespace devmenu

