﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*
 * インクルード
 */
#include <cstring>

#include "QREL.h"

/*
 * 定数定義
 */
enum ConstantValue_GenerateQECode
{
    FormatInfoMask = 0x5412,
    FormatInfoDataLength = 15,
    VersionInfoDataLength = 18,
};
/****************************************************************
 * 外部関数
 ****************************************************************/
/*
 * 関数名 QREL_GenerateQRCode
 * 機能   最終シンボルを完成する
 * 引数   versioninfo(IN)
 *        formatinfo(IN)
 *          ncodes(IN)
 *          matrix(IN/OUT)
 * 戻り値 正常終了時0以外を異常終了時0を返す
 * 説明
 *        型番情報（1）・形式情報（2）を生成し、仮シンボルに追加する（3)。
 *        これで最終シンボルの完成
 *
 */
bool QREL_GenerateQRCode(VERSIONINFO versioninfo, FORMATINFO formatinfo,
                          uint8_t **matrix)
{
    /* 変数宣言 */
    int32_t i = 0, j = 0, k = 0;
    int32_t ncells = 0;
    uint8_t formatinfodata[FormatInfoDataLength + 1];
    uint8_t versioninfodata[VersionInfoDataLength + 1];
    uint8_t formatmask[15]
        = { 1, 0, 1, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0 };

    memset(formatinfodata, 0x00, FormatInfoDataLength + 1);
    memset(versioninfodata, 0x00, VersionInfoDataLength + 1);

    QREL_SetLastError(ERR_NOERROR, true);
    if (!VER_GetnCellNum(versioninfo.version, &ncells)) {
        QREL_SetLastError(ERR_VERSIONINFOERROR, true);
        return false;
    }
    /* (1) 形式情報の生成 */
    switch (formatinfo.ecclevel) {
    case ECCL_L:
        formatinfodata[0] = 0;
        formatinfodata[1] = 1;
        break;

    case ECCL_M:
        formatinfodata[0] = 0;
        formatinfodata[1] = 0;
        break;

    case ECCL_Q:
        formatinfodata[0] = 1;
        formatinfodata[1] = 1;
        break;

    case ECCL_H:
        formatinfodata[0] = 1;
        formatinfodata[1] = 0;
        break;

    default:
        QREL_SetLastError(ERR_ILLEGALPARAM, true);
        return false;
    }
    switch (formatinfo.mp) {
    case Mask000:
        formatinfodata[2] = 0;
        formatinfodata[3] = 0;
        formatinfodata[4] = 0;
        break;

    case Mask001:
        formatinfodata[2] = 0;
        formatinfodata[3] = 0;
        formatinfodata[4] = 1;
        break;

    case Mask010:
        formatinfodata[2] = 0;
        formatinfodata[3] = 1;
        formatinfodata[4] = 0;
        break;

    case Mask011:
        formatinfodata[2] = 0;
        formatinfodata[3] = 1;
        formatinfodata[4] = 1;
        break;

    case Mask100:
        formatinfodata[2] = 1;
        formatinfodata[3] = 0;
        formatinfodata[4] = 0;
        break;

    case Mask101:
        formatinfodata[2] = 1;
        formatinfodata[3] = 0;
        formatinfodata[4] = 1;
        break;

    case Maslk110:
        formatinfodata[2] = 1;
        formatinfodata[3] = 1;
        formatinfodata[4] = 0;
        break;

    case Mask111:
        formatinfodata[2] = 1;
        formatinfodata[3] = 1;
        formatinfodata[4] = 1;
        break;

    default:
        QREL_SetLastError(ERR_ILLEGALPARAM, true);
        return false;
    }
    /* BCHコードを生成する。ここまでは(0 | 1)*/
    encode_bch(15, 5, reinterpret_cast<int8_t *>(formatinfodata + 0),
        reinterpret_cast<int8_t *>(formatinfodata + 5));
    /* マスクをかける
     * (0 | 1)を(0 | 0xFF)に変換する。
     */
    for (i = 0; i < 15; i++)
        formatinfodata[i] = formatinfodata[i] != formatmask[i] ? 0xFF : 0;
     /* (2) 型番情報の生成 */
    if (versioninfo.version >= 7) {
        /* 型番情報は７型以上にしか存在しない */
        int8_t versionbits[6 + 1];
        int32_t checkbit;
        memset(versionbits, 0, 6 + 1);
        checkbit = 0x20;    /* 100000 */
        for (i = 0; i < 6; i++) {
            versioninfodata[i] = versioninfo.version & checkbit ? 1 : 0;
            checkbit >>= 1;
        }
        /* BCHコードを生成する。ここまでは(0 | 1)*/
        encode_bch(18, 6, reinterpret_cast<int8_t *>(versioninfodata + 0),
            reinterpret_cast<int8_t *>(versioninfodata + 6));
        /* (0 | 1)を(0 | 0xFF)に変換する。*/
        for (i = 0; i < 18; i++)
            versioninfodata[i] = versioninfodata[i] == 1 ? 0xff : 0;
    }
    /* (3) 仮シンボルに配置する 形式情報（正） */
    for (i = 0, k = 0; i < 9; i++) {
        if (i != 6) {
            matrix[8][i] = formatinfodata[k];
            k++;
        }
    }
    for (i = 7; i >= 0; i--) {
        if (i != 6) {
            matrix[i][8] = formatinfodata[k];
            k++;
        }
    }
    /* 形式情報（副） */
    for (i = ncells - 1, k = 0; i > ncells - 1 - 7; i--, k++)
        matrix[i][8] = formatinfodata[k];
    matrix[ncells - 1 - 7][8] = (uint8_t)0xFF;
    /* 固定の暗モジュール（セル） */
    for (i = ncells - 1 - 7; i < ncells; i++, k++)
        matrix[8][i] = formatinfodata[k];
     /* 型番情報 */
    if (versioninfo.version >= 7) {
        /* 型番情報は７型以上にしか存在しない 型番情報（正） */
        for (i = 0, k = 17; i < 6; i++)
            for (j = 0; j < 3; j++, k--)
                matrix[i][j + (ncells - 11)] = versioninfodata[k];
             /* 型番情報（副） */
        for (j = 0, k = 17; j < 6; j++)
            for (i = 0; i < 3; i++, k--)
                matrix[i + (ncells - 11)][j] = versioninfodata[k];
    }
    return true;
}// NOLINT(impl/function_size)
