﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <glv_CustomVerticalListView.h>
#include <glv_ScissorBoxView.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_InstallTaskBase.h>
#include <nn/ns/ns_InstallApi.h>
#include <nn/os.h>
#include <nn/os/os_ThreadTypes.h>
#include <nn/util/util_ScopeExit.h>

#include "Common/DevMenu_CommonScene.h"
#include "DevMenu_Config.h"
#include "DevMenu_Common.h"
#include "DevMenu_RootSurface.h"

namespace devmenu { namespace system { namespace program {

// Dev_SystemProgramPage.h との相互参照を防ぐためのクラス参照
class SystemProgramPage;

namespace {
    const int64_t MaxDirectoryNameLength = 300;  //!< SD カード内アプリケーション名の最大文字数（終端文字を含む）

    enum InstallState {
        InstallState_NotStarted,
        InstallState_Running,
        InstallState_Committed,
        InstallState_Completed
    };

#if !defined( DEBUG_DATA )

    template< size_t TemporarySize, size_t OutBufferSize >
    glv::WideCharacterType* BuildUtf16ForInstaller( glv::WideCharacterType( &pOutBuffer )[ OutBufferSize ], const char* pFormat, ... ) NN_NOEXCEPT
    {
        va_list vaList;
        va_start( vaList, pFormat );

        char pTemp[ TemporarySize ];
        std::vsprintf( pTemp, pFormat, vaList );
        nn::util::ConvertStringUtf8ToUtf16Native( reinterpret_cast< uint16_t* >( pOutBuffer ), OutBufferSize, pTemp );

        va_end( vaList );
        return pOutBuffer;
    }

#endif // !defined( DEBUG_DATA )
}

/**
 * @brief SD カード内のアプリケーションプロパティ型です。
 */
struct FilePropertyType
{
    /**
     * @brief SD カード内のアプリケーションリスト用プロパティデータを構築します。
     */
    void Prepare( const nn::fs::DirectoryEntry& dirEntry, const char* pFilePath ) NN_NOEXCEPT
    {
        BuildUtf16ForInstaller< MaxDirectoryNameLength >( name, "%s%s", pFilePath, dirEntry.name );
        auto delimitedSizeStr = devmenu::GetDelimitedNumberString( dirEntry.fileSize );
        BuildUtf16ForInstaller< 32 >( size, "%s", delimitedSizeStr.c_str() );
    }

    glv::WideCharacterType name[ MaxDirectoryNameLength ];
    glv::WideCharacterType size[ 32 ];
};
/**
 * @brief リストビュー
 */
class SdCardListView : public glv::CustomVerticalListView< FilePropertyType >
{
public:

    //----------------------------------------------------------

    typedef glv::CustomVerticalListView< FilePropertyType > ParentType;

    static const glv::space_t HorizontalMargin_Name;  //!< リスト要素中の[ プログラムName ]表示左マージン( ピクセル単位 )
    static const glv::space_t HorizontalLength_Name;  //!< リスト要素中の[ プログラムName ]表示横幅( ピクセル単位 )

    /**
     * @brief コンストラクタです。
     */
    explicit SdCardListView( const glv::Rect& parentClipRegion ) NN_NOEXCEPT
        : ParentType( parentClipRegion )
    {
        SetTouchAndGo( true );
        glv::Style* pStyle = new glv::Style();
        pStyle->color = glv::Style::standard().color;
        pStyle->color.selection.set( 0.1f, 0.85f, 0.2f );
        style( pStyle );
        font().size( 20.f );
    }

protected:
    //----------------------------------------------------------
    /**
     * @copydoc CustomVerticalListView<>::OnQueryBounds( const CustomVerticalListView<>::ItemType&, glv::space_t&, glv::space_t& )
     */
    virtual void OnQueryBounds( const ItemType& item, glv::space_t& outWidth, glv::space_t& outHeight ) NN_NOEXCEPT NN_OVERRIDE
    {
        this->font().getBounds( outWidth, outHeight, item.name );
        outWidth = this->width();
    }

    //----------------------------------------------------------
    /**
     * @copydoc CustomVerticalListView<>::OnDrawItem( const CustomVerticalListView<>::ItemType&, const CustomVerticalListView<>::IndexType, const glv::Rect& )
     */
    virtual void OnDrawItem( const ItemType& item, const IndexType index, const glv::Rect& contentRegion ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED( index );
        glv::Font& font = this->font();

        glv::space_t outWidth, outHeight;

        font.getBounds( outWidth, outHeight, item.size );
        glv::space_t posName = contentRegion.left() + HorizontalMargin_Name;
        font.render( item.name, posName, contentRegion.top() );

        font.getBounds( outWidth, outHeight, item.size );
        const glv::space_t sizeExpect = contentRegion.right() - ( outWidth + ( paddingX() * 2 ) + 4.f );
        const glv::space_t sizeLimit = posName + HorizontalLength_Name + 12.f;
        font.render( item.size, ( sizeExpect < sizeLimit ) ? sizeLimit : sizeExpect, contentRegion.top() );
    }
};

/**
 * @brief インストーラー用のモーダルビュー
 */
class InstallView : public devmenu::MessageView
{
    NN_DISALLOW_COPY( InstallView );
    NN_DISALLOW_MOVE( InstallView );

    static const size_t       InitialActionCapacity; // 実行情報の初期数
    static const glv::space_t InstallViewWidth;
    static const glv::space_t ProgressBarMargineX;
    static const glv::space_t MessageButtonPaddingX;
    static const glv::space_t MessageButtonPaddingY;

public:
    /**
     * @brief       コンストラクタです。
     */
    InstallView() NN_NOEXCEPT;

    /**
     * @brief       デストラクタです。
     */
    virtual ~InstallView() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief      メッセージを追加します。
     *
     * @param[in]   pProgressCaption       状態を表示するためのラベル
     * @param[in]   pInstallingFileName    インストールしているファイル名を表示するためのラベル
     * @param[in]   pProgressLabel         インストール容量を表示するラベル
     */
    InstallView& AddProgressMessage( glv::Label* pProgressCaption, glv::Label* pProgressFileName, glv::Label* pProgressLabel ) NN_NOEXCEPT;

    /**
     * @brief       ボタンを追加します。
     *
     * @param[in]   pButton   ボタン
     */
    InstallView& AddButton( glv::Button* pButton ) NN_NOEXCEPT;

    /**
     * @brief       ボタンを追加します。
     *
     * @param[in]   pButton     ボタン
     * @param[in]   pFunc               ボタン押下時に実行される関数ポインタ
     * @param[in]   pParam              ボタン押下時に実行される関数への引数
     */
    InstallView& AddButton( glv::Button* pButton, ActionFunc pFunc, void* pParam ) NN_NOEXCEPT;

    /**
     * @brief       進捗状況バーを追加します。
     *
     * @param[in]   pProgressBar   進捗状況バーのスライダー
     */
    InstallView& AddProgressBar( glv::Slider* pProgressBar ) NN_NOEXCEPT;

    /**
     * @brief       インストール状態を取得します。
     */
    InstallState GetInstallState() NN_NOEXCEPT { return m_InstallState; };

    /**
     * @brief       インストール状態を設定します。
     *
     * @param[in]   installState    インストール状態
     */
    void SetInstallState( const InstallState installState ) NN_NOEXCEPT { m_InstallState = installState; };

    /**
     * @brief       インストールしているファイル名を設定します。
     *
     * @param[in]   fileName        ファイル名
     */
    void SetProgressFileName( const char* fileName ) NN_NOEXCEPT { m_pProgressFileName->setValue( fileName ); };

    /**
     * @brief       インストール進捗メッセージを設定します。
     *
     * @param[in]   message        表示内容
     */
    void SetProgressLabel( const char* message ) NN_NOEXCEPT { m_pProgressLabel->setValue( message ); };

    /**
     * @brief       直近でインストールしたファイルのサイズを取得します。
     * @return      ファイルのサイズ
     */
    int64_t GetLastNspTotalSize() NN_NOEXCEPT { return m_LastNspTotalSize; };

    /**
     * @brief       直近でインストールしたファイルのサイズを設定します。
     *
     * @param[in]   totalSize       ファイルサイズ
     */
    void SetLastNspTotalSize( const int64_t totalSize ) NN_NOEXCEPT { m_LastNspTotalSize = totalSize; };

    /**
     * @brief       進捗状況バーの進捗度を設定します。
     *
     * @param[in]   value       進捗度( 0.0f - 1.0f )
     */
    void SetProgressBarValue( const double value ) NN_NOEXCEPT { m_pProgressBar->setValue( value ); };

    /**
     * @brief       インストール完了時に表示されるボタンを取得します。
     * @return      ボタン
     */
    glv::Button* GetButton() NN_NOEXCEPT { return m_pButton; };

    /**
     * @brief       インストール状態メッセージを設定します。
     *
     * @param[in]   message        表示内容
     */
    void SetProgressCaption( const char* message ) NN_NOEXCEPT { m_pProgressCaption->setValue( message ); };

protected:
    virtual void Initialize() NN_NOEXCEPT;

    glv::Table                      m_ProgressBarsTable;    //!< スライダー配置テーブル
    glv::Slider*                    m_pProgressBar;         //!< 進捗状況を表示するためのバー
    glv::Label*                     m_pProgressCaption;     //!< 状態を表示するためのラベル
    glv::Label*                     m_pProgressFileName;    //!< インストールしているファイル名を表示するためのラベル
    glv::Label*                     m_pProgressLabel;       //!< インストール容量を表示するラベル
    glv::Button*                    m_pButton;              //!< インストール完了時に表示されるボタン
    InstallState                    m_InstallState;         //!< インストール状態
    int64_t                         m_LastNspTotalSize;     //!< インストール完了表示用の直近でインストールしたファイルサイズ
};


/**
 * @brief アプリケーションインストールシーン
 */
class InstallScene : public Scene
{
public:
    /**
     * @brief コンストラクタです。
     */
    InstallScene( SystemProgramPage* pParentPage, Scene* pParentScene = nullptr ) NN_NOEXCEPT;

    void FinalizeProperties() NN_NOEXCEPT;

    virtual void Refresh() NN_NOEXCEPT;

    nn::Result EnumerateFiles( std::vector< FilePropertyType >& entries, const char* directoryPath ) NN_NOEXCEPT;

    /**
     * @brief SD カードをマウントしファイルリストを作成。
     */
    nn::Result MakeNspList() NN_NOEXCEPT;

    /**
     * @brief       インストール進捗表示を更新します。
     */
    void UpdateInstallProgress() NN_NOEXCEPT;

    /**
     * @brief       表示する文字数に上限を設けて超えた場合は省略した形を生成します。
     *
     * @param[in]   message         表示内容
     * @param[in]   maxLength       表示する文字の上限数
     * @return      整形された文字列
     */
    const char* ClipMessage( const char* message, size_t maxLength ) NN_NOEXCEPT;


private:
    /**
     * @brief SD カード内のアプリケーションプロパティコレクションを登録します。
     * @details 登録内容に応じてリスト or アイテムなしメッセージの切り替えを行います。
     */
    void EntryProperties( const SdCardListView::CollectionType* pNsps ) NN_NOEXCEPT;

    /**
     * @brief ダイアログを表示して確認を取った後、全 nsp のインストールを行います。
     */
    void ConfirmInstall( const FilePropertyType* pProperty = nullptr ) NN_NOEXCEPT;

    /**
     * @brief 選択された nsp ファイルのインストールを行います。
     */
    void InstallNspFile() NN_NOEXCEPT;

    /**
     * @brief SD カード内の全 nsp ファイルのインストールを行います。
     */
    void InstallNspFiles() NN_NOEXCEPT;

    /**
     * @brief SD カードをマウントします。
     * @return      マウントに成功したら true
     */
    bool MountSdCard() NN_NOEXCEPT;

    /**
     * @brief インストールする nsp ファイルをオープンします。
     */
    bool OpenFile( nn::fs::FileHandle& handle, const char* fileName ) NN_NOEXCEPT;

    /**
     * @brief nsp ファイルをインストールします。(Install All 用の TORIAEZU 版, 進捗表示なし）
     */
    void ExecuteInstall( const char* fileName ) NN_NOEXCEPT;

    /**
     * @brief インストールスレッドのメイン関数です。
     */
    static void ThreadFunc( void* args ) NN_NOEXCEPT;

    /**
     * @brief インストール処理を別スレッドで実行します。
     */
    void InstallTaskExecute( nn::ncm::InstallTaskBase* task ) NN_NOEXCEPT;

    /**
     * @brief インストール処理を別スレッドで実行します。(Install All 用の TORIAEZU 版）
     */
    void WaitInstallTaskExecute( nn::ncm::InstallTaskBase* task ) NN_NOEXCEPT;

    /**
     * @brief チェックボックスの全選択。
     */
    void SelectAllCheckboxes() NN_NOEXCEPT;

    Button*                                         m_pButtonAllCheckboxes;
    SdCardListView::CollectionType*                 m_pNsps;
    SdCardListView*                                 m_pNspListView;
    glv::Label*                                     m_pLabelNoNsp;
    Button*                                         m_pButtonToCatalog;
    std::vector< FilePropertyType >                 m_FileList;
    devmenu::Scene*                                 m_pParentScene;
    SystemProgramPage*                              m_pParentPage;

    // TORIAEZU: インストール状況表示関連は全て InstallScene のメンバ変数にしておく
    nn::os::ThreadType                              m_InstallTrigThread;
    nn::os::ThreadType                              m_InstallExecThread;
    nn::ncm::SubmissionPackageInstallTask*          m_pInstallTask;
    nn::fs::FileHandle                              m_Handle;
    bool                                            m_IsInstallAll;
    InstallView*                                    m_pInstallWiew;
    nn::os::Tick                                    m_StartTick;
    InstallView*                                    m_pInstallView;         //!< インストールの進捗表示用のモーダル
};

}}} // ~namespace devmenu::system::program, ~namespace devmenu::system, ~namespace devmenu

