﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "DevMenu_SystemProgram.h"
#include "DevMenu_SystemProgramInstaller.h"

using namespace nn;

namespace devmenu { namespace system { namespace program {

const glv::space_t PropertyListView::HorizontalMargin_ProgramId = 4.f;
const glv::space_t PropertyListView::HorizontalMargin_ProgramVersion = 4.f;
const glv::space_t PropertyListView::HorizontalLength_ProgramId = 332.f;
const glv::space_t PropertyListView::HorizontalLength_ProgramVersion = 240.f;

/**
 * @brief コンストラクタです。
 */
PropertyListView::PropertyListView( const glv::Rect& parentClipRegion ) NN_NOEXCEPT
    : CustomVerticalListView( parentClipRegion )
{
    SetTouchAndGo( false );
    glv::Style* pStyle = new glv::Style();
    pStyle->color = glv::Style::standard().color;
    pStyle->color.selection.set( 0.1f, 0.85f, 0.2f );
    style( pStyle );
    font().size( 25.f );

    // 親クラスで OKボタン( A )に対してUpdate::Actionイベントが発生するので Hack.
    changePadClickDetectableButtons( glv::BasicPadEventType::Button::X::Mask );
    changePadClickDetectableButtons( glv::DebugPadEventType::Button::X::Mask );
}

/**
 * @copydoc CustomVerticalListView<>::OnQueryBounds( const CustomVerticalListView<>::ItemType&, glv::space_t&, glv::space_t& )
 */
void PropertyListView::OnQueryBounds( const ItemType& item, glv::space_t& outWidth, glv::space_t& outHeight ) NN_NOEXCEPT
{
    this->font().getBounds( outWidth, outHeight, item.id );
    outWidth = this->width();
}

/**
 * @copydoc CustomVerticalListView<>::OnDrawItem( const CustomVerticalListView<>::ItemType&, const CustomVerticalListView<>::IndexType, const glv::Rect& )
 */
void PropertyListView::OnDrawItem( const ItemType& item, const IndexType index, const glv::Rect& contentRegion ) NN_NOEXCEPT
{
    NN_UNUSED( index );
    glv::Font& font = this->font();

    glv::space_t outWidth, outHeight;
    font.getBounds( outWidth, outHeight, item.version );
    glv::space_t diffVersion = HorizontalLength_ProgramVersion - outWidth;

    glv::space_t posId = contentRegion.left() + HorizontalMargin_ProgramId;
    glv::space_t posVersion = posId + HorizontalLength_ProgramId + HorizontalMargin_ProgramVersion + diffVersion;
    font.render( item.id, posId, contentRegion.top() );
    font.render( item.version, posVersion, contentRegion.top() );

    font.getBounds( outWidth, outHeight, item.name );
    const glv::space_t nameExpect = contentRegion.right() - ( outWidth + ( paddingX() * 2 ) + 4.f );
    const glv::space_t nameLimit = posId + HorizontalLength_ProgramId + HorizontalMargin_ProgramVersion + HorizontalLength_ProgramVersion + 12.f;
    font.render(item.name, ( nameExpect < nameLimit ) ? nameLimit : nameExpect, contentRegion.top() );
}

/**
 * @brief コンストラクタ
 */
CatalogScene::CatalogScene( SystemProgramPage* pParentPage, const glv::Rect& rect, InstallScene* pChildScene ) NN_NOEXCEPT
    : Scene( pParentPage, rect, true ),
    m_pItems( nullptr ), m_pItemListView( nullptr ), m_pSwitchButton( nullptr ) , m_pLabelNoItem( nullptr ),
    m_pChildScene( pChildScene ), m_pParentPage( pParentPage )
{
    // シザーボックスの領域設定
    static const glv::space_t CatalogSceneHeaderRegion = 48.f;
    static const glv::space_t CatalogSceneFooterRegion = 96.f;
    glv::ScissorBoxView* pListContainer = new glv::ScissorBoxView( 0, CatalogSceneHeaderRegion, width(), height() - ( CatalogSceneHeaderRegion + CatalogSceneFooterRegion ) );

    // リスト要素のパディング ( ボーダーを表示する場合は 2 以上必要 )
    static const glv::space_t ListMarginL = 16.f;
    static const glv::space_t ListMarginR = 16.f;
    static const glv::space_t ListMarginY = 2.f;

    // 高さのオフセット
    glv::space_t heightOffset = 0.f;
    heightOffset += 32.f;

    // ヘッダ
    auto pHeader = new glv::Group( glv::Rect( width() , CatalogSceneHeaderRegion ) );
    {
        auto pHeaderId = new glv::Label( "Program ID", glv::Label::Spec( glv::Place::TL, ListMarginL + 8, 8, CommonValue::InitialFontSize ) );
        auto posVersion = PropertyListView::HorizontalMargin_ProgramId + PropertyListView::HorizontalLength_ProgramId + PropertyListView::HorizontalMargin_ProgramVersion;
        auto pHeaderVersion = new glv::Label( "Version", glv::Label::Spec( glv::Place::TL, posVersion, 8, CommonValue::InitialFontSize ) );
        auto pHeaderName = new glv::Label( "Name", glv::Label::Spec( glv::Place::TR, -( ListMarginL + 16 ), 8, CommonValue::InitialFontSize ) );

        *pHeader << pHeaderId << pHeaderVersion << pHeaderName;
        *this << pHeader;
    }

    // リストビュー
    {
        glv::Rect clipRegion( ListMarginL, ListMarginY, pListContainer->width() - ( ListMarginL + ListMarginR ), pListContainer->height() - ( ListMarginY * 2 ) );
        PropertyListView* pListView = new PropertyListView( clipRegion );
        pListView->attach( OnPropertyListUpdateNotification, glv::Update::Action, this );
        *pListContainer << pListView;
        m_pItemListView = pListView;
    }

    // アイテムなしメッセージ
    auto pLabelNoItem = new glv::Label( "No system programs are installed.", glv::Label::Spec( glv::Place::CC, 0.f, 0.f, 32.f ) );
    {
        *pListContainer << pLabelNoItem;
        *this << pListContainer;
        m_pLabelNoItem = pLabelNoItem;
    }

    // フッタ
    auto pFooterExplaination = new glv::Label( "X: Delete the selected system program",
        glv::Label::Spec( glv::Place::BR, -( ListMarginL + 16 ), -8, CommonValue::InitialFontSize ) );
    *this << pFooterExplaination;

    // 全システムプログラム削除ボタン
    auto button = new glv::Button( glv::Rect( ListMarginL, height() - CatalogSceneFooterRegion, width() - ListMarginL - ListMarginR, 48 ), true );
    button->attach( []( const glv::Notification& notification )->void { notification.receiver< CatalogScene >()->OnDeleteAllUnknown(); }, glv::Update::Clicked, this );
    auto deleteAllLabel = new glv::Label( "Delete all unknown", glv::Label::Spec( glv::Place::CC, 0, 0, CommonValue::InitialFontSize ) );
    *button << deleteAllLabel;
    *this << button;

    m_pSwitchButton = new Button( "Install from SD", [&]{
        if ( true == m_pChildScene->MakeNspList().IsFailure() ) { return; }
        m_pParentPage->SwitchScene( this, m_pChildScene, false ); }
        , CommonValue::InitialFontSize, 16.f );
    m_pSwitchButton->anchor( glv::Place::BL ).pos( glv::Place::TL, ListMarginL, -CatalogSceneFooterRegion + 4.f + button->height() );
    *this << m_pSwitchButton;

    NN_ABORT_UNLESS_RESULT_SUCCESS( m_NameMap.Initialize() );

    m_QueryThread.Initialize();
    nn::ns::GetApplicationRecordUpdateSystemEvent( &m_systemEvent );
}

void CatalogScene::OnLoopBeforeSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT
{
    NN_UNUSED( context );

    // ユーザーアクションのコールバック関数を決定
    {
        bool deleteTriggered = false;
        const glv::DebugPadEventType& dpad = events.GetDebugPad();
        if ( true == dpad.HasAnyEvent() )
        {
            deleteTriggered = dpad.IsButtonUp< nn::hid::DebugPadButton::X >();
        }
        else if ( events.GetAvailableBasicPadCount() > 0 )
        {
            const glv::BasicPadEventType& bpad = events.GetBasicPad( 0 );
            deleteTriggered = bpad.IsButtonUp< glv::BasicPadEventType::Button::X >();
        }

        if ( true == deleteTriggered )
        {
            m_ListUpdateCallback = [&]( const glv::Notification& notification ){
                NN_UNUSED( notification );
                OnDelete();
            };
        }
    }
    // ApplicationRecord の更新チェック
    if ( true == visible() && m_systemEvent.TryWait() )
    {
        m_systemEvent.Clear();
        QueryProperties();
    }
}

/**
 * @brief リスト選択結果の受信。
 */
void CatalogScene::OnPropertyListUpdateNotification( const glv::Notification& notification ) NN_NOEXCEPT
{
    notification.receiver< CatalogScene >()->m_ListUpdateCallback( notification );
}

void CatalogScene::OnDeleteAllUnknown() NN_NOEXCEPT
{
    auto pView = new MessageView();

    // 表示メッセージ追加
    pView->AddMessage( "Delete All Unknown System Program" );
    pView->AddMessage( "Are you sure you want to delete all unknown system program?" );

    pView->AddButton( "Cancel" );

    // 実行ボタン追加
    pView->AddButton(
        "Delete",
        [&]( void* pParam, nn::TimeSpan& timespan )
        {
            ncm::ContentMetaDatabase db;
            ncm::ContentStorage storage;
            NN_ABORT_UNLESS_RESULT_SUCCESS(ncm::OpenContentMetaDatabase( &db, ncm::StorageId::BuildInSystem ) );
            NN_ABORT_UNLESS_RESULT_SUCCESS(ncm::OpenContentStorage( &storage, ncm::StorageId::BuildInSystem ) );
            ncm::ContentManagerAccessor accessor( &db, &storage );

            std::vector< ncm::ContentMetaKey > list;
            list.resize( 1024 );
            auto listCount = db.ListContentMeta( list.data(), static_cast< int >( list.size() ), ncm::ContentMetaType::Unknown );
            list.resize( static_cast< size_t >( listCount.listed ) );
            for ( auto& key : list )
            {
                if ( !m_NameMap.HasName( key.id ) )
                {
                    accessor.DeleteAll( key.id );
                }
            }
            QueryProperties();
        },
        nullptr,
        MessageView::ButtonTextColor::Red
    );

    m_pParentPage->GetRootSurfaceContext()->StartModal( pView, true );
}

void CatalogScene::OnDelete() NN_NOEXCEPT
{
    auto pView = new MessageView();

    // 表示メッセージ追加
    pView->AddMessage( "Delete System Program" );
    pView->AddMessage( "Are you sure you want to delete this system program?" );

    pView->AddButton( "Cancel" );

    // 実行ボタン追加
    pView->AddButton(
        "Delete",
        [&]( void* pParam, nn::TimeSpan& timespan )
        {
            const SystemProgramPropertyType* pCurrentProperty = m_pItemListView->GetSelectedValue();
            if ( nullptr == pCurrentProperty )
            {
                DEVMENU_LOG( "Could not delete Save Data - Selected Value was null.\n" );
            }
            else
            {
                ncm::ContentMetaDatabase db;
                ncm::ContentStorage storage;
                NN_ABORT_UNLESS_RESULT_SUCCESS( ncm::OpenContentMetaDatabase( &db, ncm::StorageId::BuildInSystem ) );
                NN_ABORT_UNLESS_RESULT_SUCCESS( ncm::OpenContentStorage( &storage, ncm::StorageId::BuildInSystem ) );
                ncm::ContentManagerAccessor accessor( &db, &storage );

                accessor.DeleteAll( pCurrentProperty->meta.id );
                QueryProperties();
            }
        },
        nullptr,
        MessageView::ButtonTextColor::Red
    );

    m_pParentPage->GetRootSurfaceContext()->StartModal( pView, true );
}

void CatalogScene::QueryProperties() NN_NOEXCEPT
{
    FinalizeProperties();
    PropertyListView::CollectionType* pItems;
    m_pItems = pItems = new PropertyListView::CollectionType();

    NN_ABORT_UNLESS_RESULT_SUCCESS( m_NameMap.Initialize() );

    ncm::ContentMetaDatabase db;
    NN_ABORT_UNLESS_RESULT_SUCCESS( ncm::OpenContentMetaDatabase( &db, ncm::StorageId::BuildInSystem ) );

    std::vector< ncm::ContentMetaKey > list;
    list.resize( MaxShowListCount );
    auto listCount = db.ListContentMeta( list.data(), static_cast< int >( list.size() ), ncm::ContentMetaType::Unknown );
    list.resize( static_cast< size_t >( listCount.listed ) );

    pItems->resize( list.size() );
    for ( size_t i = 0; i < list.size(); i++ )
    {
        pItems->at( i ).Prepare( list[i], m_NameMap.GetName( list[i].id ) );
        pItems->at(i).isQueried = false;
    }

    // プロパティリスト登録
    EntryProperties( pItems );

    // スレッド経由で問い合わせ開始
    m_QueryThread.Query();
}

/**
 * @brief プロパティコレクションを登録します。
 * @details 登録内容に応じてリスト or アイテムなしメッセージの切り替えを行います。
 */
void CatalogScene::EntryProperties( const PropertyListView::CollectionType* pItems ) NN_NOEXCEPT
{
    const glv::Property::t focusableProperty = glv::Property::t::Controllable | glv::Property::t::HitTest;
    PropertyListView* pListView;
    if ( nullptr != ( pListView = m_pItemListView ) )
    {
        pListView->EntryCollection( *pItems );
        if ( true == pItems->empty() )
        {
            pListView->disable( focusableProperty );
            m_pLabelNoItem->enable( glv::Property::t::Visible );
            // Focus 可能な View を設定する
            if ( true == visible() )
            {
                m_pParentPage->GetRootSurfaceContext()->setFocus( m_pSwitchButton );
            }
        }
        else
        {
            pListView->enable( focusableProperty );
            m_pLabelNoItem->disable( glv::Property::t::Visible );
        }
    }
}

void CatalogScene::FinalizeProperties() NN_NOEXCEPT
{
    m_QueryThread.CancelQuery();

    PropertyListView::CollectionType* pItems;
    if ( nullptr != ( pItems = m_pItems ) )
    {
        m_pItems = nullptr;
        delete pItems;
    }
}

/**
 * @brief シーンにフォーカスが与えられた際に、フォーカスを横取る子供を指定します。
 */
glv::View* CatalogScene::GetFocusableChild() NN_NOEXCEPT
{
    PropertyListView::CollectionType* pItems;
    return ( nullptr == ( pItems = m_pItems ) || pItems->empty() )
        ? static_cast< glv::View* >( m_pLabelNoItem )
        : static_cast< glv::View* >( m_pItemListView );
}

void CatalogScene::Refresh() NN_NOEXCEPT
{
}

SystemProgramPage::SystemProgramPage( int pageId, const glv::WideCharacterType* pageCaption, glv::Rect rect ) NN_NOEXCEPT
    : Page( pageId, pageCaption, rect ),
    m_pCatalogScene( nullptr ),
    m_pInstallScene( nullptr )
{
}

void SystemProgramPage::OnAttachedPage() NN_NOEXCEPT
{
    // カタログシーン
    m_pCatalogScene = new CatalogScene( this, rect() );
    {
        m_pCatalogScene->anchor( glv::Place::TL ).pos( glv::Place::TL, 0.f, 0.f );
        *this << m_pCatalogScene;
    }

    // インストールシーン
    m_pInstallScene = new InstallScene( this, m_pCatalogScene );
    {
        m_pInstallScene->anchor( glv::Place::TL ).pos( glv::Place::TL, 0.f, 0.f );
        m_pInstallScene->disable( glv::Property::Visible );
        m_pInstallScene->SetBackButtonCallback( [&] {
            this->QueryProperties();
            this->SwitchScene( m_pInstallScene, m_pCatalogScene, true ); } );
        *this << m_pInstallScene;
    }

    m_pCatalogScene->SetChildScene( m_pInstallScene );
}

void SystemProgramPage::OnDetachedPage() NN_NOEXCEPT
{
    FinalizeProperties();
    m_pCatalogScene->FinalizeQueryThread();
}

void SystemProgramPage::OnActivatePage() NN_NOEXCEPT
{
    QueryProperties();
    GetRootSurfaceContext()->MoveFocusToMenuTabs();
}

void SystemProgramPage::OnDeactivatePage() NN_NOEXCEPT
{
    FinalizeProperties();
}

void SystemProgramPage::OnChangeIntoBackground() NN_NOEXCEPT
{
    FinalizeProperties();
}

void SystemProgramPage::OnChangeIntoForeground() NN_NOEXCEPT
{
    m_pCatalogScene->QueryProperties();
}

void SystemProgramPage::OnLoopBeforeSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT
{
    NN_UNUSED( context );
    NN_UNUSED( events );

    m_pCatalogScene->OnLoopBeforeSceneRenderer( context, events );
    m_pInstallScene->UpdateInstallProgress();

    // ページが変わると呼ばれなくなる。
    m_pCatalogScene->ObserveReceiveProperty();
}

void SystemProgramPage::OnLoopAfterSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT
{
    NN_UNUSED( context );
    NN_UNUSED( events );
}

glv::View* SystemProgramPage::GetFocusableChild() NN_NOEXCEPT
{
    return m_pCatalogScene->GetFocusableChild();
}

void SystemProgramPage::QueryProperties() NN_NOEXCEPT
{
    m_pCatalogScene->QueryProperties();
}

void SystemProgramPage::FinalizeProperties() NN_NOEXCEPT
{
    m_pCatalogScene->FinalizeProperties();
    //    m_pInstallScene->Finalize(); // ToDO: 現状だとここで実行すると操作手順によっては Abort する
}

/**
 * @brief シーンの切替え。
 */
void SystemProgramPage::SwitchScene( devmenu::Scene* current, devmenu::Scene* next, bool isBack ) NN_NOEXCEPT
{
    current->SetLastFocusedView( lastFocusedView() );
    current->disable( glv::Property::Visible );
    next->enable( glv::Property::Visible );

    if ( true == isBack )
    {
        GetRootSurfaceContext()->setFocus( next->GetLastFocusedView() );
    }
    else
    {
        GetRootSurfaceContext()->setFocus( next->GetFirstFocusTargetView() );
    }
    return;
}

/**
 * @brief ページ生成 ( 専用クリエイター )
 */
template< size_t ID >
class SystemProgramPageCreator : devmenu::PageCreatorBase
{
public:
    /**
     * @brief コンストラクタです。
     */
    explicit SystemProgramPageCreator( const char* pPageName ) NN_NOEXCEPT
        : devmenu::PageCreatorBase( ID, pPageName ) {}

protected:

    /**
     * @brief ページインスタンスを生成します。
     */
    virtual glv::PageBase* newInstance() NN_NOEXCEPT NN_OVERRIDE
    {
        int resolution[ 2 ];
        const auto& display = glv::ApplicationFrameworkGetRuntimeContext().GetDisplay();
        display.GetResolution( resolution[ 0 ], resolution[ 1 ] );
        const auto width = static_cast< glv::space_t >( resolution[ 0 ] );
        const auto height = static_cast< glv::space_t >( resolution[ 1 ] );
        //const glv::Rect pageBounds( width - ( ( 12.0f ) * 2.0f ), height - 118.0f );    // 横は 8 + 4 マージン
        const glv::Rect pageBounds( width - 218.f, height - 118.0f );
        return new SystemProgramPage( ID, GLV_TEXT_API_WIDE_STRING( "Sys Program" ), pageBounds );
    }
};

/**
 * @brief Declearation for the statical instance of page creator.
 */
#define LOCAL_PAGE_CREATOR( _id, _name ) SystemProgramPageCreator< _id > g_SystemProgramPageCreator##_id( _name );
LOCAL_PAGE_CREATOR( DevMenuPageId_SystemProgram, "Sys Program" );

}}} // ~namespace devmenu::system::program, ~namespace devmenu::system, ~namespace devmenu
