﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/ncm/ncm_StorageId.h>
#include <nn/ns/ns_SdCardApi.h>
#include <nn/settings/system/settings_SystemApplication.h>

#include "DevMenu_Notification.h"
#include "DevMenu_RetailInteractiveDisplay.h"
#include "DevMenu_RootSurface.h"
#include "Launcher/DevMenu_Launcher.h"

#include "DevMenuCommand_Common.h"

namespace devmenu { namespace rid {

namespace {
#if !defined( NN_BUILD_CONFIG_OS_WIN )
        const nn::ncm::ApplicationId RetailInteractiveDisplayMenuApplicationId = { PROGRAM_ID_OF_RIDMENU };
#else
        const nn::ncm::ApplicationId RetailInteractiveDisplayMenuApplicationId = { 0x0ULL };
#endif

}

bool IsRetailInteractiveDisplay() NN_NOEXCEPT
{
    static nn::util::optional< bool > s_IsRetailInteractiveDisplay;
    static nn::os::Mutex s_Mutex( false );

    std::lock_guard< nn::os::Mutex > lock( s_Mutex );

    if ( !s_IsRetailInteractiveDisplay )
    {
        s_IsRetailInteractiveDisplay = nn::settings::system::GetQuestFlag();
    }
    return *s_IsRetailInteractiveDisplay;
}

bool IsRetailInteractiveDisplayMenu( nn::ncm::ApplicationId applicationId ) NN_NOEXCEPT
{
    return RetailInteractiveDisplayMenuApplicationId == applicationId;
}

bool IsRetailInteractiveDisplayMenuInstalled() NN_NOEXCEPT
{
    // CustomerSupportTool が DevMenuCommand_Common.cpp を参照すると別のビルドエラーが発生するので、費用対効果で仕方なく実装分岐を選択
#if !defined( NN_CUSTOMERSUPPORTTOOL )
    const auto retailInteractiveDisplayMenuProgramId = static_cast< nn::ncm::ProgramId >( RetailInteractiveDisplayMenuApplicationId );

    if ( !devmenuUtil::IsProgramInstalled( retailInteractiveDisplayMenuProgramId, nn::ncm::StorageId::BuiltInUser ) )
    {
        // SD カードにインストールされているのは想定外だが、アプリケーション開発者の手元ではインストールされ得る
        if ( nn::ns::CheckSdCardMountStatus().IsSuccess() )
        {
            return devmenuUtil::IsProgramInstalled( retailInteractiveDisplayMenuProgramId, nn::ncm::StorageId::SdCard );
        }
        return false;
    }
#endif
    return true;
}

bool IsRetailInteractiveDisplayMenuRunning( bool isRetailInteractiveDisplayModeRequired ) NN_NOEXCEPT
{
    if ( isRetailInteractiveDisplayModeRequired )
    {
        // Retail Interactive Display であり、Application ID が一致することを確認する
        return IsRetailInteractiveDisplay() && ( RetailInteractiveDisplayMenuApplicationId == launcher::GetActiveApplicationId() );
    }
    else
    {
        // Application ID の一致のみを確認する
        return RetailInteractiveDisplayMenuApplicationId == launcher::GetActiveApplicationId();
    }
}

void LaunchRetailInteractiveDisplayMenu( bool isAutoBoot ) NN_NOEXCEPT
{
    NN_ASSERT( IsRetailInteractiveDisplay() );
    launcher::RequestApplicationLaunchOrResume( rid::RetailInteractiveDisplayMenuApplicationId, isAutoBoot );
}

ApplicationExitMenu::ApplicationExitMenu() NN_NOEXCEPT
    : MessageView( false )
{
    AddMessage( "Quit playing the demo applicaiton?" );

    AddButton( "Cancel", [&]( void* pParam, nn::TimeSpan& timespan )
        {
            RequestToCancelRidExitMenu();
        }
    );

    AddButton( "Quit", [&]( void* pParam, nn::TimeSpan& timespan )
        {
            launcher::SetRidDemoApplicationExitedBySystem( true );
            LaunchRetailInteractiveDisplayMenu( false );
        },
        nullptr,
        MessageView::ButtonTextColor::Red
    );
}

}} // ~namespace devmenu::rid, ~namespace devmenu::

