﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#if !defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )

#include <cstdlib>
#include <glv.h>
#include <mutex>

#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/vi/vi_DisplayEvents.h>

// メッセージ処理
#include <nn/oe.h>
#include <nn/oe/oe_PowerStateControlApi.private.h>

#include "DevMenu_Config.h"
#include "DevMenu_Notification.h"
#include "DevMenu_NotificationMessageNotifier.h"

namespace devmenu {

// アプリモードでは、OE ライブラリを利用してメッセージ通知をコントロールします。

namespace {

    // スレッド用スタック領域
    const int ThreadStackSize = 4 * 1024;
    NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStack[ ThreadStackSize ];

    // メッセージ通知コントローラ
    class NotificationController
    {
        NN_DISALLOW_COPY( NotificationController );
        NN_DISALLOW_MOVE( NotificationController );

    public:
        // コンストラクタ
        NotificationController()  NN_NOEXCEPT
        :   m_IsInitialized( false ),
            m_IsForeground( false ),
            m_IsRequestedToExit( false ),
            m_IsRequestedToSuspend( false ),
            m_IsWaitingForResume( false ),
            m_IsDeviceRebootRequiredAfterApplicationExit( false )
        {
        }

        // デストラクタ
        ~NotificationController()  NN_NOEXCEPT
        {
            Finalize();
        }

        // メッセージコントローラを初期化します。
        void Initialize() NN_NOEXCEPT
        {
            NN_ASSERT( m_IsInitialized == false );
            m_IsInitialized = true;

            // イベントの初期化
            nn::os::InitializeEvent( &m_SuspendEvent, false, nn::os::EventClearMode_AutoClear );
            nn::os::InitializeEvent( &m_ResumeEvent, false, nn::os::EventClearMode_AutoClear );
            nn::os::InitializeEvent( &m_ExitEvent, false, nn::os::EventClearMode_AutoClear );

            // 終了要求のハンドリングを開始
            DEVMENU_LOG( "Invoke nn::oe::EnableHandlingForExitRequest()\n" );
            nn::oe::EnterExitRequestHandlingSection();

            // スレッド作成
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                nn::os::CreateThread(
                    &m_Thread, ThreadFunc, this,
                    g_ThreadStack, ThreadStackSize, nn::os::HighestThreadPriority ) );
            nn::os::SetThreadNamePointer( &m_Thread, "DevMenuNotificationControllerThread" );
            nn::os::StartThread( &m_Thread );
        }

        // メッセージコントローラを終了します。
        void Finalize() NN_NOEXCEPT
        {
            if ( m_IsInitialized == true )
            {
                m_IsInitialized = false;

                // スレッド破棄
                ReadyToExit();
                nn::os::WaitThread( &m_Thread );
                nn::os::DestroyThread( &m_Thread );

                // イベント破棄
                nn::os::FinalizeEvent( &m_SuspendEvent );
                nn::os::FinalizeEvent( &m_ResumeEvent );
                nn::os::FinalizeEvent( &m_ExitEvent );

                m_NotificationMessageNotifier.Finalize();
            }
        }

        // 初回の FG 化まで待機する。
        void WaitForInitialize() NN_NOEXCEPT
        {
            NN_ASSERT( m_IsWaitingForResume == false );
        }

        // 終了要求されているか判定して返す。
        bool IsRequestedToExit() const NN_NOEXCEPT
        {
            return m_IsRequestedToExit;
        }

        // 終了準備完了を通知する。
        void ReadyToExit() NN_NOEXCEPT
        {
            nn::os::SignalEvent( &m_ExitEvent );
        }

        // スリープハンドリングしないので使用されません。
        bool IsRequestedToSleep() const NN_NOEXCEPT
        {
            NN_ABORT( "Cannot sleep handling." );
            return false;
        }

        // スリープハンドリングしないので使用されません。
        void ReadyToSleepAndWaitForAwake() NN_NOEXCEPT
        {
            NN_ABORT( "Cannot sleep handling." );
        }

        // サスペンド要求されているか判定して返す。
        bool IsRequestedToSuspend() const NN_NOEXCEPT
        {
            return m_IsRequestedToSuspend;
        }

        // サスペンド準備完了を通知し、レジュームするまで待機する。
        void ReadyToSuspendAndWaitForResume() NN_NOEXCEPT
        {
            NN_ASSERT( m_IsRequestedToSuspend == true );
            NN_ASSERT( m_IsWaitingForResume == false );

            nn::os::SignalEvent( &m_SuspendEvent );
            m_IsWaitingForResume = true;
            nn::os::WaitEvent( &m_ResumeEvent );
            m_IsWaitingForResume = false;
        }

        // レシーバー管理コンテキスト取得
        NN_FORCEINLINE NotificationMessageNotifier& GetNotificationMessageNotifier() NN_NOEXCEPT
        {
            return m_NotificationMessageNotifier;
        }

    private:
        // 終了を要求する。
        void RequestToExitAndWaitForReady() NN_NOEXCEPT
        {
            m_IsRequestedToExit = true;

            // レジューム待機中は再開させる
            if ( m_IsWaitingForResume == true )
            {
                RequestToResume();
            }
            nn::os::WaitEvent( &m_ExitEvent );
        }

        // サスペンドを要求し、準備完了するまで待機する。
        void RequestToSuspendAndWaitForReady() NN_NOEXCEPT
        {
            NN_ASSERT( m_IsRequestedToSuspend == false );

            m_IsRequestedToSuspend = true;
            nn::os::WaitEvent( &m_SuspendEvent );
            m_IsRequestedToSuspend = false;
        }

        // レジュームを要求する。
        void RequestToResume() NN_NOEXCEPT
        {
            NN_ASSERT( m_IsWaitingForResume == true );

            nn::os::SignalEvent( &m_ResumeEvent );
        }

        // 状態を FG に遷移する。
        void ChangeIntoForeground() NN_NOEXCEPT
        {
            // FG 遷移時に必要な処理を書く
        }

        // 状態を BG に遷移する。
        void ChangeIntoBackground() NN_NOEXCEPT
        {
            // BG 遷移時に必要な処理を書く
        }

        // スレッド関数
        static void ThreadFunc( void* arg ) NN_NOEXCEPT
        {
            DEVMENU_LOG( "NotificationController thred start.\n" );

            // 初期化処理
            auto& controller = *( reinterpret_cast< NotificationController* >( arg ) );

            // メッセージ通知処理
            do
            {
                auto message = nn::oe::PopNotificationMessage();
                switch( message )
                {
                // フォーカス状態変化
                case nn::oe::MessageFocusStateChanged:
                    {
                        DEVMENU_LOG( "Received Message_ChangeIntoBackground\n" );

                        const auto state = nn::oe::GetCurrentFocusState();
                        switch ( state )
                        {
                        // BG ⇒ FG 遷移
                        case nn::oe::FocusState_InFocus:
                            {
                                DEVMENU_LOG( "Changed into in-focus state.\n ");
                                controller.GetNotificationMessageNotifier()
                                    .NotifyExecute( NotificationMessageReceiver::Message_ChangeIntoForeground );
                                controller.RequestToResume();
                            }
                            break;

                        case nn::oe::FocusState_OutOfFocus:
                            {
                                // 必要になった段階で書く
                            }
                            break;

                        // FG ⇒ BG 遷移
                        case nn::oe::FocusState_Background:
                            {
                                DEVMENU_LOG( "Changed into background state.\n ");
                                controller.GetNotificationMessageNotifier()
                                    .NotifyExecute( NotificationMessageReceiver::Message_ChangeIntoBackground );
                                controller.RequestToSuspendAndWaitForReady();
                                controller.ChangeIntoBackground();
                            }
                            break;
                        default: NN_UNEXPECTED_DEFAULT;
                        }
                    }
                    break;

                // 中断状態からの復帰
                case nn::oe::MessageResume:
                    {
                        DEVMENU_LOG( "Received MessageResume\n" );
                    }
                    break;

                // 動作モード変化
                case nn::oe::MessageOperationModeChanged:
                    {
                        DEVMENU_LOG( "Received MessageOperationModeChanged\n" );
                    }
                    break;

                // 性能モード変化
                case nn::oe::MessagePerformanceModeChanged:
                    {
                        DEVMENU_LOG( "Received MessagePerformanceModeChanged\n" );
                    }
                    break;

                // 終了
                case nn::oe::MessageExitRequest:
                    {
                        DEVMENU_LOG( "Received MessageExitRequest\n" );
                        controller.RequestToExitAndWaitForReady();
                        DEVMENU_LOG( "Invoke DisableHandlingForExitRequest" );
                        nn::oe::LeaveExitRequestHandlingSection();
                        DEVMENU_LOG( "NotificationController thred finished.\n" );
                    }
                    return;

                default:
                    {
                        DEVMENU_LOG( "Received unknown message = 0x%08x", message );
                    }
                    break;
                }

            } while( NN_STATIC_CONDITION( true ) );
        } // NOLINT(impl/function_size)

    private:
        bool    m_IsInitialized;
        bool    m_IsForeground;
        bool    m_IsRequestedToExit;
        bool    m_IsRequestedToSuspend;
        bool    m_IsWaitingForResume;
        bool    m_IsDeviceRebootRequiredAfterApplicationExit;

        nn::os::EventType       m_SuspendEvent;     // サスペンド待機イベント
        nn::os::EventType       m_ResumeEvent;      // レジューム待機イベント
        nn::os::EventType       m_ExitEvent;        // 終了待機イベント
        nn::os::ThreadType      m_Thread;           // メッセージ通知処理スレッド

        NotificationMessageNotifier m_NotificationMessageNotifier;
    };

    // 変数宣言(メッセージ通知をコントロールしないのでひとまずコメントアウト)
    // NotificationController g_NotificationController;

} // end of unnamed namespace

// メッセージ通知をコントロールしません。

void InitializeNotification() NN_NOEXCEPT
{
}

void FinalizeNotification() NN_NOEXCEPT
{
}

void WaitForNotificationInitialized() NN_NOEXCEPT
{
}

bool IsRequestedToExit() NN_NOEXCEPT
{
    return false;
}

void ReadyToExit() NN_NOEXCEPT
{
}

bool IsRequestedToSleep() NN_NOEXCEPT
{
    return false;
}

void ReadyToSleepAndWaitForAwake() NN_NOEXCEPT
{
}

bool IsRequestedToSuspend() NN_NOEXCEPT
{
    return false;
}

void ReadyToSuspendAndWaitForResume() NN_NOEXCEPT
{
}

void RequestRebootDevice() NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    nn::oe::RequestToReboot();
#endif
}

bool IsRequestedToStartRidExitMenu() NN_NOEXCEPT
{
    return false;
}

void RequestToCancelRidExitMenu() NN_NOEXCEPT
{
}

bool IsSdCardDetectionStateChanged() NN_NOEXCEPT
{
    return false;
}

bool IsHdcpAuthenticationFailed() NN_NOEXCEPT
{
    return false;
}

bool IsExternalDisplayConnected() NN_NOEXCEPT
{
    nn::vi::Display* pDisplay;
    if( nn::vi::OpenDisplay(&pDisplay, "External").IsSuccess() )
    {
        nn::vi::HotplugState hotplug;
        bool isConnected = nn::vi::GetDisplayHotplugState(&hotplug, pDisplay).IsSuccess() &&
                           hotplug == nn::vi::HotplugState_Connected;
        nn::vi::CloseDisplay(pDisplay);
        return isConnected;
    }

    return false;
}

bool NotificationMessageReceiver::EnableNotificationMessageReceiving( bool permission ) NN_NOEXCEPT
{
    NN_UNUSED( permission );
    return IsLinked();
}

} // ~namespace devmenu

#endif
