﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <glv.h>
#include <glv_resources.h>

/**
 * @brief 充電アイコンユーティリティ。
 * @details デフォルト実装では、psm ライブラリに依存しています。
 */
namespace devmenu {

class BatteryIndicator : public glv::View
{
    NN_DISALLOW_COPY( BatteryIndicator );
    NN_DISALLOW_MOVE( BatteryIndicator );

public:
    /**
     * @brief インジケータアイコンのデザインです。
     */
    typedef enum IconDesign
    {
        IconDesign_Machine,             //!< 本体端末( タブレット )の残量表示用アイコンです。
        IconDesign_PadController,       //!< 充電式パッドコントローラの残量表示用アイコンです。
    } IconDesign;

    /**
     * @brief 充電器の状態です。
     */
    typedef enum BatteryChargerState
    {
        BatteryChargerState_Unconnected,  //!< 充電器が接続されていません。
        BatteryChargerState_EnoughPower,  //!< 充電可能で、本体の動作状況にかかわらず供給電力が十分な充電器が接続されています。
        BatteryChargerState_LowPower,     //!< 充電可能ですが、本体の動作状況によっては供給電力が十分でない充電器が接続されています。
        BatteryChargerState_NotSupported, //!< 充電不可能な充電器が接続されています。
    } BatteryChargerState;

public:
    /**
     * @brief コンストラクタです。
     *
     * @param[in]   iconDesign          インジケータアイコンのデザインを指定します。
     * @param[in]   percentageFontSize  電池残量の数字表示に使う文字サイズを指定します。
     */
    explicit BatteryIndicator( IconDesign iconDesign = IconDesign_Machine, bool percentageDisplay = true, float percentageFontSize = 20.f ) NN_NOEXCEPT;

    /**
     * @brief デストラクタです。
     */
    virtual ~BatteryIndicator() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief GLVの描画コールバックです。
     *
     * @param[in]   glvContext  GLVコンテキストです。
     */
    virtual void onDraw( glv::GLV& glvContext ) NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief 残量値が安全圏内( 20%以上 )の場合のインジケータ色を指定します。
     *
     * @param[in]   red     赤チャンネル
     * @param[in]   green   緑チャンネル
     * @param[in]   blue    青チャンネル
     * @param[in]   alpha   アルファチャンネル
     */
    NN_FORCEINLINE void SetColorSafety( float red, float green, float blue, float alpha = 1.0f ) NN_NOEXCEPT
    {
        m_ColorSafety.set( red, green, blue, alpha );
    }

    /**
     * @brief 残量値が危険域( 20%未満 )の場合のインジケータ色を指定します。
     *
     * @param[in]   red     赤チャンネル
     * @param[in]   green   緑チャンネル
     * @param[in]   blue    青チャンネル
     * @param[in]   alpha   アルファチャンネル
     */
    NN_FORCEINLINE void SetColorCritical( float red, float green, float blue, float alpha = 1.0f ) NN_NOEXCEPT
    {
        m_ColorCritical.set( red, green, blue, alpha );
    }

    /**
     * @brief 任意の座標系での描画を行います。
     */
    void Draw( glv::space_t originLeft, glv::space_t originTop, int percentage, BatteryChargerState chargerState ) NN_NOEXCEPT;

protected:
    /**
     * @brief 電池残量及び充電器状態の問い合わせを行います。
     *
     * @param[out]  outPercentageOnBatteryPower 電池残量を 1～100 で表現した値を返します。
     *
     * @details onDraw( glv::GLV& ) から常に問い合わせされます。
     */
    virtual const BatteryChargerState OnQueryBatteryState( int& outPercentageOnBatteryPower ) NN_NOEXCEPT;

private:
    glv::Drawable*      m_pDrawable;           //!< 電池残量アイコン画像
    glv::space_t        m_Width;               //!< ビュー領域幅( pixel )
    glv::space_t        m_Height;              //!< ビュー領域高さ( pixel )
    glv::Color          m_ColorSafety;         //!< 安全残量時のインジケータカラー
    glv::Color          m_ColorCritical;       //!< 安全残量時のインジケータカラー
    const bool          m_IsDisplayPercentage; //!< バッテリー残量のパーセント表示有無
    const float         m_FontSize;            //!< 電池残量値表示用フォントサイズ
    const IconDesign    m_IconDesign;          //!< アイコンデザイン
};

} // ~namespace devmenu
