﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <mutex>

#include <nn/nn_BitTypes.h>
#include <nn/nn_Result.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Directory.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/ns/ns_ApplicationEntitySystemApi.h>
#include <nn/ns/ns_ApplicationManagerSystemApi.h>
#include <nn/ns/ns_InstallApi.h>
#include <nn/ns/ns_Result.h>
#include <nn/os/os_Mutex.h>

#include "DevMenu_ApplicationInstaller.h"
#include "DevMenu_ApplicationCatalog.h"
#include "DevMenu_Notification.h"
#include "DevMenu_Result.h"
#include "DevMenuCommand_InstallUtil.h"

//#define DEBUG_DATA

namespace devmenu { namespace application {

const glv::space_t SdCardListView::HorizontalMargin_Name = 4.0f;
const glv::space_t SdCardListView::HorizontalLength_Name = 768.0f;

namespace {

    const size_t InstallBufferSize = 1 * 1024 * 1024;
    const size_t MaxDisplayedNspsCountInConfirmView = 5;

    const size_t MaxModalMessageLength    = 55;
    const size_t MaxCatalogFileNameLength = 75;

    /**
     * @brief       ファイル名に指定した拡張子( '.' を含む)が含まれるかをチェックします。
     */
    bool HasExtension( const char* str, const char* extension ) NN_NOEXCEPT
    {
        if ( strlen( str ) <= strlen( extension ) + 1 )
        {
            return false;
        }
        return ( nn::util::string_view( str ).substr( strlen( str ) - strlen( extension ) ) == nn::util::string_view( extension ) );
    }

    void ConvertWideStringToString( char* pOutStr, const glv::WideCharacterType* fileName ) NN_NOEXCEPT
    {
        int fileListLength = 0;
        const uint16_t* pFileList = reinterpret_cast< const uint16_t* >( fileName );
        nn::util::GetLengthOfConvertedStringUtf16NativeToUtf8( &fileListLength, pFileList, MaxDirectoryNameLength );
        nn::util::ConvertStringUtf16NativeToUtf8( pOutStr, fileListLength, pFileList );
    }

} // end of anonymous namespace

/**************************************
 * class ConfirmView
 **************************************/

const RadioButtons< nn::ncm::StorageId >::ButtonInfoList ConfirmView::RadioButtonInfoList = {
    { ConfirmView::TargetInstallLocation_Any, nn::ncm::StorageId::Any, "Auto (SD Card is preferentially selected)" },
    { ConfirmView::TargetInstallLocation_BuiltInUser, nn::ncm::StorageId::BuiltInUser, "System Memory" },
    { ConfirmView::TargetInstallLocation_SdCard, nn::ncm::StorageId::SdCard, "SD Card" },
};

ConfirmView::ConfirmView( bool isRadioButtonRequired ) NN_NOEXCEPT
    : RadioButtonsConfirmView< nn::ncm::StorageId >( RadioButtonInfoList, TargetInstallLocation_Any, true )
{
    Remove();

    // Table
    m_ViewTable << m_MessagesTable;
    if ( isRadioButtonRequired )
    {
        m_ViewTable.arrangement( "<, <, x" );
        AddRadioButtons();
    }
    else
    {
        m_ViewTable.arrangement( "<, x" );
    }

    m_ViewTable << m_ButtonsTable;
    m_ViewTable.arrange();
    *this << m_ViewTable;
}

/**************************************
 * class InstallView
 **************************************/

const size_t       InstallView::InitialActionCapacity = 4;
const glv::space_t InstallView::InstallViewWidth      = 800.0f;
const glv::space_t InstallView::ProgressBarMargineX   = 8.0f;
const glv::space_t InstallView::MessageButtonPaddingX = 32.0f;
const glv::space_t InstallView::MessageButtonPaddingY = 16.0f;
const glv::space_t InstallView::MessageViewPaddingX   = 32.0f;
const glv::space_t InstallView::MessageViewPaddingY   = 32.0f;

const glv::Rect        InstallView::ProgressBarRect( InstallView::InstallViewWidth - InstallView::ProgressBarMargineX - InstallView::ProgressBarMargineX, CommonValue::InitialFontSize );
const glv::Label::Spec InstallView::DefaultLabelSpec( glv::Place::TL, 0.0f, 0.0f, CommonValue::InitialFontSize );

InstallView::InstallView( const ButtonCallback& closeCallback, const ButtonCallback& cancelCallback, const ButtonCallback& cleanupCallback, const ButtonCallback& nextCallback ) NN_NOEXCEPT
    : MessageView( false ),
    m_ProgressBarsTable( "", MessageViewPaddingX, MessageViewPaddingY, glv::Rect( 0.0f ) ),
    m_ProgressBar( InstallView::ProgressBarRect, 0.0f ),
    m_ProgressCaption( "[Install Progress]", InstallView::DefaultLabelSpec ),
    m_ProgressFileName( "", InstallView::DefaultLabelSpec ),
    m_ProgressLabel( "Prepare installing...", InstallView::DefaultLabelSpec ),
    m_CloseButton( "Close", closeCallback, CommonValue::InitialFontSize, 24.0f ),
    m_CancelButton( "Cancel", cancelCallback, CommonValue::InitialFontSize, 24.0f ),
    m_CleanupButton( "Clean Up", cleanupCallback, CommonValue::InitialFontSize, 24.0f ),
    m_NextButton( "Next", nextCallback, CommonValue::InitialFontSize, 24.0f ),
    m_InstallResult( nn::ResultSuccess() )
{
    Initialize();
}

InstallView::~InstallView() NN_NOEXCEPT
{
}

void InstallView::Initialize() NN_NOEXCEPT
{
    enable( glv::Property::DrawBorder | glv::Property::DrawBack );
    disable( glv::Property::FocusHighlight );
    m_Actions.reserve( InstallView::InitialActionCapacity );

    // MessageView で構築されたビューツリーをクリア
    Remove();

    // Table
    m_ViewTable.arrangement( "<,x,x" );
    *this << ( m_ViewTable << m_MessagesTable << m_ProgressBarsTable << m_ButtonsTable );

    // メッセージ
    AddProgressMessage( &m_ProgressCaption, &m_ProgressFileName, &m_ProgressLabel );

    // プログレスバー
    AddProgressBar( &m_ProgressBar );

    const glv::Property::t focusableProperty = glv::Property::t::Controllable | glv::Property::t::HitTest;
    m_ProgressBar.disable( focusableProperty );

    // 下部ボタン
    m_CloseButton.IncreaseLabelSize( InstallView::MessageButtonPaddingX , InstallView::MessageButtonPaddingY );
    m_CancelButton.IncreaseLabelSize( InstallView::MessageButtonPaddingX , InstallView::MessageButtonPaddingY );
    m_CleanupButton.IncreaseLabelSize( InstallView::MessageButtonPaddingX , InstallView::MessageButtonPaddingY );
    m_NextButton.IncreaseLabelSize( InstallView::MessageButtonPaddingX , InstallView::MessageButtonPaddingY );

    // ModalView の表示位置調整のために非表示で追加しておく
    AddButton( &m_CloseButton );
    m_CloseButton.disable( glv::Property::Visible );
}

InstallView& InstallView::AddProgressMessage( glv::Label* pProgressCaption, glv::Label* pProgressFileName, glv::Label* pProgressLabel ) NN_NOEXCEPT
{
    // レイアウト調整
    m_MessagesTable << pProgressCaption;
    m_MessagesTable << pProgressFileName;
    m_MessagesTable << pProgressLabel;
    m_MessagesTable.arrange();
    m_ViewTable.arrange();

    return *this;
}

InstallView& InstallView::AddButton( glv::Button* pButton ) NN_NOEXCEPT
{
    return AddButton( pButton, nullptr, nullptr );
}

InstallView& InstallView::AddButton( glv::Button* pButton, ActionFunc pFunc, void* pParam ) NN_NOEXCEPT
{
    pButton->attach( ReceiveUpdateNotification, glv::Update::Clicked, this );

    // 配置方法指定
    const std::string align = m_ButtonsTable.arrangement() + "x";
    m_ButtonsTable.arrangement(align.c_str());

    // レイアウト調整
    m_ButtonsTable << pButton;
    m_ButtonsTable.arrange();
    m_ViewTable.arrange();
    extent( m_ViewTable.width(), m_ViewTable.height() );

    // 実行情報を登録
    Action action;
    action.pButton = pButton;
    action.pFunc = pFunc;
    action.pParam = pParam;
    m_Actions.push_back( action );

    return *this;
}

InstallView& InstallView::AddProgressBar( glv::Slider* pProgressBar ) NN_NOEXCEPT
{
    // 配置方法指定
    const std::string align = m_ProgressBarsTable.arrangement() + "x";
    m_ProgressBarsTable.arrangement( align.c_str() );

    // レイアウト調整
    m_ProgressBarsTable << pProgressBar;
    m_ProgressBarsTable.arrange();
    m_ViewTable.arrange();
    extent( m_ViewTable.width(), m_ViewTable.height() );

    return *this;
}

void InstallView::SetErrorMessageView( bool isLastTarget ) NN_NOEXCEPT
{
    SetProgressCaption( "[Install Failed]" );

    std::string errorMessage;
    bool requestsCleanup = false;
    bool isSdCardAccessError = false;

    if ( nn::fs::ResultPortSdCardNoDevice::Includes( m_InstallResult ) )
    {
        isSdCardAccessError = true;
        CreateErrorMessage( &errorMessage, "SD card is not inserted. ", m_InstallResult, true );
    }
    else if ( nn::fs::ResultSdCardAccessFailed::Includes( m_InstallResult ) )
    {
        isSdCardAccessError = true;
        CreateErrorMessage( &errorMessage, "Failed to access to the SD card. ", m_InstallResult, true );
    }
    else if ( nn::ncm::ResultContentNotFound::Includes( m_InstallResult ) )
    {
        CreateErrorMessage( &errorMessage, "Installable content is not found. Confirm the nsp file.\n", m_InstallResult, true );
    }
    else if ( nn::ncm::ResultNotEnoughInstallSpace::Includes( m_InstallResult ) )
    {
        CreateErrorMessage( &errorMessage, "There is not enough free space. More than nsp file size is required.\n", m_InstallResult, true );
    }
    else if ( nn::ns::ResultSdCardNoOwnership::Includes( m_InstallResult ) || nn::ns::ResultSdCardDatabaseCorrupted::Includes( m_InstallResult) )
    {
        requestsCleanup = true;
        CreateErrorMessage( &errorMessage,
            "SD card was initialized on another device or database is corrupted.\nCleanup is required to use SD card.", m_InstallResult, true );
    }
    else if ( nn::ns::ResultSdCardNotMounted::Includes( m_InstallResult ) )
    {
        CreateErrorMessage( &errorMessage,
            "SD card was inserted after device boot.\nDevice reboot is required to install contents to the card.\n", m_InstallResult, true );
    }
    else if ( nn::ncm::ResultContentHashNotMatched::Includes( m_InstallResult ) )
    {
        CreateErrorMessage( &errorMessage, "nsp file is corrupted. ", m_InstallResult, false );
    }
    else if ( ResultApplicationAndPatchTypeNotMatched::Includes( m_InstallResult ) )
    {
        CreateErrorMessage( &errorMessage, "The application is not one that is used to create the patch.\n", m_InstallResult, true );
    }
    else
    {
        CreateErrorMessage( &errorMessage, "Failed to install the nsp file. ", m_InstallResult, false );
    }

    SetProgressLabel( errorMessage.c_str() );
    m_ProgressBarsTable.disable( glv::Property::Visible );

    RemoveButtons();

    auto& g = reinterpret_cast< glv::GLV& >( root() );

    // Cancel ボタン
    if ( !isLastTarget )
    {
        AddButton( &m_CancelButton );
    }

    // Cleanup ボタン
    if ( requestsCleanup )
    {
        AddButton( &m_CleanupButton );
    }

    // Next ボタン
    if ( !isLastTarget && !isSdCardAccessError )
    {
        AddButton( &m_NextButton );
        g.setFocus( &m_NextButton );
    }

    // Close ボタン
    if ( isLastTarget )
    {
        AddButton( &m_CloseButton );
        m_CloseButton.enable( glv::Property::Visible );
        g.setFocus( &m_CloseButton );
    }
}

void InstallView::SetCloseButton() NN_NOEXCEPT
{
    RemoveButtons();

    AddButton( &m_CloseButton );
    m_CloseButton.enable( glv::Property::Visible );

    auto& g = reinterpret_cast< glv::GLV& >( root() );
    g.setFocus( &m_CloseButton );
}

void InstallView::RemoveButtons() NN_NOEXCEPT
{
    m_CloseButton.remove();
    m_CancelButton.remove();
    m_CleanupButton.remove();
    m_NextButton.remove();
}

/**************************************
 * class InstallScene
 **************************************/

// シザーボックスの領域設定
const glv::space_t InstallScene::HeaderRegionLength = 80.0f;
const glv::space_t InstallScene::FooterRegionLength = 48.0f;

// リスト要素のパディング ( ボーダーを表示する場合は 2 以上必要 )
const glv::space_t InstallScene::ListMarginLeft = 16.0f;
const glv::space_t InstallScene::ListMarginRight = 16.0f;
const glv::space_t InstallScene::ListMarginVertical = 2.0f;

const glv::Label::Spec InstallScene::LabelSpecNoNsp( glv::Place::CC, 0.0f, 0.0f, 32.0f );

/**
 * @brief コンストラクタです。
 */
InstallScene::InstallScene( ApplicationCatalogPage* pParentPage, Scene* pParentScene ) NN_NOEXCEPT
    : Scene( pParentPage, pParentPage->rect() ),
    m_ButtonAllCheckboxes( "Install All", [&]{ m_IsInstallAll = true; ConfirmInstall(); }, CommonValue::InitialFontSize, 16.0f ),
    m_ButtonToCatalog( "< Back", [&]{ this->OnBackButtonPressed(); }, CommonValue::InitialFontSize, 16.0f ),
    m_ListContainer( glv::Rect( 0.0f, InstallScene::HeaderRegionLength, width(), height() - ( InstallScene::HeaderRegionLength + InstallScene::FooterRegionLength ) ) ),
    m_NspListView( glv::Rect( InstallScene::ListMarginLeft, InstallScene::ListMarginVertical,
    m_ListContainer.width() - ( InstallScene::ListMarginLeft + InstallScene::ListMarginRight ), m_ListContainer.height() - ( InstallScene::ListMarginVertical * 2 ) ) ),
    m_LabelNoNsp( "No nsp is found.", InstallScene::LabelSpecNoNsp ),
    m_pParentScene( pParentScene ),
    m_pParentPage( pParentPage ),
    m_Mutex( true ),
    m_pInstallTask( nullptr ),
    m_IsInstallAll( false ),
    m_StartTick( 0 ),
    m_pInstallView( nullptr ),
    m_InstallState( InstallState_NotStarted ),
    m_IsLastTarget( false ),
    m_IsCancelRequested( false )
{
    // ボタンの登録
    {
        // Catalog に戻るボタン
        m_ButtonToCatalog.anchor( glv::Place::TL ).pos( glv::Place::TL, ListMarginLeft, 4.0f );
        *this << m_ButtonToCatalog;

        // 全 nsp の選択ボタン ( → 現状は選択できる実装になっていないので、Install All にしておく ）
        m_ButtonAllCheckboxes.anchor( glv::Place::TL ).pos( glv::Place::TL, m_ButtonToCatalog.right() + 16.0f, 4.0f );
        *this << m_ButtonAllCheckboxes;
    }

    // ヘッダ
    auto pHeaderName = new glv::Label( "Name", glv::Label::Spec( glv::Place::TL, ListMarginLeft, m_ButtonAllCheckboxes.bottom() + 8.0f, CommonValue::InitialFontSize ) );
    auto pHeaderSize = new glv::Label( "Size(Byte)", glv::Label::Spec( glv::Place::TR, -( ListMarginRight + 16.0f ), pHeaderName->top(), CommonValue::InitialFontSize ) );
    {
        *this << pHeaderName;
        *this << pHeaderSize;
    }

    // リスト
    {
        // 個別インストール時のコールバック関数を登録
        m_NspListView.attach(
            []( const glv::Notification& notification )->void {
                auto pScene = notification.receiver< InstallScene >();
                const FilePropertyType* pProperty = notification.data< SdCardListView::ItemType >();
                if ( nullptr != pScene && nullptr != pProperty )
                {
                    pScene->m_IsInstallAll = false;
                    pScene->ConfirmInstall( pProperty );
                }
            },
            glv::Update::Action,
            this
        );

        m_ListContainer << m_NspListView;

        // NSP ファイルなしメッセージ
        m_ListContainer << m_LabelNoNsp;

        *this << m_ListContainer;
    }

    // フッタ
    auto pFooterExplaination = new glv::Label( "A: Install the selected file", glv::Label::Spec( glv::Place::BR, -( ListMarginLeft + 16.f ), -8.f, CommonValue::InitialFontSize ) );
    *this << pFooterExplaination;

    this->SetFirstFocusTargetView( &m_NspListView );
}

// ToDo: どこで使うのかが決まっていない
void InstallScene::FinalizeProperties() NN_NOEXCEPT
{
}

void InstallScene::Refresh() NN_NOEXCEPT
{

}

const nn::Result InstallScene::EnumerateFiles( std::vector< FilePropertyType >* pOutEntryList, const char* directoryPath ) NN_NOEXCEPT
{
    nn::fs::DirectoryHandle directoryHandle;
    NN_RESULT_DO( nn::fs::OpenDirectory( &directoryHandle, directoryPath, nn::fs::OpenDirectoryMode_All ) );
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseDirectory( directoryHandle );
    };

    while ( NN_STATIC_CONDITION( true ) )
    {
        int64_t entryCount = 0;
        nn::fs::DirectoryEntry entry;
        NN_RESULT_DO( nn::fs::ReadDirectory( &entryCount, &entry, directoryHandle, 1 ) );
        if ( 0 >= entryCount )
        {
            break;
        }
        if ( entry.directoryEntryType == nn::fs::DirectoryEntryType_File )
        {
            if ( HasExtension( entry.name, ".nsp" ) || HasExtension( entry.name, ".nspu" ) )
            {
                FilePropertyType filePropery;
                filePropery.Prepare( entry, directoryPath );
                pOutEntryList->push_back( filePropery );
            }
        }

        else if ( entry.directoryEntryType == nn::fs::DirectoryEntryType_Directory )
        {
            char childDirectoryPath[ MaxDirectoryNameLength ];
            nn::util::SNPrintf( childDirectoryPath, sizeof( childDirectoryPath ), "%s%s/", directoryPath, entry.name );

            // 子ディレクトリに対して再帰的に関数を呼び出す
            NN_RESULT_DO( EnumerateFiles( pOutEntryList, childDirectoryPath ) );
        }
    }
    NN_RESULT_SUCCESS;
}

/**
 * @brief SD カードをマウントしファイルリストを作成。
 */
const nn::Result InstallScene::MakeNspList() NN_NOEXCEPT
{
#if !defined ( NN_BUILD_CONFIG_OS_WIN )
    // SD カードをマウントしファイルリストを作成
    m_NspList.clear();
    NN_RESULT_DO( nn::fs::MountSdCardForDebug( "sd" ) );

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount( "sd" );
    };

    std::vector< FilePropertyType > fileList;

    NN_RESULT_DO( EnumerateFiles( &m_NspList, "sd:/" ) );

    // 表示ファイル名の調整
    for ( auto& iter : m_NspList )
    {
        // wchar_t 型の標準関数( snwprintfなど )は、gccでは 4byteのためクロスプラットフォームでは使えないので
        // 一度 char* 型に変換して ClipMessage を行う。
        char fileName[ MaxDirectoryNameLength ];
        ConvertWideStringToString( fileName, iter.name );

        BuildUtf16< MaxDirectoryNameLength >( iter.name, "%s", ClipMessage( fileName, MaxCatalogFileNameLength ).c_str() );
    }

    // リストに反映
    EntryProperties( &m_NspList );
#endif

    NN_RESULT_SUCCESS;
}

/**
 * @brief SD カード内のプロパティコレクションを登録します。
 * @details 登録内容に応じてリスト or アイテムなしメッセージの切り替えを行います。
 */
void InstallScene::EntryProperties( const SdCardListView::CollectionType* pNsps ) NN_NOEXCEPT
{
    const glv::Property::t focusableProperty = glv::Property::t::Controllable | glv::Property::t::HitTest;

    m_NspListView.EntryCollection( m_NspList );
    if ( true == m_NspList.empty() )
    {
        m_NspListView.disable( focusableProperty );
        m_LabelNoNsp.enable( glv::Property::t::Visible );
        m_LabelNoNsp.bringToFront();
        this->SetFirstFocusTargetView( &m_ButtonToCatalog );
    }
    else
    {
        m_NspListView.enable( focusableProperty );
        m_LabelNoNsp.disable( glv::Property::t::Visible );
        this->SetFirstFocusTargetView( &m_NspListView );
    }
}

/**
 * @brief シーンにフォーカスが与えられた際に、フォーカスを横取る子供を指定します。
 */
glv::View* InstallScene::GetFocusableChild() NN_NOEXCEPT
{
    return &m_ButtonAllCheckboxes;
}

/**
 * @brief ダイアログを表示して確認を取った後、インストールを行います。
 */
void InstallScene::ConfirmInstall( const FilePropertyType* pProperty ) NN_NOEXCEPT
{
    const auto fileListSize = m_NspList.size();
    auto pView = new ConfirmView( fileListSize > 0 );

    if ( fileListSize == 0 )
    {
        pView->AddMessage( "No file is selected. Select at least one nsp." );
        pView->AddButton( "Close" );
    }
    else
    {
        size_t targetCount = ( m_IsInstallAll == true ) ? fileListSize : 1;
        size_t overSize = 0;

        // 確認表示時のファイル数が 1 つ
        if ( 1 == targetCount )
        {
            pView->AddMessage( "Install the selected file?" );
        }
        // 確認表示時のファイル数が複数
        else
        {
            pView->AddMessage( "Install these selected files?" );
            // 上限を MaxDisplayedNspsCountInConfirmView 個までとする。
            if ( targetCount > MaxDisplayedNspsCountInConfirmView )
            {
                overSize = targetCount - MaxDisplayedNspsCountInConfirmView;
                targetCount = MaxDisplayedNspsCountInConfirmView;
            }
        }

        // ファイル名を表示
        std::string fileNamesStr( "" );

        for ( size_t i = 0; i < targetCount; i++ )
        {
            char fileName[ MaxDirectoryNameLength ];
            // リストから 1 つを選択したとき
            if ( m_IsInstallAll == false )
            {
                ConvertWideStringToString( fileName, pProperty->name );
            }
            else
            {
                ConvertWideStringToString( fileName, m_NspList.at( i ).name );
            }
            // モーダルにインストールするファイル名を表示、文字数が多い場合は最初と最後から数文字だけを表示
            fileNamesStr += ClipMessage( fileName, MaxModalMessageLength );
            if ( i < targetCount - 1 )
            {
                fileNamesStr += '\n';
            }

        }

        pView->AddMessage( fileNamesStr );

        // 列挙するファイル数が MaxComfirmNum を超えた場合、残りの数を表示して省略
        if ( overSize > 0 )
        {
            if ( overSize == 1 )
            {
                pView->AddMessage( "and another file." );
            }
            else
            {
                char message[ MaxModalMessageLength ];
                nn::util::SNPrintf( message, sizeof( message ), "and the other %d files.", overSize );
                pView->AddMessage( message );
            }
        }

        pView->AddMessage( "Target Location:" );

        pView->AddButton("Cancel");

        // 開始ボタンを追加
        pView->AddButton(
            "Start",
            [ this, pView ]( void* pParam, nn::TimeSpan& timespan )
            {
                m_TargetStorageId = pView->GetSelectedValue();
                InstallNspFile();
            },
            nullptr,
            MessageView::ButtonTextColor::Green
        );
    }
    pView->ExtentSize();
    this->GetRootSurfaceContext()->StartModal( pView, true );
}

/**
 * @brief 選択された nsp ファイルのインストールを行います。
 */
void InstallScene::InstallNspFile() NN_NOEXCEPT
{
    // インストールスレッドのスタック
    static NN_OS_ALIGNAS_THREAD_STACK nn::Bit8 s_InstallStack[ 32 * 1024 ];

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::CreateThread( &m_InstallTriggerThread, InstallTriggerThreadFunc, this, s_InstallStack, sizeof( s_InstallStack ), nn::os::DefaultThreadPriority )
    );
    nn::os::StartThread( &m_InstallTriggerThread );

    // インストール進捗ビューを登録
    m_pInstallView = new InstallView( [&]{ CloseButtonCallback(); }, [&]{ CancelButtonCallback(); }, [&]{ CleanupButtonCallback(); }, [&]{ NextButtonCallback(); } );

    // コンテンツのインストール処理中はフローティング起動をブロックする
    this->StartModal( m_pInstallView, true, true, false, true );
}

/**
 * @brief インストール進捗画面の Close ボタン押下時のコールバック関数です。
 */
void InstallScene::CloseButtonCallback() NN_NOEXCEPT
{
    m_InstallState = InstallState_NotStarted;
    DestroyInstallTriggerThread();

    if ( !m_IsInstallAll )
    {
        this->SetFocus( &m_NspListView );
    }
}

/**
 * @brief インストール進捗画面の Cancel ボタン押下時のコールバック関数です。
 */
void InstallScene::CancelButtonCallback() NN_NOEXCEPT
{
    m_InstallState = InstallState_NotStarted;
    m_IsCancelRequested = true;
    DestroyInstallTriggerThread();
}

/**
 * @brief インストール進捗画面の Cleanup ボタン押下時のコールバック関数です。
 */
void InstallScene::CleanupButtonCallback() NN_NOEXCEPT
{
    m_InstallState = InstallState_NotStarted;
    m_IsCancelRequested = true;
    DestroyInstallTriggerThread();

    // SD カードのクリーンアップ処理実行
    auto pView = new MessageView( true );

    pView->AddMessage( "Start the SD card cleanup?\nDevice will be automatically rebooted after cleanup." );
    pView->AddButton( "Cancel " );
    pView->AddButton(
        "Start",
        [ this ]( void* pParam, nn::TimeSpan& timespan ){
            const auto result = nn::ns::CleanupSdCard();

            if ( result.IsFailure() )
            {
                auto pView = new MessageView( false );
                CreateErrorMessageView( pView, "Cleanup failed. SD card is in unexpected state.", result );
                this->StartModal( pView, true, true, true );

                return;
            }
            RequestRebootDevice();
        },
        nullptr,
        MessageView::ButtonTextColor::Green
    );
    this->StartModal( pView, true, true );
}

/**
 * @brief インストール進捗画面の Next ボタン押下時のコールバック関数です。
 */
void InstallScene::NextButtonCallback() NN_NOEXCEPT
{
    m_InstallState = InstallState_NotStarted;

    // インストールを続けるので View を再生成する
    m_pInstallView = new InstallView( [&]{ CloseButtonCallback(); }, [&]{ CancelButtonCallback(); }, [&]{ CleanupButtonCallback(); }, [&]{ NextButtonCallback(); } );

    // コンテンツのインストール処理中はフローティング起動をブロックする
    this->StartModal( m_pInstallView, true, true, false, true );
}

/**
 * @brief インストールトリガスレッドのメイン関数です。
 */
void InstallScene::InstallTriggerThreadFunc( void* args ) NN_NOEXCEPT
{
    InstallScene* self = reinterpret_cast< InstallScene* >( args );

    NN_ASSERT_NOT_NULL( self );
    NN_ASSERT_EQUAL( self->m_InstallState, InstallState_NotStarted );


#if !defined ( NN_BUILD_CONFIG_OS_WIN ) // ビルドエラーを回避
    auto result = nn::fs::MountSdCardForDebug( "sd" );
    if ( result.IsFailure() )
    {
        self->SetInstallResult( result );
        return;
    }
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount( "sd" );
    };
#endif

    char fileName[ MaxDirectoryNameLength ];

    // Install All ボタンでの処理
    if ( self->m_IsInstallAll == true )
    {
        self->m_IsCancelRequested = false;
        const auto fileListSize = self->m_NspList.size();

        for ( size_t i = 0; i < fileListSize; ++i )
        {
            ConvertWideStringToString( fileName, self->m_NspList.at( i ).name );
            while ( InstallState_NotStarted != self->m_InstallState )
            {
                nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( 100 ) );
            }
            if ( self->m_IsCancelRequested ) // Cancel ボタン押下
            {
                break;
            }

            self->m_IsLastTarget = ( i == ( fileListSize - 1 ) );
            self->SetInstallResult( self->InstallAndVerifyCombination( fileName ) );
        }
        self->m_IsInstallAll = false;
    }
    // nsp リストから 1 つを選択時の処理
    else
    {
        self->m_IsLastTarget = true;
        ConvertWideStringToString( fileName, self->m_NspList.at( self->m_NspListView.GetSelectedPosition() ).name );
        self->SetInstallResult( self->InstallAndVerifyCombination( fileName ) );
    }
}

/**
 * @brief インストールトリガスレッドを破棄します。
 */
void InstallScene::DestroyInstallTriggerThread() NN_NOEXCEPT
{
    nn::os::WaitThread( &m_InstallTriggerThread );
    nn::os::DestroyThread( &m_InstallTriggerThread );
}

/**
 * @brief インストール結果を設定します。
 */
void InstallScene::SetInstallResult( const nn::Result& result ) NN_NOEXCEPT
{
    std::lock_guard< nn::os::Mutex > lock( m_Mutex );
    m_InstallState = result.IsSuccess() ? InstallState_Committed : InstallState_Failed;
    m_pInstallView->SetInstallResult( result );
}

/**
 * @brief nsp ファイルをインストールして、組み合わせをチェックします。
 */
const nn::Result InstallScene::InstallAndVerifyCombination( const char* filePath ) NN_NOEXCEPT
{
    const int maxContentMetaKeyCount = 2048;
    int contentMetaKeyCount;
    bool isSuccess;
    nn::ncm::ApplicationId applicationId;
    std::unique_ptr< nn::ncm::StorageContentMetaKey[] > contentMetaKeys( new nn::ncm::StorageContentMetaKey[ maxContentMetaKeyCount ] );

    NN_RESULT_DO( ExecuteInstall( contentMetaKeys.get(), &contentMetaKeyCount, &applicationId, maxContentMetaKeyCount, filePath ) );
    NN_RESULT_DO( devmenuUtil::VerifyCombinationWithApplicationAtPatchInstall( &isSuccess, contentMetaKeys.get(), contentMetaKeyCount, applicationId ) );
    if ( !isSuccess )
    {
        NN_RESULT_DO( devmenuUtil::DeleteContent( contentMetaKeys.get(), contentMetaKeyCount, applicationId ) );
        NN_RESULT_THROW( ResultApplicationAndPatchTypeNotMatched() );
    }

    NN_RESULT_SUCCESS;
}

/**
 * @brief nsp ファイルをインストールします。
 */
const nn::Result InstallScene::ExecuteInstall( nn::ncm::StorageContentMetaKey* pOutInstalledKey, int* pOutCount, nn::ncm::ApplicationId* pOutId, int maxOutCount, const char* filePath ) NN_NOEXCEPT
{
#if !defined ( NN_BUILD_CONFIG_OS_WIN )

    // ターゲットが SD カードの場合は状態をチェックする
    if ( nn::ncm::StorageId::SdCard == m_TargetStorageId )
    {
        NN_RESULT_DO( nn::ns::CheckSdCardMountStatus() );
    }

    nn::fs::FileHandle handle;

    NN_RESULT_DO( nn::fs::OpenFile( &handle, filePath, nn::fs::OpenMode::OpenMode_Read ) );
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile( handle );
    };

    // 不要なアプリケーション実体を削除
    NN_RESULT_DO( nn::ns::DeleteRedundantApplicationEntity() );

    // TORIAEZU: Pad からの入力で Progress の ModalView が消えないようするための Work Aruond
    // ひとつ前の ModalView が EndModal してから呼ぶ必要があるが、タイミングは適当
    this->SetFocus( nullptr );

    // インストールタスクの生成
    std::unique_ptr< nn::ns::ApplicationInstallTask > task( new nn::ns::ApplicationInstallTask );
    m_pInstallTask = task.get();

    // インストールタスクを初期化して処理を開始
    const bool isNspuFile = HasExtension( filePath, ".nspu" );
    std::unique_ptr< char[] > buffer( new char[ InstallBufferSize ] );
    NN_RESULT_DO( task->Initialize( handle, m_TargetStorageId, buffer.get(), InstallBufferSize, isNspuFile ) );
    DEVMENU_LOG( "Preparing to install %s...\n", filePath );

    NN_RESULT_DO( task->Prepare() );

    bool needsCleanup = true;
    NN_UTIL_SCOPE_EXIT
    {
        if ( needsCleanup == true )
        {
            task->Cleanup();
        }
    };

    nn::ncm::ApplicationContentMetaKey applicationKey;

    const int maxContentMetaCount = 2048;
    int contentMetaKeyCount;
    int applicationMetaKeyCount;
    std::unique_ptr< nn::ncm::StorageContentMetaKey[] > keys( new nn::ncm::StorageContentMetaKey[ maxContentMetaCount ] );

    // インストール前にインストーラブルなコンテンツが入っているかをチェックする
    {
        NN_RESULT_DO( task->ListContentMetaKey( &contentMetaKeyCount, keys.get(), maxContentMetaCount, 0 ) );
        NN_RESULT_DO( task->ListApplicationContentMetaKey( &applicationMetaKeyCount, &applicationKey, 1, 0 ) );

        // key が 1 つも返って来ない == application が含まれていないシステムコンテンツの可能性あり
        if ( 0 == applicationMetaKeyCount )
        {
            needsCleanup = true;
            NN_RESULT_THROW( nn::ncm::ResultContentNotFound() );
        }
        const nn::ncm::ContentMetaType type = applicationKey.key.type;

        // インストールできるコンテンツでなければエラーを投げる
        if ( nn::ncm::ContentMetaType::Application != type && nn::ncm::ContentMetaType::AddOnContent != type && nn::ncm::ContentMetaType::Patch != type )
        {
            needsCleanup = true;
            NN_RESULT_THROW( nn::ncm::ResultContentNotFound() );
        }

        // 必要な情報は取得したので、いったん cleanup
        needsCleanup = false;
        NN_RESULT_DO( task->Cleanup() );
    }

    const nn::ncm::ApplicationId applicationId = applicationKey.applicationId;

    // アプリケーション記録の再生成要求管理
    bool requestsRecreateRecord = false;
    NN_UTIL_SCOPE_EXIT
    {
        RecreateApplicationRecord( applicationId, requestsRecreateRecord );
    };

    // 既にインストールされているかどうかチェックする
    {
        for ( int i = 0; i < contentMetaKeyCount; ++i )
        {
            auto key = keys[ i ];

            // 既にインストール済みで、そちらのバージョンが高い場合は --force オプションがない場合エラー
            bool existsHigherOrEqualVersion;
            NN_RESULT_DO( devmenuUtil::ExistsHigherOrEqualVersion( &existsHigherOrEqualVersion, applicationId, key.key ) );

            if ( existsHigherOrEqualVersion )
            {
                // 新しいものを削除し、アプリケーション記録の再生成要求をしておく
                NN_RESULT_TRY( devmenuUtil::DeleteBasedOnDatabase( key.key.type, key.key.id, key.storageId ) )
                    NN_RESULT_CATCH( nn::ncm::ResultContentMetaNotFound ) {}
                NN_RESULT_END_TRY
                requestsRecreateRecord = true;
            }
        }
        NN_RESULT_DO( task->Prepare() );
        needsCleanup = true;
    }

    // TORIAEZU: 念のため、もう一度呼んでおく
    this->SetFocus( nullptr );

    m_pInstallView->SetProgressFileName( ClipMessage( filePath, MaxModalMessageLength ).c_str() );
    NN_RESULT_DO( StartAndWaitForExitInstallExecutionThread() );
    m_pInstallView->SetLastInstalledNspTotalSize( task->GetProgress().totalSize );
    NN_RESULT_DO( task->Commit() );
    requestsRecreateRecord = true;
    needsCleanup = false;

    nn::ns::InvalidateApplicationControlCache( applicationId );

    *pOutCount = std::min( maxOutCount, contentMetaKeyCount );
    std::memcpy( pOutInstalledKey, &keys[ 0 ], sizeof( keys[ 0 ] ) * ( *pOutCount ) );
    *pOutId = applicationKey.applicationId;
#else
    NN_UNUSED( pOutInstalledKey );
    NN_UNUSED( pOutId );
    NN_UNUSED( maxOutCount );
    NN_UNUSED( filePath );
    *pOutCount = 0;
#endif // !defined ( NN_BUILD_CONFIG_OS_WIN )

    NN_RESULT_SUCCESS;
}

/**
 * @brief InstallTask 実行の事前準備を行います。
 */
void InstallScene::RecreateApplicationRecord( const nn::ncm::ApplicationId& applicationId, bool requestsRecreateRecord ) NN_NOEXCEPT
{
    nn::Result result;
    if ( requestsRecreateRecord )
    {
        result = devmenuUtil::PushApplicationRecordBasedOnDatabase( applicationId );
        if ( result.IsFailure() )
        {
            DEVMENU_LOG( "PushApplicationRecord failed: %0x08x\n", result.GetInnerValueForDebug() );
        }
    }

    result = nn::ns::CleanupUnrecordedApplicationEntity( applicationId );
    if ( result.IsFailure() )
    {
        DEVMENU_LOG( "Cleanup failed: %0x08x\n", result.GetInnerValueForDebug() );
    }
}

/**
 * @brief インストール処理を別スレッドで実行します。
 */
nn::Result InstallScene::StartAndWaitForExitInstallExecutionThread() NN_NOEXCEPT
{
    static NN_OS_ALIGNAS_THREAD_STACK nn::Bit8 s_Stack[ 16 * 1024 ];
    static nn::Result s_Result = nn::ResultSuccess();

    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &m_InstallExecutionThread, []( void* p ) {
        s_Result = reinterpret_cast< nn::ncm::InstallTaskBase* >( p )->Execute();
    }, m_pInstallTask, s_Stack, sizeof( s_Stack ), nn::os::DefaultThreadPriority ) );
    nn::os::StartThread( &m_InstallExecutionThread );

    {
        std::lock_guard< nn::os::Mutex > lock( m_Mutex );
        m_InstallState = InstallState_Running;
        m_StartTick = nn::os::Tick( 0 );
    }

    while ( NN_STATIC_CONDITION( true ) )
    {
        const auto isFinished = nn::os::TryWaitAny( &m_InstallExecutionThread ) >= 0;
        if ( isFinished )
        {
            break;
        }
        nn::os::SleepThread( nn::TimeSpan::FromSeconds( 1 ) );
    }
    nn::os::WaitThread( &m_InstallExecutionThread );
    nn::os::DestroyThread( &m_InstallExecutionThread );

    return s_Result;
}

/**
 * @brief インストール状況を更新します。
 */
void InstallScene::UpdateInstallProgress() NN_NOEXCEPT
{
    // Note: 以下の遷移は別スレッドで行われる
    //   - InstallState_NotStarted -> InstallState_Running
    //   - InstallState_NotStarted -> InstallState_Failed
    //   - InstallState_Running    -> InstallState_Committed
    //   - InstallState_Running    -> InstallState_Failed (異常ケースで稀)

    std::lock_guard< nn::os::Mutex > lock( m_Mutex );

    switch ( m_InstallState )
    {
    case InstallState_NotStarted:
        // ApplicationInstallTask が Execute されると InstallState_Running に遷移する
        // ApplicationInstallTask を Execute するまでの処理に Fail すると InstallState_Failed に遷移する
        break;
    case InstallState_Running:
        // ApplicationInstallTask が Commit されると InstallState_Committed に遷移する
        // ApplicationInstallTask の Commit に失敗すると InstallState_Failed に遷移する
        {
            const int64_t pastTimeMilliSeconds = nn::os::ConvertToTimeSpan( nn::os::GetSystemTick() - m_StartTick ).GetMilliSeconds();
            if ( 1000LL < pastTimeMilliSeconds )
            {
                const auto progress = m_pInstallTask->GetProgress();
                if ( 0 == progress.installedSize || 0 == progress.totalSize )
                {
                    return;
                }

                // 進捗 100.0% が 2 度続けてログ出力されないようにする
                if ( progress.installedSize != progress.totalSize )
                {
                    char progressStr[ 48 ];
                    const auto installedSizeStr = GetDelimitedNumberString( progress.installedSize );
                    const auto totalSizeStr = GetDelimitedNumberString( progress.totalSize );

                    const float progressRate = static_cast< float >( progress.installedSize ) / static_cast< float >( progress.totalSize );
                    const float progressRatePercent = 100 * progressRate;
                    nn::util::SNPrintf( progressStr, sizeof( progressStr ), "%s / %s (%.1f%)", installedSizeStr.c_str(), totalSizeStr.c_str(), progressRatePercent );
                    m_pInstallView->SetProgressLabel( progressStr );
                    m_pInstallView->SetProgressBarValue( progressRate );
                    DEVMENU_LOG( "%s\n", progressStr );
                }
                m_StartTick = nn::os::GetSystemTick();
            }
        }
        break;
    case InstallState_Committed:
        {
            // 進捗 100% の表示
            char progressStr[ 48 ];
            const auto totalSizeStr = GetDelimitedNumberString( m_pInstallView->GetLastInstalledNspTotalSize() );
            nn::util::SNPrintf( progressStr, sizeof( progressStr ), "%s / %s (100.0%)", totalSizeStr.c_str(), totalSizeStr.c_str() );
            m_pInstallView->SetProgressCaption( "[Install Completed]" );
            m_pInstallView->SetProgressLabel( progressStr );
            m_pInstallView->SetProgressBarValue( 1.0f );
            DEVMENU_LOG( "%s\n", progressStr );

            // 次の state を決定する
            m_InstallState = m_IsLastTarget ? InstallState_Completed : InstallState_NotStarted;
        }
        break;
    case InstallState_Completed:
        {
            m_pInstallView->SetCloseButton();
            m_InstallState = InstallState_NotStarted;
        }
        break;
    case InstallState_Failed:
        {
            // TORIAEZU: InstallAll 時は失敗したらボタンを押すまでインストール処理を止める
            m_pInstallView->SetErrorMessageView( m_IsLastTarget );
            m_InstallState = InstallState_WaitUserAction;
        }
        break;
    case InstallState_WaitUserAction:
        // ボタン押下待ち. 押下されると InstallState_NotStarted にステート遷移する
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

/**
 * @brief       表示する文字数に上限を設けて超えた場合は省略した形を生成します。
 *
 * @param[in]   string          文字列
 * @param[in]   maxLength       表示する文字の上限数( 9 以上 MaxDirectoryNameLength - 1 以内 )
 */
const std::string InstallScene::ClipMessage( const char* string, size_t maxLength ) NN_NOEXCEPT
{
    NN_ASSERT( 9 <= maxLength && maxLength < MaxDirectoryNameLength );

    const size_t stringLength = strlen( string );

    // 文字数が多い場合は、最初と最後から数文字だけを表示
    if ( stringLength > maxLength )
    {
        // " ... " と終端文字の長さを引く
        const size_t includeMessageLength = maxLength - 6;

        // 表示する文字を前半:後半 = 1:2 とする
        const size_t firstSize = includeMessageLength / 3;
        const size_t endSize = std::ceilf( includeMessageLength * 2 / 3 );

        const std::string firstMessage( string, string + firstSize - 1 );
        const std::string endMessage( string + stringLength - endSize, string + stringLength );

        return firstMessage + std::string( " ... " ) + endMessage;
    }
    return std::string( string );
}

/**
 * @brief チェックボックスの全選択。
 */
void InstallScene::SelectAllCheckboxes() NN_NOEXCEPT
{

}

}} // ~namespace devmenu::application, ~namespace devmenu
