﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <nn/ns/ns_ApplicationEntitySystemApi.h>
#include <nn/util/util_StringUtil.h>

#include "DevMenu_ApplicationCatalog.h"
#include "DevMenu_ApplicationFileListSelector.h"
#include "DevMenu_Notification.h"
#include "DevMenuCommand_GameCardWriterImpl.h"

namespace devmenu { namespace application { namespace select {

const glv::space_t SdCardListView::HorizontalMargin_Name = 4.0f;
const glv::space_t SdCardListView::HorizontalLength_Name = 768.0f;

namespace {

    const size_t MaxCatalogFileNameLength = 75;

    /**
     * @brief       ファイル名に指定した拡張子( '.' を含む)が含まれるかをチェックします。
     */
    bool HasExtension( const char* str, const char* extension ) NN_NOEXCEPT
    {
        if ( strlen( str ) <= strlen( extension ) + 1 )
        {
            return false;
        }
        return ( nn::util::string_view( str ).substr( strlen( str ) - strlen( extension ) ) == nn::util::string_view( extension ) );
    }

    void ConvertWideStringToString( char* pOutStr, const glv::WideCharacterType* fileName ) NN_NOEXCEPT
    {
        int fileListLength = 0;
        const uint16_t* pFileList = reinterpret_cast< const uint16_t* >( fileName );
        nn::util::GetLengthOfConvertedStringUtf16NativeToUtf8( &fileListLength, pFileList, MaxDirectoryNameLength );
        nn::util::ConvertStringUtf16NativeToUtf8( pOutStr, fileListLength, pFileList );
    }

} // end of anonymous namespace

/**************************************
 * class FileListSelectorScene
 **************************************/

// シザーボックスの領域設定
const glv::space_t FileListSelectorScene::HeaderRegionLength = 80.0f;
const glv::space_t FileListSelectorScene::FooterRegionLength = 48.0f;

// リスト要素のパディング ( ボーダーを表示する場合は 2 以上必要 )
const glv::space_t FileListSelectorScene::ListMarginLeft = 16.0f;
const glv::space_t FileListSelectorScene::ListMarginRight = 16.0f;
const glv::space_t FileListSelectorScene::ListMarginVertical = 2.0f;

const glv::Label::Spec FileListSelectorScene::LabelSpecNoNsp( glv::Place::CC, 0.0f, 0.0f, 32.0f );

/**
 * @brief コンストラクタです。
 */
FileListSelectorScene::FileListSelectorScene( application::ApplicationCatalogPage* pParentPage, Scene* pParentScene ) NN_NOEXCEPT
    : Scene( pParentPage, pParentPage->rect() ),
    m_ButtonToCatalog( "< Back", [&]{ m_pParentPage->SwitchScene( this, m_pParentScene, true ); }, CommonValue::InitialFontSize, 16.0f ),
    m_ListContainer( glv::Rect( 0.0f, FileListSelectorScene::HeaderRegionLength, width(), height() - ( FileListSelectorScene::HeaderRegionLength + FileListSelectorScene::FooterRegionLength ) ) ),
    m_NspListView( glv::Rect( FileListSelectorScene::ListMarginLeft, FileListSelectorScene::ListMarginVertical,
    m_ListContainer.width() - ( FileListSelectorScene::ListMarginLeft + FileListSelectorScene::ListMarginRight ), m_ListContainer.height() - ( FileListSelectorScene::ListMarginVertical * 2 ) ) ),
    m_LabelNoNsp( "No nsp file is found.", FileListSelectorScene::LabelSpecNoNsp ),
    m_pParentScene( pParentScene ),
    m_pParentPage( pParentPage ),
    m_IsNspFileSelected( false )
{
    // ボタンの登録
    {
        // Catalog に戻るボタン
        m_ButtonToCatalog.anchor( glv::Place::TL ).pos( glv::Place::TL, ListMarginLeft, 4.0f );
        *this << m_ButtonToCatalog;
    }

    // ヘッダ
    auto pHeaderName = new glv::Label( "Name", glv::Label::Spec( glv::Place::TL, ListMarginLeft, 40.0f + 8.0f, CommonValue::InitialFontSize ) );
    auto pHeaderSize = new glv::Label( "Size(Byte)", glv::Label::Spec( glv::Place::TR, -( ListMarginRight + 16.0f ), pHeaderName->top(), CommonValue::InitialFontSize ) );
    {
        *this << pHeaderName;
        *this << pHeaderSize;
    }

    // リスト
    {
        m_IsNspFileSelected = false;
        m_FileName[0] = '\0';

        // 一覧からNSP選択時のコールバック関数を登録
        m_NspListView.attach(
            []( const glv::Notification& notification )->void {
                auto pScene = notification.receiver< FileListSelectorScene >();
                const FilePropertyType* pProperty = notification.data< SdCardListView::ItemType >();
                if ( nullptr != pScene && nullptr != pProperty )
                {
                    pScene->GetSelectedFilePathAndChangeScene();
                }
            },
            glv::Update::Action,
            this
        );

        m_ListContainer << m_NspListView;

        // NSP ファイルなしメッセージ
        m_ListContainer << m_LabelNoNsp;

        *this << m_ListContainer;
    }

    this->SetFirstFocusTargetView( &m_NspListView );
}

void FileListSelectorScene::Refresh() NN_NOEXCEPT
{
}

const nn::Result FileListSelectorScene::EnumerateFiles( std::vector< FilePropertyType >* pOutEntryList, const char* directoryPath ) NN_NOEXCEPT
{
    nn::fs::DirectoryHandle directoryHandle;
    NN_RESULT_DO( nn::fs::OpenDirectory( &directoryHandle, directoryPath, nn::fs::OpenDirectoryMode_All ) );
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseDirectory( directoryHandle );
    };

    while ( NN_STATIC_CONDITION( true ) )
    {
        int64_t entryCount = 0;
        nn::fs::DirectoryEntry entry;
        NN_RESULT_DO( nn::fs::ReadDirectory( &entryCount, &entry, directoryHandle, 1 ) );
        if ( 0 >= entryCount )
        {
            break;
        }
        if ( entry.directoryEntryType == nn::fs::DirectoryEntryType_File )
        {
            if ( HasExtension( entry.name, ".nsp" ) || HasExtension( entry.name, ".nspu" ) )
            {
                FilePropertyType filePropery;
                filePropery.Prepare( entry, directoryPath );
                pOutEntryList->push_back( filePropery );
            }
        }

        else if ( entry.directoryEntryType == nn::fs::DirectoryEntryType_Directory )
        {
            char childDirectoryPath[ MaxDirectoryNameLength ];
            nn::util::SNPrintf( childDirectoryPath, sizeof( childDirectoryPath ), "%s%s/", directoryPath, entry.name );

            // 子ディレクトリに対して再帰的に関数を呼び出す
            NN_RESULT_DO( EnumerateFiles( pOutEntryList, childDirectoryPath ) );
        }
    }
    NN_RESULT_SUCCESS;
}

/**
 * @brief SD カードをマウントしファイルリストを作成。
 */
const nn::Result FileListSelectorScene::MakeNspList() NN_NOEXCEPT
{
    m_FileName[0] = '\0';
    m_IsNspFileSelected = false;

#if !defined ( NN_BUILD_CONFIG_OS_WIN )
    // SD カードをマウントしファイルリストを作成
    m_NspList.clear();

    // 初期状態をリストに反映（SDの挿抜でリスト数が残る場合の対策）
    EntryProperties(&m_NspList);

    NN_RESULT_DO( nn::fs::MountSdCardForDebug( "sd" ) );

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount( "sd" );
    };

    std::vector< FilePropertyType > fileList;

    NN_RESULT_DO( EnumerateFiles( &m_NspList, "sd:/" ) );

    // 表示ファイル名の調整
    for ( auto& iter : m_NspList )
    {
        // wchar_t 型の標準関数( snwprintfなど )は、gccでは 4byteのためクロスプラットフォームでは使えないので
        // 一度 char* 型に変換して ClipMessage を行う。
        char fileName[ MaxDirectoryNameLength ];
        ConvertWideStringToString( fileName, iter.name );

        iter.longName = std::string( fileName );
        BuildUtf16< MaxDirectoryNameLength >( iter.name, "%s", ClipMessage( fileName, MaxCatalogFileNameLength ).c_str() );
    }

    // リストに反映
    EntryProperties( &m_NspList );
#endif

    NN_RESULT_SUCCESS;
}

/**
* @brief NSPファイルが選択されているかを判定します。
*/
bool FileListSelectorScene::IsNspFileSelected() NN_NOEXCEPT
{
    return m_IsNspFileSelected;
}

/**
* @brief ファイル選択情報を初期化します。
*/
void FileListSelectorScene::InitializeFileSelected() NN_NOEXCEPT
{
    m_IsNspFileSelected = false;
    m_ContentMetaType = nn::ncm::ContentMetaType::Unknown;
}

/**
* @brief 選択されたNSPファイルパスを取得します。
*/
const char* FileListSelectorScene::GetSelectedNspFilePath() NN_NOEXCEPT
{
    return m_FileName;
}

/**
* @brief コンテンツタイプを設定
*/
void FileListSelectorScene::SetContentType( nn::ncm::ContentMetaType contentMetaType ) NN_NOEXCEPT
{
    m_ContentMetaType = contentMetaType;
}

/**
* @brief コンテンツタイプを取得
*/
nn::ncm::ContentMetaType FileListSelectorScene::GetContentType() NN_NOEXCEPT
{
    return m_ContentMetaType;
}

/**
 * @brief SD カード内のプロパティコレクションを登録します。
 * @details 登録内容に応じてリスト or アイテムなしメッセージの切り替えを行います。
 */
void FileListSelectorScene::EntryProperties( const SdCardListView::CollectionType* pNsps ) NN_NOEXCEPT
{
    const glv::Property::t focusableProperty = glv::Property::t::Controllable | glv::Property::t::HitTest;

    m_NspListView.EntryCollection( m_NspList );
    if ( true == m_NspList.empty() )
    {
        m_NspListView.disable( focusableProperty );
        m_LabelNoNsp.enable( glv::Property::t::Visible );
        m_LabelNoNsp.bringToFront();
        this->SetFirstFocusTargetView( &m_ButtonToCatalog );
    }
    else
    {
        m_NspListView.enable( focusableProperty );
        m_LabelNoNsp.disable( glv::Property::t::Visible );
        this->SetFirstFocusTargetView( &m_NspListView );
    }
}

/**
 * @brief シーンにフォーカスが与えられた際に、フォーカスを横取る子供を指定します。
 */
glv::View* FileListSelectorScene::GetFocusableChild() NN_NOEXCEPT
{
    // デフォルトフォーカスはFileListSelector AllボタンからBuckボタンに変更
    return &m_ButtonToCatalog;
}

/**
* @brief 選択された nsp ファイルパス取得及び画面遷移を行います。
*/
void FileListSelectorScene::GetSelectedFilePathAndChangeScene() NN_NOEXCEPT
{
    nn::Result result;
    bool isValidContent = false;
    std::string errorMessage( "" );

    result = CheckContentMeta( &isValidContent, &errorMessage );
    if ( result.IsSuccess() && isValidContent )
    {
        m_IsNspFileSelected = true;
        int fileLength = sizeof( m_FileName );
        nn::util::Strlcpy( m_FileName, m_NspList.at(m_NspListView.GetSelectedPosition()).longName.c_str(), fileLength );
        m_pParentPage->SwitchScene( this, m_pParentScene, true );
    }
    else
    {
        auto pView = new MessageView( false );
        pView->AddMessage( errorMessage );
        pView->AddButton( "Close" );
        m_pParentPage->GetRootSurfaceContext()->StartModal( pView, true );
    }
}

/**
 * @brief       表示する文字数に上限を設けて超えた場合は省略した形を生成します。
 *
 * @param[in]   string          文字列
 * @param[in]   maxLength       表示する文字の上限数( 9 以上 MaxDirectoryNameLength - 1 以内 )
 */
const std::string FileListSelectorScene::ClipMessage( const char* string, size_t maxLength ) NN_NOEXCEPT
{
    NN_ASSERT( 9 <= maxLength && maxLength < MaxDirectoryNameLength );

    const size_t stringLength = strlen( string );

    // 文字数が多い場合は、最初と最後から数文字だけを表示
    if ( stringLength > maxLength )
    {
        // " ... " と終端文字の長さを引く
        const size_t includeMessageLength = maxLength - 6;

        // 表示する文字を前半:後半 = 1:2 とする
        const size_t firstSize = includeMessageLength / 3;
        const size_t endSize = std::ceilf( includeMessageLength * 2 / 3 );

        const std::string firstMessage( string, string + firstSize - 1 );
        const std::string endMessage( string + stringLength - endSize, string + stringLength );

        return firstMessage + std::string( " ... " ) + endMessage;
    }
    return std::string( string );
}

/**
* @brief       コンテンツタイプをチェックする
*
* @param[out]  pOutValue        チェック結果(true:コンテンツタイプが一致  false:コンテンツタイプが不一致)
* @param[out]  pOutErrorMessage エラーメッセージ
* @return      チェックの成否を Result 値で返します。
*/
nn::Result FileListSelectorScene::CheckContentMeta( bool* pOutValue, std::string* pOutErrorMessage ) NN_NOEXCEPT
{
    nn::Result result;
    *pOutValue = false;

#if !defined ( NN_BUILD_CONFIG_OS_WIN )
    result = nn::fs::MountSdCardForDebug( "sd" );
    if ( result.IsFailure() )
    {
        *pOutErrorMessage = "Failed to check file";
        return result;
    }
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount( "sd" );
    };
#endif

    const char* pNspFilePath = m_NspList.at( m_NspListView.GetSelectedPosition() ).longName.c_str();
    result = ::CheckContentMeta( pOutValue, pNspFilePath, m_ContentMetaType );
    if ( result.IsSuccess() )
    {
        if ( !*pOutValue )
        {
            if ( m_ContentMetaType == nn::ncm::ContentMetaType::Application )
            {
                *pOutErrorMessage = "Application file is invalid.\nPlease select application file written into Game Card.\nApplication : " + std::string( pNspFilePath );
            }
            else
            {
                *pOutErrorMessage = "Patch file is invalid.\nPlease select patch file written into Game Card.\nPatch : " + std::string( pNspFilePath );
            }
        }
    }
    else
    {
        *pOutErrorMessage = "Failed to check file";
    }
    return result;
}

}}} // ~namespace devmenu::application::select, ~namespace devmenu::application, ~namespace devmenu
