﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "DevMenu_CommonScene.h"
#include "../DevMenu_RootSurface.h"

namespace devmenu {

/*********************************
 * class Scene
 *********************************/

Scene::Scene( Page* pParentPage, const glv::Rect& rect, bool isDefaultbackButtonCallbackUsed, const std::function< void() >& backButtonCallback ) NN_NOEXCEPT
    : glv::Group( rect )
    , m_pParentPage( pParentPage )
    , m_pFirstFocusTargetView( nullptr )
    , m_pLastFocusedView( nullptr )
    , m_BackButtonCallback( isDefaultbackButtonCallbackUsed ? [&]{ this->GetRootSurfaceContext()->MoveFocusToMenuTabs(); } : backButtonCallback )
{
    NN_ASSERT_NOT_NULL( pParentPage );
    this->Initialize();
}

Scene::Scene( const glv::Rect& rect, const std::function< void() >& backButtonCallback ) NN_NOEXCEPT
    : glv::Group( rect )
    , m_pParentPage( nullptr )
    , m_pFirstFocusTargetView( nullptr )
    , m_pLastFocusedView( nullptr )
    , m_BackButtonCallback( backButtonCallback )
{
    this->Initialize();
}

void Scene::Initialize() NN_NOEXCEPT
{
    // Set this scene to be able to detect clicks
    this->enable( glv::Property::HitTest | glv::Property::GestureDetectability );

    // Attach a click handler to detect if the B button was pressed
    this->attach( []( const glv::Notification& notification )->void
        {
            auto pSelf = notification.sender< Scene >();

            auto& glvRoot = reinterpret_cast< glv::GLV& >( pSelf->root() );
            if ( glvRoot.getBasicPadEvent().IsButtonUp< glv::BasicPadEventType::Button::B >() ||
                glvRoot.getDebugPadEvent().IsButtonUp< glv::DebugPadEventType::Button::B >())
            {
                // Call the given function when the B button is detected
                pSelf->OnBackButtonPressed();
            }
        }, glv::Update::Clicked, this );
}

glv::View* Scene::GetLastFocusedView() const NN_NOEXCEPT
{
    return IsDescendant( m_pLastFocusedView ) ? m_pLastFocusedView : nullptr;
}

bool Scene::IsDescendant( const glv::View* const pTargetView ) const NN_NOEXCEPT
{
    auto pView = pTargetView;

    while ( NN_STATIC_CONDITION( true ) )
    {
        if ( nullptr == pView->parent )
        {
            break;
        }

        pView = pView->parent;

        if ( this == pView )
        {
            return true;
        }
    }

    return false;
}

void Scene::SwitchScene( int nextSceneIndex, bool isBack ) NN_NOEXCEPT
{
    m_pParentPage->SwitchScene( nextSceneIndex, isBack );
}

RootSurfaceContext* Scene::GetRootSurfaceContext() const NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL( m_pParentPage );
    return m_pParentPage->GetRootSurfaceContext();
}

void Scene::StartModal( ModalView* pModalView, bool isDisposable, bool isDelayExecution, bool isAutomaticExitAtBackgroundEnabled, bool isFloatingApplicationLaunchBlocked ) NN_NOEXCEPT
{
    auto pRootSurfaceContext = this->GetRootSurfaceContext();
    NN_ASSERT_NOT_NULL( pRootSurfaceContext );
    if ( nullptr != pRootSurfaceContext )
    {
        this->GetRootSurfaceContext()->StartModal( pModalView, isDisposable, isDelayExecution, isAutomaticExitAtBackgroundEnabled, isFloatingApplicationLaunchBlocked );
    }
}

void Scene::SetFocus( glv::View* const pView ) NN_NOEXCEPT
{
    auto pRootSurfaceContext = this->GetRootSurfaceContext();
    NN_ASSERT_NOT_NULL( pRootSurfaceContext );
    if ( nullptr != pRootSurfaceContext )
    {
        this->GetRootSurfaceContext()->setFocus( pView );
    }
}

/*********************************
 * class ProgressModalSceneBase
 *********************************/
void ProgressModalSceneBase::StartProgressModalThread(
    const std::string& executionStr,
    const std::function< nn::Result( bool* ) >& startCallback,
    const std::function< void( const nn::Result&, bool ) >& completionCallback ) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL( startCallback );
    NN_ABORT_UNLESS_NOT_NULL( completionCallback );

    // 多重呼び出しチェック
    if ( ExecutionState_None != m_ExecutionState )
    {
        return;
    }

    m_StartCallback = startCallback;
    m_CompletionCallback    = completionCallback;
    m_ProgressMessageBase   = executionStr;
    nn::os::FenceMemoryStoreLoad();

    static NN_OS_ALIGNAS_THREAD_STACK char s_ThreadStack[ ProgressModalSceneBase::ThreadStackSize ];
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &m_ProgressDisplayThread, []( void* pParam )
        {
            reinterpret_cast< ProgressModalSceneBase* >( pParam )->ProgressDisplayThreadFunction();
        },
        this, s_ThreadStack, sizeof( s_ThreadStack ), nn::os::DefaultThreadPriority ) );
    nn::os::StartThread( &m_ProgressDisplayThread );
}

void ProgressModalSceneBase::StartModalForProgress( bool isFloatingApplicationLaunchBlocked ) NN_NOEXCEPT
{
    this->StartModal( &m_ProgressModalView, false, false, false, isFloatingApplicationLaunchBlocked );
}

void ProgressModalSceneBase::UpdateExecutionProgress() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC( int32_t, s_Count, = 0 );
    NN_FUNCTION_LOCAL_STATIC( nn::os::TimerEvent, s_Timer, ( nn::os::EventClearMode_ManualClear ) );

    switch ( m_ExecutionState )
    {
    case ExecutionState_Preparing:
        {
            // Complete まで大して時間がかからない( < 1500 ms ) 場合は Exporting 中のメッセージを表示しない
            s_Timer.StartPeriodic( nn::TimeSpan::FromMilliSeconds( 1500 ), nn::TimeSpan::FromMilliSeconds( 1000 ) );
            m_ExecutionState = ExecutionState_Running;
            m_ProgressModalView.SetMessage( "" );

            // ダイアログを表示する
            this->StartModalForProgress( true );

            // 透明なModalViewで事前表示しておく
            this->GetRootSurfaceContext()->SetTransparentStyle();

            // ProgressModalViewも透明にするためプロパティをdisableにする
            m_ProgressModalView.disable( glv::Property::DrawBack | glv::Property::Controllable
                | glv::Property::DrawBorder | glv::Property::HitTest );

            // Pad からの入力で Progress の ModalView が消えないようするための Work Around
            this->SetFocus( nullptr );
        }
        break;
    case ExecutionState_Running:
        {
            if ( s_Timer.TryWait() )
            {
                s_Timer.Clear();
                std::string progressMessage = m_ProgressMessageBase;
                for ( int i = 0; i < s_Count % ProgressModalSceneBase::ProgressMessageLoopCount; ++i )
                {
                    progressMessage += '.';
                }
                m_ProgressModalView.SetMessage( progressMessage );
                m_ProgressModalView.DisplayMessage();

                if ( true == GetRootSurfaceContext()->IsModalTransparentStyle() )
                {
                    // 透明なModalViewで事前表示されていたら非透明に戻す
                    this->GetRootSurfaceContext()->SetOpaqueStyle();

                    // ProgressModalViewもプロパティをenableに戻す
                    m_ProgressModalView.enable( glv::Property::DrawBack
                        | glv::Property::DrawBorder );

                    // Pad からの入力で Progress の ModalView が消えないようするための Work Around
                    this->SetFocus( nullptr );
                }

                if ( false == GetRootSurfaceContext()->IsModalViewRunning() )
                {
                    // サブスレッド処理中はアプリケーションのフローティング起動をブロックする
                    this->StartModalForProgress( true );

                    // Pad からの入力で Progress の ModalView が消えないようするための Work Around
                    this->SetFocus( nullptr );
                }
                s_Count++;
            }
        }
        break;
    case ExecutionState_Completed:
        {
            const auto isFinished = nn::os::TryWaitAny( &m_ProgressDisplayThread ) >= 0;
            if ( isFinished )
            {
                nn::os::WaitThread( &m_ProgressDisplayThread );
                nn::os::DestroyThread( &m_ProgressDisplayThread );
                s_Timer.Clear();
                s_Timer.Stop();
                s_Count = 0;
                m_ExecutionState = OnCompleteExecuteProgress( m_ExecutionResult, m_IsExecutionSuccess );
                m_ProgressMessageBase = "";
            }
        }
        break;
    case ExecutionState_None:
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

void ProgressModalSceneBase::ProgressDisplayThreadFunction() NN_NOEXCEPT
{
    std::string executeMessage = std::string( m_ProgressMessageBase ).append( "..." );

    m_ProgressModalView.SetMessage( executeMessage );
    m_ProgressModalView.ArrangeViewTable();
    m_ProgressModalView.HideCloseButton( this->GetRootSurfaceContext() );
    m_ExecutionState  = ExecutionState_Preparing;
    m_ExecutionResult = m_StartCallback( &m_IsExecutionSuccess );
    m_ExecutionState  = ExecutionState_Completed;
}

void ProgressModalSceneBase::SetModalViewMessage( const std::string& message ) NN_NOEXCEPT
{
    m_ProgressModalView.SetMessage( message );
}

ProgressModalSceneBase::ExecutionState ProgressModalSceneBase::OnCompleteExecuteProgress( const nn::Result& result, bool isSuccess ) NN_NOEXCEPT
{
    m_CompletionCallback( m_ExecutionResult, isSuccess );

    m_ProgressModalView.ArrangeViewTable();
    m_ProgressModalView.DisplayMessage();
    m_ProgressModalView.DisplayCloseButton( this->GetRootSurfaceContext() );

    if ( !this->GetRootSurfaceContext()->IsModalViewRunning() )
    {
        // Timer が初めてシグナルされる前に ExecuteState_Completed に遷移すると StartModal していない
        this->StartModalForProgress( false );
    }

    if ( this->GetRootSurfaceContext()->IsModalTransparentStyle() )
    {
        // 透明なModalViewで事前表示されていたら非透明に戻す
        this->GetRootSurfaceContext()->SetOpaqueStyle();

        // ProgressModalViewもプロパティをenableに戻す
        m_ProgressModalView.enable( glv::Property::DrawBack | glv::Property::Controllable
            | glv::Property::DrawBorder | glv::Property::HitTest );
    }

    return ExecutionState_None;
}

} // ~namespace devmenu
