﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdlib>
#include <utility>

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>

namespace devmenu {

class Buffer
{
    NN_DISALLOW_COPY(Buffer);

private:
    void* pointer;
    size_t size;
public:
    Buffer() NN_NOEXCEPT
        : pointer(nullptr)
        , size(0u)
    {
    }
    explicit Buffer(size_t s) NN_NOEXCEPT
        : pointer(malloc(s))
        , size(s)
    {
        NN_ABORT_UNLESS(s);
    }
    Buffer(Buffer&& rhs) NN_NOEXCEPT
        : pointer(rhs.pointer)
        , size(rhs.size)
    {
        rhs.pointer = nullptr;
        rhs.size = 0u;
    }
    ~Buffer() NN_NOEXCEPT
    {
        if (pointer != nullptr)
        {
            free(pointer);
        }
    }
    Buffer& operator=(Buffer&& rhs) NN_NOEXCEPT
    {
        Buffer tmp(std::move(rhs));
        tmp.Swap(*this);
        return *this;
    }
    void Swap(Buffer& rhs) NN_NOEXCEPT
    {
        std::swap(pointer, rhs.pointer);
        std::swap(size, rhs.size);
    }
    void* GetPointer() const NN_NOEXCEPT
    {
        return pointer;
    }
    size_t GetSize() const NN_NOEXCEPT
    {
        return size;
    }
    NN_EXPLICIT_OPERATOR bool() const NN_NOEXCEPT
    {
        return pointer != nullptr;
    }
};

} // ~namespace devmenu
