﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include "DevMenu_CloudBackupScene.h"
#include "DevMenu_AccountView.h"
#include <glv/detail/glv_GfxTextApi.h>

namespace devmenu { namespace cloudbackup {

/**************************************
class TaskControl
**************************************/

/**
* コンストラクタ
*/
TaskControl::TaskControl(const glv::Rect& rect) NN_NOEXCEPT
    : glv::Table("<", 0.f, 12.f, rect)
    , m_Spacer(rect.width() - DefaultRightPadding, 1.f)
    , m_CheckBoxToStopAutonomousTask("Stop Autonomous Task")
    , m_CheckBoxToStopTransferTask("Stop Transfer Task")
{
    // データ更新検知の待機
    m_CheckBoxToStopAutonomousTask.SetCallback(
        [](const glv::Notification& notification)->void
        {
            // TODO:処理は後で追加予定
        }
        , this);

    // 新しいUL,DLタスクの実行
    m_CheckBoxToStopTransferTask.SetCallback(
        [](const glv::Notification& notification)->void
        {
            // TODO:処理は後で追加予定
        }
        , this);

    // 更新
    Refresh();
}

/**
* 更新
*/
void TaskControl::Refresh() NN_NOEXCEPT
{
    Clear();

    // データ更新検知の待機
    // TODO:値はnn::olscから取得予定
    m_CheckBoxToStopAutonomousTask.SetValue(false);
    // 新しいUL,DLタスクの実行
    // TODO:値はnn::olscから取得予定
    m_CheckBoxToStopTransferTask.SetValue(false);

    *this
        << m_CheckBoxToStopAutonomousTask
        << m_CheckBoxToStopTransferTask
        << m_Spacer;

    this->arrange();
}

/**
* クリア
*/
void TaskControl::Clear() NN_NOEXCEPT
{
    // 一旦remove
    m_CheckBoxToStopAutonomousTask.remove();
    m_CheckBoxToStopTransferTask.remove();
    m_Spacer.remove();
}

/**
* 最初にFocusを当てるViewを取得
*/
glv::View* TaskControl::GetFirstFocusTargetView() NN_NOEXCEPT
{
    return m_CheckBoxToStopAutonomousTask.GetButtonFocus();
}

/**
* 最後にFocusを当てるViewを取得
*/
glv::View* TaskControl::GetLastFocusTargetView() NN_NOEXCEPT
{
    return m_CheckBoxToStopTransferTask.GetButtonFocus();
}

/**
* フォーカス設定
*/
void TaskControl::SetFocusTransitionPath(FocusManager* pFocusManager, glv::View* pPreviousView, glv::View* pNextView) NN_NOEXCEPT
{
    // ボタンは上下のみ想定。自身との前後だけ設定する。処理は必要に応じて追加
    if (pFocusManager != nullptr)
    {
        if (pPreviousView != nullptr)
        {
            pFocusManager->AddFocusSwitch<FocusButtonUp>(GetFirstFocusTargetView(), pPreviousView);
            pFocusManager->AddFocusSwitch<FocusButtonDown>(pPreviousView, GetFirstFocusTargetView());
        }

        if (pNextView != nullptr)
        {
            pFocusManager->AddFocusSwitch<FocusButtonUp>(pNextView, GetLastFocusTargetView());
            pFocusManager->AddFocusSwitch<FocusButtonDown>(GetLastFocusTargetView(), pNextView);
        }
    }
}

/**************************************
class DeviceInfo
**************************************/
/**
* コンストラクタ
*/
DeviceInfo::DeviceInfo(const glv::Rect& rect) NN_NOEXCEPT
    : glv::Table("< > , < > , < -", 0.f, 12.f, rect)
    , m_Spacer(rect.width() - DefaultRightPadding, 1.f)
    , m_LabelShowTransferTaskList("UL, DL Task List", DefaultLabelSpec)
    , m_ButtonShowTransferTaskList("Show", nullptr, glv::Rect(DefaultButtonWidth, DefaultButtonHeight))
    , m_LabelShowErrorList("Error List", DefaultLabelSpec)
    , m_ButtonShowErrorList("Show", nullptr, glv::Rect(DefaultButtonWidth, DefaultButtonHeight))
{
    // 更新
    Refresh();
}

/**
* タスクリスト遷移ボタン押下コールバック設定
*/
void DeviceInfo::SetClickCallbackToShowTransferTaskList(const std::function<void()>& callback) NN_NOEXCEPT
{
    m_ButtonShowTransferTaskList.SetCallback(callback);
}

/**
* エラーリスト遷移ボタン押下コールバック設定
*/
void DeviceInfo::SetClickCallbackToShowErrorList(const std::function<void()>& callback) NN_NOEXCEPT
{
    m_ButtonShowErrorList.SetCallback(callback);
}

/**
* 更新
*/
void DeviceInfo::Refresh() NN_NOEXCEPT
{
    Clear();

    *this
        << m_LabelShowTransferTaskList
        << m_ButtonShowTransferTaskList
        << m_LabelShowErrorList
        << m_ButtonShowErrorList
        << m_Spacer;

    this->arrange();
}

/**
* クリア
*/
void DeviceInfo::Clear() NN_NOEXCEPT
{
    // 一旦remove
    m_LabelShowTransferTaskList.remove();
    m_ButtonShowTransferTaskList.remove();
    m_LabelShowErrorList.remove();
    m_ButtonShowErrorList.remove();
    m_Spacer.remove();
}

/**
* 最初にFocusを当てるViewを取得
*/
glv::View* DeviceInfo::GetFirstFocusTargetView() NN_NOEXCEPT
{
    return &m_ButtonShowTransferTaskList;
}

/**
* 最後にFocusを当てるViewを取得
*/
glv::View* DeviceInfo::GetLastFocusTargetView() NN_NOEXCEPT
{
    return &m_ButtonShowErrorList;
}

/**
* フォーカス設定
*/
void DeviceInfo::SetFocusTransitionPath(FocusManager* pFocusManager, glv::View* pPreviousView, glv::View* pNextView) NN_NOEXCEPT
{
    // ボタンは上下のみ想定。前後だけ設定する。処理は必要に応じて追加
    if (pFocusManager != nullptr)
    {
        if (pPreviousView != nullptr)
        {
            pFocusManager->AddFocusSwitch<FocusButtonUp>(GetFirstFocusTargetView(), pPreviousView);
            pFocusManager->AddFocusSwitch<FocusButtonDown>(pPreviousView, GetFirstFocusTargetView());
        }

        if (pNextView != nullptr)
        {
            pFocusManager->AddFocusSwitch<FocusButtonUp>(pNextView, GetLastFocusTargetView());
            pFocusManager->AddFocusSwitch<FocusButtonDown>(GetLastFocusTargetView(), pNextView);
        }
    }
}

/**************************************
class CloubBackupScene
**************************************/
CloudBackupScene::CloudBackupScene(ParentPageInterface* pParentPage, const glv::Rect& rect) NN_NOEXCEPT
    : Scene(rect)
    , m_pParentPage(pParentPage)
    , m_LayoutTable("<", 0.f, 0.f, glv::Rect(rect.width(), rect.height()))
    , m_pTaskControlHeader(nullptr)
    , m_TaskControl(glv::Rect(rect.w - DefaultRightPadding, rect.h))
    , m_pDeviceInfoHeader(nullptr)
    , m_DeviceInfo(glv::Rect(rect.w - DefaultRightPadding, rect.h))
    , m_ScrollContainer("<", glv::Rect(rect.w, rect.h))
    , m_pAccountInfoListHeader(nullptr)
    , m_AccountInfoList()
    , m_LabelNoItemAccount("No accounts", DefaultLabelSpec)
{
    // 自律処理制御
    m_pTaskControlHeader = CreateCategoryHeader("Task Control", DefaultLabelSpec,
        glv::Rect(rect.width() - DefaultRightPadding, DefaultCategoryHeaderHeight));

    // デバイス情報
    m_pDeviceInfoHeader = CreateCategoryHeader("Device Info", DefaultLabelSpec,
        glv::Rect(rect.width() - DefaultRightPadding, DefaultCategoryHeaderHeight));

    m_DeviceInfo.SetClickCallbackToShowTransferTaskList(
        [&]
        {
            m_pParentPage->SwitchScene(SceneType_TransferTaskList, false);
        });

    m_DeviceInfo.SetClickCallbackToShowErrorList(
        [&]
        {
            m_pParentPage->SwitchScene(SceneType_ErrorList, false);
        });

    // アカウント
    m_pAccountInfoListHeader = CreateCategoryHeader("Account Info", DefaultLabelSpec,
        glv::Rect(w - DefaultRightPadding, DefaultCategoryHeaderHeight));

    m_AccountInfoList.SetCallback(
        [&] (const nn::account::Uid& uid)
        {
            m_pParentPage->OpenAccountInfo(uid);
        });

    // 遅延更新
    RequireRefresh();

}

/**
* アカウントリストの更新
*/
void CloudBackupScene::RefreshAccountList(ScrollableBoxView* pScrollBox) NN_NOEXCEPT
{
    if (pScrollBox != nullptr)
    {
        // ユーザー情報カテゴリヘッダ
        *pScrollBox << m_pAccountInfoListHeader;

        // 2分割表示する
        m_AccountInfoList.Refresh();
        auto accountInfoList = m_AccountInfoList.GetList();
        const int accountCount = accountInfoList.size();

        if (accountCount > 0)
        {
            // TODO:要調整
            auto pTable = new glv::Table(
                "< <"
                , 12.f, 12.f
                , glv::Rect(pScrollBox->w - DefaultRightPadding, pScrollBox->h));

            for (auto pView : accountInfoList)
            {
                *pTable << pView;
            }

            pTable->fit(false);
            pTable->arrange();
            *pScrollBox << pTable;
            // scrollBoxのarrangeはしない
        }
        else
        {
            *pScrollBox << m_LabelNoItemAccount;
        }
    }
}

/**
* アカウントリストの除去
*/
void CloudBackupScene::RemoveAccountList() NN_NOEXCEPT
{
    m_pAccountInfoListHeader->remove();
    m_LabelNoItemAccount.remove();
    m_AccountInfoList.Clear();
}

/**
* クリア
*/
void  CloudBackupScene::Clear() NN_NOEXCEPT
{
    // フォーカスクリア
    auto& rootGlv = reinterpret_cast< glv::GLV& >(this->root());
    NN_ASSERT(glv::GLV::valid(&rootGlv));
    rootGlv.setFocus(nullptr);

    // 全て一旦removeしてコンテナをRefresh
    m_pTaskControlHeader->remove();
    m_TaskControl.Clear();
    m_TaskControl.remove();
    m_pDeviceInfoHeader->remove();
    m_DeviceInfo.Clear();
    m_DeviceInfo.remove();
    RemoveAccountList();
    m_ScrollContainer.remove();
    m_ScrollContainer.Refresh();
}

/**
* リフレッシュ
*/
void CloudBackupScene::Refresh() NN_NOEXCEPT
{
    Clear();
    ClearRefreshRequest();

    // 更新
    m_TaskControl.Refresh();
    m_DeviceInfo.Refresh();

    // ScrollBoxへは1つずつ入れる
    m_ScrollContainer << m_pTaskControlHeader;
    m_ScrollContainer << m_TaskControl;
    m_ScrollContainer << m_pDeviceInfoHeader;
    m_ScrollContainer << m_DeviceInfo;

    // アカウント情報
    RefreshAccountList(&m_ScrollContainer);

    m_ScrollContainer.ArrangeTable();

    *this << m_ScrollContainer;

    // フォーカス設定を再設定する
    RefreshFocusSetting();

    // アカウントのサイズが変わっている可能性があるので
    // フォーカスを一番上に戻す
    auto& rootGlv = reinterpret_cast< glv::GLV& >(this->root());
    NN_ASSERT(glv::GLV::valid(&rootGlv));
    rootGlv.setFocus(GetFirstFocusTargetView());// TODO暫定
}

/**
* 描画前コールバック
*/
void CloudBackupScene::OnLoopBeforeSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT
{
    if (IsRefreshRequired())
    {
        Refresh();
    }
}

/**
* アクティブ処理
*/
void CloudBackupScene::OnActivatePage() NN_NOEXCEPT
{
    Refresh();
}

/**
* シーンに入る際の遷移イベント
*/
void CloudBackupScene::OnEnterScene() NN_NOEXCEPT
{
    // スクロールを戻す
    m_ScrollContainer.ScrollToTop();

    // フォーカスを先頭に設定しなおす
    auto& g = reinterpret_cast<glv::GLV&>(this->root());
    NN_ASSERT(glv::GLV::valid(&g));
    g.setFocus(GetFirstFocusTargetView());
}


/**
* フォーカス設定を行う
*/
void CloudBackupScene::RefreshFocusSetting() NN_NOEXCEPT
{
    m_FocusManager.Clear();
    m_FocusManager.SetParentScroll(&m_ScrollContainer);

    // 現在必要最低限。必要に応じて処理を追加すること
    // アカウントリストの先頭を取得する
    glv::View* pAccountView = nullptr;
    auto accountInfoList = m_AccountInfoList.GetList();
    if (accountInfoList.size() > 0)
    {
        pAccountView = accountInfoList.at(0);
    }

    // デバイス制御フォーカス設定
    m_DeviceInfo.SetFocusTransitionPath(&m_FocusManager, m_TaskControl.GetLastFocusTargetView(), pAccountView);

    // 初期フォーカス位置設定
    SetFirstFocusTargetView(m_TaskControl.GetFirstFocusTargetView());
}

}} // ~namespace devmenu::cloudbackup, ~namespace devmenu
