﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "DevMenu_CloudBackupCommons.h"
#include "../Accounts/DevMenu_AccountsUiComponents.h"


namespace devmenu { namespace cloudbackup {

typedef std::function<void(const nn::account::Uid& uid)> SelectAccountCallback;

/**
* アカウントリストの行ビュー
*/
class AccountView : public glv::Button
{

public:

    /**
    * コンストラクタ
    */
    AccountView(const glv::Rect& rect, const SelectAccountCallback& clickCallback, const nn::account::Uid& uid ) NN_NOEXCEPT
        : glv::Button(rect, true)
        , m_Uid(uid)
        , m_Layout( ". . x," "< < <," "| | <", 0.f, 0.f, rect)
        , m_Spacer( rect.width() - 80.0f, 1)
        , m_SpacerX16(16,1)
        , m_pUserIcon(nullptr)
        , m_pLabelNickName(nullptr)
        , m_pLabelNsaId(nullptr)
        , m_Callback(clickCallback)
    {
        // ニックネーム
        glv::WideCharacterType nickname[64];
        accounts::GetNickname(nickname, sizeof(nickname) / sizeof(nickname[0]), uid);
        m_pLabelNickName = new glv::Label(nickname, glv::Label::Spec(glv::Place::TL, 0, 0, 22));

        // NSA ID
        char nsaIdString[64];
        int stringPos = 0;
        stringPos += nn::util::SNPrintf(&nsaIdString[stringPos], sizeof(nsaIdString) - stringPos, "NSA ID: ");
        auto nsaId = accounts::GetNetworkServiceAccountId(uid);
        if (nsaId)
        {
            stringPos += accounts::NsaIdToString(&nsaIdString[stringPos], sizeof(nsaIdString) - stringPos, nsaId);
        }
        else
        {
            stringPos += nn::util::SNPrintf(&nsaIdString[stringPos], sizeof(nsaIdString) - stringPos, "(not available)");
        }
        m_pLabelNsaId = new glv::Label(nsaIdString, SmallLabelSpec);

        // アイコン
        Buffer buffer = accounts::LoadUserIcon(uid);
        m_pUserIcon = new accounts::UserIcon(
            64.f, 64.f, 8.f, 8.f
        );

        if (buffer)
        {
            m_pUserIcon->SetTexture(buffer);
        }
        else
        {
            m_pUserIcon->ClearTexture();
        }

        // セット
        m_Layout
            << m_Spacer
            << m_pUserIcon
            << m_SpacerX16
            << m_pLabelNickName
            << m_pLabelNsaId;

        *this << m_Layout;

        m_Layout.arrange();
        m_Layout.fit(false);
        m_Layout.disable(glv::Property::HitTest);

        changePadClickDetectableButtons(glv::BasicPadEventType::Button::Ok::Mask);
        changePadClickDetectableButtons(glv::DebugPadEventType::Button::Ok::Mask);

        attach([](const glv::Notification& n)->void
        {
            n.receiver< AccountView >()->OnClick();
        }, glv::Update::Clicked, this);
    }

private:

    /**
    * クリックイベント
    */
    void OnClick() NN_NOEXCEPT
    {
        if (m_Callback != nullptr)
        {
            m_Callback(m_Uid);
        }
    }

    // Uidだけ持っておく
    nn::account::Uid m_Uid;

    // レイアウト
    glv::Table m_Layout;

    // 上余白
    Spacer m_Spacer;

    // アイコンとニックネームとの余白
    Spacer m_SpacerX16;

    // アイコン
    accounts::UserIcon* m_pUserIcon;

    // ニックネーム
    glv::Label* m_pLabelNickName;

    // Nintendo Service Account ID
    glv::Label* m_pLabelNsaId;

    // 選択コールバック
    const SelectAccountCallback& m_Callback;
};

/**
* AccountViewのリスト
*/
class AccountInfoList
{
public:

    /**
    * コンストラクタ
    */
    AccountInfoList(const SelectAccountCallback& callback = SelectAccountCallback()) NN_NOEXCEPT
        : m_Callback(callback)
    {
        // NOOP
    }

    /**
    * 行選択コールバック設定
    */
    void SetCallback(const SelectAccountCallback& callback) NN_NOEXCEPT
    {
        m_Callback = callback;
    }

    /**
    * 更新
    */
    void Refresh() NN_NOEXCEPT
    {
        Clear();

        // アカウント情報
        // ユーザーリストの取得
        nn::account::Uid userIds[nn::account::UserCountMax];
        int count;
        nn::account::ListAllUsers(&count, userIds, sizeof(userIds) / sizeof(userIds[0]));

        if (count > 0)
        {
            for (int i=0; i<count; i++)
            {
                auto pAccountView = new AccountView(
                    glv::Rect(500.f, 80.f),// TODO:要調整
                    m_Callback, userIds[i]);

                m_AccountViewList.push_back(pAccountView);
            }
        }
    }

    /**
    * クリア
    */
    void Clear() NN_NOEXCEPT
    {
        for (auto pView : m_AccountViewList)
        {
            if (pView != nullptr)
            {
                pView->remove();
                delete pView;
                pView = nullptr;
            }
        }
        m_AccountViewList.clear();
    }

    /**
    * クリア
    */
    const std::vector<AccountView*>& GetList() const NN_NOEXCEPT
    {
        return m_AccountViewList;
    }

private:
    // 行選択コールバック
    SelectAccountCallback m_Callback;

    // 行Viewリスト
    std::vector< AccountView* > m_AccountViewList;
};

}} // ~namespace devmenu::cloudbackup, ~namespace devmenu
