﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "DevMenu_AccountInfoScene.h"
#include <nn/util/util_Country.h>
#include <nn/account/account_Api.h>
#include <nn/account/account_Result.h>
#include <nn/time/time_TimeZoneApi.h>
#include "../Accounts/DevMenu_AccountsSdkHelper.h"

namespace devmenu { namespace cloudbackup {

/**************************************
class AccountInfoScene::AccountInfoHeader
TODO:後でファイル分ける
**************************************/
/**
* コンストラクタ
*/
AccountInfoScene::AccountInfoHeader::AccountInfoHeader(const glv::Rect& rect) NN_NOEXCEPT
    : glv::Table("<", 8.f, 12.f)
    , m_Spacer(rect.w - DefaultRightPadding, 4.f)
    , m_pLabelNickname(nullptr)
    , m_pLabelNsaId(nullptr)
{
    disable(glv::Property::Controllable | glv::Property::HitTest);
    this->enable(glv::Property::DrawBack | glv::Property::DrawBorder);
}

/**
* クリア
*/
void AccountInfoScene::AccountInfoHeader::Clear() NN_NOEXCEPT
{
    if (m_pLabelNickname != nullptr)
    {
        m_pLabelNickname->remove();
        delete m_pLabelNickname;
        m_pLabelNickname = nullptr;
    }

    if (m_pLabelNsaId != nullptr)
    {
        m_pLabelNsaId->remove();
        delete m_pLabelNsaId;
        m_pLabelNsaId = nullptr;
    }

    m_Spacer.remove();
}

/**
* 更新
*/
void AccountInfoScene::AccountInfoHeader::Refresh(const nn::account::Uid& uid) NN_NOEXCEPT
{
    // TODO:AccountSceneDetailから流用。処理を修正する際は一緒に修正する必要がある
    Clear();

    //
    if (uid != nn::account::InvalidUid)
    {
        // ニックネーム
        glv::WideCharacterType nickname[64];
        accounts::GetNickname(nickname, sizeof(nickname) / sizeof(nickname[0]), uid);
        m_pLabelNickname = new glv::Label(nickname, DefaultLabelSpec);

        // NSA ID
        char nsaIdString[64];
        int stringNsaIdPos = 0;
        stringNsaIdPos += nn::util::SNPrintf(&nsaIdString[stringNsaIdPos], sizeof(nsaIdString) - stringNsaIdPos, "NSA ID: ");
        auto nsaId = accounts::GetNetworkServiceAccountId(uid);
        if (nsaId)
        {
            stringNsaIdPos += accounts::NsaIdToString(&nsaIdString[stringNsaIdPos], sizeof(nsaIdString) - stringNsaIdPos, nsaId);
        }
        else
        {
            stringNsaIdPos += nn::util::SNPrintf(&nsaIdString[stringNsaIdPos], sizeof(nsaIdString) - stringNsaIdPos, "(not available)");
        }
        m_pLabelNsaId = new glv::Label(nsaIdString, SmallLabelSpec);

        //ニックネーム、スペース、NSA IDの順にTableに追加
        *this
            << m_pLabelNickname
            << m_Spacer// 間隔を空けるために間に挟む
            << m_pLabelNsaId;

        this->fit(false);
        this->arrange();
    }
}

/**************************************
class AccountInfoScene::AccountInfoDetail
TODO:後でファイル分ける
**************************************/
AccountInfoScene::AccountInfoDetail::AccountInfoDetail(const glv::Rect& rect) NN_NOEXCEPT
    : glv::Table("< - , < <", 0.f, 0.f)
    , m_Spacer(rect.w - DefaultRightPadding, 1)
    , m_UserIcon(128.f, 128.f, 0.f, 0.f)
    , m_pLabelStatus(nullptr)
{
    *this << m_Spacer;
    disable(glv::Property::Controllable | glv::Property::DrawBack | glv::Property::DrawBorder | glv::Property::HitTest);
}

/**
* クリア
*/
void AccountInfoScene::AccountInfoDetail::Clear() NN_NOEXCEPT
{
    // 一旦外す
    m_UserIcon.remove();

    if (m_pLabelStatus != nullptr)
    {
        m_pLabelStatus->remove();
        delete m_pLabelStatus;
        m_pLabelStatus = nullptr;
    }
}

/**
* 更新
*/
void AccountInfoScene::AccountInfoDetail::Refresh(const nn::account::Uid& uid) NN_NOEXCEPT
{
    // TODO:AccountSceneDetailから流用。処理を修正する際は一緒に修正する必要がある
    Clear();

    if (uid != nn::account::InvalidUid)
    {
        // アイコンの取得
        Buffer buffer = accounts::LoadUserIcon(uid);
        if (buffer)
        {
            m_UserIcon.SetTexture(buffer);
        }
        else
        {
            m_UserIcon.ClearTexture();
        }

        nn::account::NetworkServiceAccountManager manager;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountManager(&manager, uid));
        auto availability = manager.CheckNetworkServiceAccountAvailability();
        NN_ABORT_UNLESS(availability.IsSuccess() || nn::account::ResultNetworkServiceAccountUnavailable::Includes(availability));

        Buffer accountInfoBuffer(nn::account::RequiredBufferSizeForCachedNintendoAccountInfo);
        nn::account::CachedNintendoAccountInfoForSystemService info;
        auto resultGetAccountInfo = manager.LoadCachedNintendoAccountInfo(&info, accountInfoBuffer.GetPointer(), accountInfoBuffer.GetSize());
        auto isNaAvailable = resultGetAccountInfo.IsSuccess();
        {
            char stringBuffer[512];

            int stringPos = 0;

            stringPos += nn::util::SNPrintf(&stringBuffer[stringPos], sizeof(stringBuffer) - stringPos, "UID: ");
            stringPos += accounts::UidToString(&stringBuffer[stringPos], sizeof(stringBuffer) - stringPos, uid);

            stringPos += nn::util::SNPrintf(&stringBuffer[stringPos], sizeof(stringBuffer) - stringPos, "\nNSA Status: ");
            if (availability.IsSuccess())
            {
                stringPos += nn::util::SNPrintf(&stringBuffer[stringPos], sizeof(stringBuffer) - stringPos, "Available");
            }
            else
            {
                stringPos += nn::util::SNPrintf(&stringBuffer[stringPos], sizeof(stringBuffer) - stringPos, "%04d", availability.GetDescription());
            }

            stringPos += nn::util::SNPrintf(&stringBuffer[stringPos], sizeof(stringBuffer) - stringPos, "\nNA: ");
            if (isNaAvailable)
            {
                size_t sz;
                const auto* pCountryStr = info.GetCountry(&sz);
                nn::util::Country country;
                auto isCountryAvailable = nn::util::GetCountryFromIso31661Alpha2(&country, pCountryStr, sz);
                stringPos += nn::util::SNPrintf(
                    &stringBuffer[stringPos], sizeof(stringBuffer) - stringPos,
                    "\"%s\"(%d,%c,%c,%c)",
                    info.GetScreenName(&sz),
                    isCountryAvailable ? country : 0,
                    info.GetAnalyticsOptedInFlag() ? 'A' : '-',
                    info.IsChild() ? 'C' : '-',
                    info.IsLinkedWithNintendoNetwork() ? 'N' : '-');
            }
            else
            {
                stringPos += nn::util::SNPrintf(&stringBuffer[stringPos], sizeof(stringBuffer) - stringPos, "(not available)");
            }

            nn::account::CachedNetworkServiceLicenseInfo licenseInfo;
            auto isLicenseInfoCached = manager.GetCachedNetworkServiceLicenseInfo(&licenseInfo).IsSuccess();

            stringPos += nn::util::SNPrintf(&stringBuffer[stringPos], sizeof(stringBuffer) - stringPos, "\nLicense: ");
            if (isLicenseInfoCached)
            {
                switch (licenseInfo.license)
                {
                case nn::account::NetworkServiceLicense_None:
                    stringPos += nn::util::SNPrintf(&stringBuffer[stringPos], sizeof(stringBuffer) - stringPos, "No license");
                    break;
                case nn::account::NetworkServiceLicense_Common:
                    {
                        nn::time::CalendarTime cal;
                        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::ToCalendarTime(&cal, nullptr, licenseInfo.expiration));
                        stringPos += nn::util::SNPrintf(
                            &stringBuffer[stringPos], sizeof(stringBuffer) - stringPos,
                            "Expires at %04d-%02d-%02d %02d:%02d",
                            cal.year, cal.month, cal.day, cal.hour, cal.minute);
                    }
                    break;
                default:
                    NN_UNEXPECTED_DEFAULT;
                }
            }
            else
            {
                stringPos += nn::util::SNPrintf(&stringBuffer[stringPos], sizeof(stringBuffer) - stringPos, "(not available)");
            }

            m_pLabelStatus = new glv::Label(stringBuffer, SmallLabelSpec);

            // アイコン、状態の順にTableに追加
            *this
                << m_UserIcon
                << m_pLabelStatus;

            this->fit(false);
            this->arrange();
        }
    }
} // NOLINT(readability/fn_size)

/**************************************
class AccountInfoScene::ServerControl
TODO:後でファイル分ける
**************************************/
AccountInfoScene::ServerControl::ServerControl(const glv::Rect& rect) NN_NOEXCEPT
    : glv::Table("< - , < > , < > , . > , < -", 0.f, 12.f)
    , m_Spacer(rect.w - DefaultRightPadding, 1.f)
    , m_CheckBoxAutoUpload("Auto Upload")
    , m_LabelScanAutoUpload("Scan Auto Upload", DefaultLabelSpec)
    , m_ButtonScanAutoUpload("Start", nullptr, glv::Rect(DefaultButtonWidth, DefaultButtonHeight))
    , m_LabelServerCache("Server Cache", DefaultLabelSpec)
    , m_ButtonUpdateServerCache("Update", nullptr, glv::Rect(DefaultButtonWidth, DefaultButtonHeight))
    , m_ButtonDeleteServerCache("Delete", nullptr, glv::Rect(DefaultButtonWidth, DefaultButtonHeight))
{
    // 自動UL一括ON/OFF
    m_CheckBoxAutoUpload.SetCallback(
        [](const glv::Notification& notification)->void
        {
            // TODO:処理は後で追加予定
        }
        , this);

    // 自動ULスキャン
    m_ButtonScanAutoUpload.SetCallback(
        [&]
        {
            // TODO:処理を後で記載
        }
        );

    // サーバー情報キャッシュの一括更新
    m_ButtonUpdateServerCache.SetCallback(
        [&]
        {
            // TODO:処理を後で記載
        }
        );

    // サーバー情報キャッシュの一括削除
    m_ButtonDeleteServerCache.SetCallback(
        [&]
        {
            // TODO:処理を後で記載
        }
        );
}

/**
* クリア
*/
void AccountInfoScene::ServerControl::Clear() NN_NOEXCEPT
{
    // 一旦removeする
    m_CheckBoxAutoUpload.remove();
    m_LabelScanAutoUpload.remove();
    m_ButtonScanAutoUpload.remove();
    m_LabelServerCache.remove();
    m_ButtonUpdateServerCache.remove();
    m_ButtonDeleteServerCache.remove();
    m_Spacer.remove();
}


// サーバー処理Viewの生成。
// TODO:修正予定
void AccountInfoScene::ServerControl::Refresh(const nn::account::Uid& uid) NN_NOEXCEPT
{
    Clear();

    // TODO:API呼び出しで値を取得予定
    m_CheckBoxAutoUpload.SetValue(false);

    *this
        << m_CheckBoxAutoUpload
        << m_LabelScanAutoUpload
        << m_ButtonScanAutoUpload
        << m_LabelServerCache
        << m_ButtonUpdateServerCache
        << m_ButtonDeleteServerCache
        << m_Spacer;

    this->fit(false);
    this->arrange();
}

/**
* 最初にFocusを当てるViewを取得
*/
glv::View* AccountInfoScene::ServerControl::GetFirstFocusTargetView() NN_NOEXCEPT
{
    return m_CheckBoxAutoUpload.GetButtonFocus();
}

/**
* 最後にFocusを当てるViewを取得
*/
glv::View* AccountInfoScene::ServerControl::GetLastFocusTargetView() NN_NOEXCEPT
{
    return &m_ButtonDeleteServerCache;
}

/**
* フォーカス設定
*/
void AccountInfoScene::ServerControl::SetFocusTransitionPath(FocusManager* pFocusManager, glv::View* pPreviousView, glv::View* pNextView) NN_NOEXCEPT
{
    if (pFocusManager != nullptr)
    {
        // 上下だけなので配列で
        std::vector<glv::View*> viewList =
        {
            pPreviousView,
            m_CheckBoxAutoUpload.GetButtonFocus(),
            &m_ButtonScanAutoUpload,
            &m_ButtonUpdateServerCache,
            &m_ButtonDeleteServerCache,
            pNextView
        };

        // 方向キー下ボタン
        {
            glv::View* pPreviousFocusedView = nullptr;
            for (auto pNextFocusedView : viewList)
            {
                if (pNextFocusedView != nullptr && pPreviousFocusedView != nullptr)
                {
                    pFocusManager->AddFocusSwitch<FocusButtonDown>(pPreviousFocusedView, pNextFocusedView);
                }
                pPreviousFocusedView = pNextFocusedView;
            }
        }

        // 方向キー上ボタン
        {
            glv::View* pPreviousFocusedView = nullptr;
            // TODO:範囲forで逆順にしたい
            for (std::vector<glv::View*>::reverse_iterator iterator = viewList.rbegin(); iterator != viewList.rend(); iterator++)
            {
                glv::View* pNextFocusedView = *iterator;
                if (pNextFocusedView != nullptr && pPreviousFocusedView != nullptr)
                {
                    pFocusManager->AddFocusSwitch<FocusButtonUp>(pPreviousFocusedView, pNextFocusedView);
                }
                pPreviousFocusedView = pNextFocusedView;
            }
        }
    }
}

/**************************************
class AccountInfoScene
**************************************/
AccountInfoScene::AccountInfoScene(ParentPageInterface* pParentPage, const glv::Rect& rect) NN_NOEXCEPT
    : Scene(rect)
    , m_pParentPage(pParentPage)
    , m_pButtonBack(nullptr)
    , m_Uid(nn::account::InvalidUid)
    , m_AccountInfoHeader(glv::Rect(rect.w, rect.h))
    , m_AccountInfoDetail(glv::Rect(rect.w, rect.h))
    , m_pServerControlHeader(nullptr)
    , m_ServerControl(glv::Rect(rect.w - DefaultRightPadding, rect.h))
    , m_pSaveDataHeader(nullptr)
    , m_SaveDataList()
    , m_LabelNoItemForSaveData("No save data is created.", DefaultLabelSpec)
    , m_LayoutTable("<", 0.f, 0.f, rect)
    , m_ScrollBox("<", glv::Rect(rect.w, 0.f))
{
    // 戻る
    m_pButtonBack = new Button("< Back"
        ,[&]
        {
            m_pParentPage->SwitchScene(SceneType_CloudBackupRoot, true);
        }
        , glv::Rect(DefaultButtonWidth, DefaultButtonHeight));

    // サーバーコントロール
    m_pServerControlHeader = CreateCategoryHeader("Server Control", DefaultLabelSpec,
        glv::Rect(w - DefaultRightPadding, DefaultCategoryHeaderHeight));

    // セーブデータ
    m_pSaveDataHeader = CreateCategoryHeader("Save Data Information", DefaultLabelSpec,
        glv::Rect(w - DefaultRightPadding, DefaultCategoryHeaderHeight));

    m_SaveDataList.SetCallback(
        [&](const int id)
        {
            m_pParentPage->SwitchScene(SceneType_ApplicationInfo, false);
        }
        );

    m_ScrollBox.enable(glv::Property::KeepWithinParent);

    m_LayoutTable
        << m_pButtonBack;

    m_LayoutTable.arrange();

    *this << m_LayoutTable;
}

/**
* セーブデータリストの更新
*/
void AccountInfoScene::RefreshSaveDataList(ScrollableBoxView& scrollBox) NN_NOEXCEPT
{
    // ユーザー情報カテゴリヘッダ
    scrollBox << m_pSaveDataHeader;

    // 3分割表示
    m_SaveDataList.Refresh(m_Uid);
    auto saveDataList = m_SaveDataList.GetList();
    const int saveDataCount = saveDataList.size();
    if (saveDataCount > 0)
    {
        // TODO:要調整
        auto pTable = new glv::Table(
            "< < <"
            , 12.f, 12.f
            , glv::Rect(scrollBox.w - DefaultRightPadding, scrollBox.h));

        for (auto pView : saveDataList)
        {
            *pTable << pView;
        }

        pTable->fit(false);
        pTable->arrange();
        scrollBox << pTable;
        // scrollBoxのarrangeはしない
    }
    else
    {
        scrollBox << m_LabelNoItemForSaveData;
    }
}

/**
* セーブデータリストの除去
*/
void AccountInfoScene::RemoveSaveDataList() NN_NOEXCEPT
{
    m_pSaveDataHeader->remove();
    m_LabelNoItemForSaveData.remove();
    m_SaveDataList.Clear();
}

/**
* フォーカス設定
*/
void AccountInfoScene::RefreshFocusSetting() NN_NOEXCEPT
{
    m_FocusManager.Clear();
    m_FocusManager.SetParentScroll(&m_ScrollBox);

    // 上からバックボタン、ServerControl、SaveDataInformationの1列目　の順に設定する
    auto saveDataViewList = m_SaveDataList.GetList();
    const auto saveDataCount = saveDataViewList.size();

    // save data list の先頭を取得
    glv::View* pFirstFocusedViewForSaveDataList = nullptr;
    if (saveDataCount > 0)
    {
        pFirstFocusedViewForSaveDataList = saveDataViewList.at(0);
    }

    // バックボタンのフォーカス設定はサーバー制御の処理で行えるのでスキップ

    // サーバー制御のフォーカス設定
    m_ServerControl.SetFocusTransitionPath(&m_FocusManager, m_pButtonBack, pFirstFocusedViewForSaveDataList);

    // save dataのフォーカス設定。上ボタンのみ
    for (size_t i = 0; i < saveDataCount || i < 3; i++)
    {
        auto pView = saveDataViewList.at(0);
        m_FocusManager.AddFocusSwitch<FocusButtonUp>(pView, m_ServerControl.GetLastFocusTargetView());
    }

    // 先頭フォーカス設定
    SetFirstFocusTargetView(m_pButtonBack);
}

/**
* クリア
*/
void AccountInfoScene::Clear() NN_NOEXCEPT
{
    // フォーカスクリア
    auto& g = reinterpret_cast< glv::GLV& >(this->root());
    NN_ASSERT(glv::GLV::valid(&g));
    g.setFocus(nullptr);

    // 一旦外す
    m_pButtonBack->remove();
    m_AccountInfoHeader.remove();
    m_AccountInfoDetail.remove();
    m_pServerControlHeader->remove();
    m_ServerControl.remove();
    RemoveSaveDataList();
    m_ScrollBox.remove();
    m_ScrollBox.Refresh();
}

/**
* リフレッシュ
*/
void AccountInfoScene::Refresh() NN_NOEXCEPT
{
    Clear();
    ClearRefreshRequest();

    // アカウント情報
    m_AccountInfoHeader.Refresh(m_Uid);
    m_AccountInfoDetail.Refresh(m_Uid);

    // サーバー処理
    m_ServerControl.Refresh(m_Uid);

    // ScrollBoxへは1つずつ入れる
    m_ScrollBox << m_AccountInfoDetail;
    m_ScrollBox << m_pServerControlHeader;
    m_ScrollBox << m_ServerControl;

    // セーブデータリスト
    RefreshSaveDataList(m_ScrollBox);

    m_ScrollBox.SetRectAndUpdateInnerSize(w, h - (m_pButtonBack->h + m_AccountInfoHeader.h));// TODO:縦幅要調整
    m_ScrollBox.ArrangeTable();

    m_LayoutTable
        << m_pButtonBack
        << m_AccountInfoHeader
        << m_ScrollBox;

    m_LayoutTable.arrange();

    // フォーカス設定を再設定する
    RefreshFocusSetting();

    auto& g = reinterpret_cast<glv::GLV&>(this->root());
    NN_ASSERT(glv::GLV::valid(&g));
    g.setFocus(GetFirstFocusTargetView());
}

/**
* シーンに入る際の遷移イベント
*/
void AccountInfoScene::OnEnterScene() NN_NOEXCEPT
{
    // スクロールを戻す
    m_ScrollBox.ScrollToTop();

    // フォーカスを先頭に設定しなおす
    auto& g = reinterpret_cast<glv::GLV&>(this->root());
    NN_ASSERT(glv::GLV::valid(&g));
    g.setFocus(GetFirstFocusTargetView());
}

/**
* アカウントUID設定
*/
void AccountInfoScene::SetAccountUid(const nn::account::Uid& uid) NN_NOEXCEPT
{
    m_Uid = uid;
}

/**
* 描画前コールバック
*/
void AccountInfoScene::OnLoopBeforeSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT
{
    if (IsRefreshRequired())
    {
        Refresh();
    }
}

/**
* アクティブ処理
*/
void AccountInfoScene::OnActivatePage() NN_NOEXCEPT
{
    // アカウントが存在する場合はRefresh、存在しない場合はルートに戻る
    bool isExisting = false;
    auto result = nn::account::GetUserExistence(&isExisting, m_Uid);
    if (result.IsSuccess() == true && isExisting == true)
    {
        Refresh();
    }
    else
    {
        m_pParentPage->SwitchScene(SceneType_CloudBackupRoot, true);
    }
}

}} // ~namespace devmenu::cloudbackup, ~namespace devmenu
