﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/account/account_SelectorForSystemServices.h>

#include "DevMenu_AccountsUiComponents.h"
#include "DevMenu_AccountsSdkHelper.h"
#include "DevMenu_AccountsSelector.h"
#include "../DevMenu_ModalView.h"
#include "../Launcher/DevMenu_Launcher.h"

namespace devmenu { namespace accounts {

UserAccountSelector::UserAccountSelector(
    std::function<void(nn::account::Uid uid)> selectCallback,
    std::function<void()> cancelCallback,
    bool requiredWithNetworkServiceAccountAvailable,
    bool enableCancel,
    const char* headerText
) NN_NOEXCEPT
    : ModalView(glv::Rect(720, 500))
    , m_Placer(*this, glv::Direction::S, glv::Place::TC, width() / 2.0, 15.0f, 12.0f)
    , m_Message(headerText, glv::Label::Spec(glv::Place::TL, 0, 0, CommonValue::InitialFontSize))
    , m_Invalid("[No valid users]", glv::Label::Spec(glv::Place::TL, 0, 0, Default::FontSizeSmall))
    , m_AddUserAccount("Add a user account and launch", [this, selectCallback] {this->ExitModal(); selectCallback(CreateUserAccount());}, glv::Rect(480, 36), glv::Place::TL)
    , m_Cancel("Cancel", [this, cancelCallback] {this->ExitModal(); cancelCallback();}, glv::Rect(240, 36), glv::Place::TL)
    , m_IsAutoSelect(false)
{
    for (int i = 0; i < nn::account::UserCountMax; ++ i)
    {
        m_Buttons[i] = nullptr;
    }

    // 自動選択できる場合はダイアログを省略する
    nn::account::Uid autoSelect;
    nn::account::ImplicitUserSelectionSettings implicitArg = nn::account::DefaultImplicitUserSelectionSettings;
    implicitArg.isNetworkServiceAccountRequired = requiredWithNetworkServiceAccountAvailable;
    if (nn::account::TrySelectUserImplicitly(&autoSelect, implicitArg))
    {
        selectCallback(autoSelect);
        m_IsAutoSelect = true;
        return;
    }

    m_Placer << m_Message;

    // ユーザーリストの生成
    nn::account::Uid userIds[nn::account::UserCountMax];
    int count;
    nn::account::ListAllUsers(&count, userIds, sizeof(userIds) / sizeof(userIds[0]));
    for (int i = 0; i < count; ++ i)
    {
        const auto& uid = userIds[i];
        NN_SDK_ASSERT(uid);

        bool valid = true;
        if (requiredWithNetworkServiceAccountAvailable)
        {
            nn::account::NetworkServiceAccountManager manager;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountManager(&manager, uid));
            NN_ABORT_UNLESS_RESULT_SUCCESS(manager.IsNetworkServiceAccountAvailable(&valid));
        }

        UserButton* pButton = nullptr;
        if (valid)
        {
            pButton = new UserButton(uid, [this, selectCallback, uid] { this->ExitModal(); selectCallback(uid); }, glv::Rect(640, 36));
            pButton->SetVailidity(true);
        }
        else
        {
            pButton = new UserButton(uid, [] { NN_SDK_ASSERT(false, "Unreachable"); }, glv::Rect(640, 36));
            pButton->SetVailidity(false);
        }
        NN_SDK_ASSERT(pButton != nullptr);
        m_Buttons[i] = pButton;

        m_Placer << pButton;
    }

    // 有効なユーザーアカウントが存在しない場合のみ表示
    if (count == 0)
    {
        m_Placer << m_Invalid;
    }

    // ユーザーアカウントが最大数より小さい場合のみ表示
    if (count < nn::account::UserCountMax)
    {
        m_Placer << m_AddUserAccount;
    }

    // キャンセル有効の場合のみ表示
    if (enableCancel)
    {
        m_Placer << m_Cancel;
    }
}

UserAccountSelector::~UserAccountSelector() NN_NOEXCEPT
{
    for (auto* p : m_Buttons)
    {
        if (p != nullptr)
        {
            delete p;
        }
    }
}

}} // ~namespace devmenu::accounts, ~namespace devmenu
