﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/swkbd/swkbd_Result.h>
#include <nn/nn_Common.h>

#include "DevMenu_AccountsConfig.h"
#include "DevMenu_AccountsSdkHelper.h"
#include "DevMenu_AccountsUiComponents.h"
#include "../DevMenu_RootSurface.h"

namespace devmenu { namespace accounts {

template <typename T>
class SelectorScene
    : public devmenu::Scene
{
public:
    class SelectItem
    {
    private:
        glv::WideString m_MainLabelString;
        glv::WideString m_SubLabelString;
        T m_Data;
    public:
        SelectItem(const glv::WideString& mainLabelString, const glv::WideString& subLabelString, T data) NN_NOEXCEPT
            : m_MainLabelString(mainLabelString)
            , m_SubLabelString(subLabelString)
            , m_Data(data)
        {
        }

        SelectItem(const glv::WideString& mainLabel, T data) NN_NOEXCEPT
            : m_MainLabelString(mainLabel)
            , m_SubLabelString()
            , m_Data(data)
        {
        }

        const glv::WideString& GetMainLabelStringRef() const NN_NOEXCEPT
        {
            return m_MainLabelString;
        }

        const glv::WideString& GetSubLabelStringRef() const NN_NOEXCEPT
        {
            return m_SubLabelString;
        }

        const T& GetDataRef() const NN_NOEXCEPT
        {
            return m_Data;
        }

        bool HasSubLabel() const NN_NOEXCEPT
        {
            return m_SubLabelString.length() > 0;
        }
    };

protected:
    void PopPreviousScene() NN_NOEXCEPT;

private:
    class SelectButton
        : public glv::Button
    {
    private:
        SelectItem m_Item;
        glv::Label* m_MainLabel;
        glv::Label* m_SubLabel;
        SelectorScene& m_Parent;

    public:
        SelectButton(SelectorScene& parent, const SelectItem& item, const glv::Rect& r, glv::Place::t anchor = glv::Place::TL) NN_NOEXCEPT;
        ~SelectButton() NN_NOEXCEPT;
    };

    class BackButton
        : public Button
    {
    private:
        const AbstractOperators& m_Op;
        SelectorScene& m_Parent;

    public:
        BackButton(const AbstractOperators& op, SelectorScene& parent, const glv::Rect& r, glv::Place::t anchor = glv::Place::TL) NN_NOEXCEPT;
    };

    void Close() NN_NOEXCEPT;

    const AbstractOperators& m_Op;

    glv::Label m_Header;
    glv::Label m_Footer;
    ScrollableBoxView m_Container;

    std::vector<std::unique_ptr<SelectButton>> m_SelectButtons;

    BackButton m_BackButton;
    Scene* m_pPreviousScene;

    std::function<void(T arg)> m_Callback;

public:
    SelectorScene(AbstractOperators& op, glv::Rect rect, Scene* pParent, const glv::WideString& sceneTitle, const std::vector<SelectItem>& items, std::function<void(T arg)> callback) NN_NOEXCEPT;
    ~SelectorScene() NN_NOEXCEPT;
    glv::View* GetPrimaryView() NN_NOEXCEPT;
    virtual void OnSelected(T data) NN_NOEXCEPT;
    void Clear() NN_NOEXCEPT;
    void CloseScene() NN_NOEXCEPT;
    virtual void Refresh() NN_NOEXCEPT final NN_OVERRIDE;
};

template <typename T>
SelectorScene<T>::BackButton::BackButton(const AbstractOperators& op, SelectorScene& parent, const glv::Rect& r, glv::Place::t anchor) NN_NOEXCEPT
    : Button("Back", [&] { m_Op.PopScene(&m_Parent, m_Parent.m_pPreviousScene); }, r, anchor)
    , m_Op(op)
    , m_Parent(parent)
{
}

template <typename T>
SelectorScene<T>::SelectButton::SelectButton(SelectorScene& parent, const SelectItem& item, const glv::Rect& r, glv::Place::t anchor) NN_NOEXCEPT
    : glv::Button(r, true)
    , m_Item(item)
    , m_Parent(parent)
{
    this->anchor(anchor);

    if( m_Item.HasSubLabel() > 0 )
    {
        m_MainLabel = new glv::Label(item.GetMainLabelStringRef(), glv::Label::Spec(glv::Place::TC, 0.0f, 8.0f, Default::FontSize));
        m_SubLabel = new glv::Label(item.GetSubLabelStringRef(), glv::Label::Spec(glv::Place::BC, 0.0f, -4.0f, Default::FontSizeSmall));
        *this << m_MainLabel;
        *this << m_SubLabel;
    }
    else
    {
        m_MainLabel = new glv::Label(item.GetMainLabelStringRef(), glv::Label::Spec(glv::Place::CC, 0.0f, 0.0f, Default::FontSize));
        *this << m_MainLabel;
        m_SubLabel = nullptr;
    }
    changePadClickDetectableButtons(glv::BasicPadEventType::Button::Ok::Mask);
    changePadClickDetectableButtons(glv::DebugPadEventType::Button::Ok::Mask);
    attach([](const glv::Notification& n)->void {
        auto button = n.receiver< SelectButton >();
        button->m_Parent.OnSelected(button->m_Item.GetDataRef());
    }, glv::Update::Clicked, this);
}

template <typename T>
SelectorScene<T>::SelectButton::~SelectButton() NN_NOEXCEPT
{
    delete m_MainLabel;
    if ( nullptr != m_SubLabel )
    {
        delete m_SubLabel;
    }
}

template <typename T>
SelectorScene<T>::SelectorScene(AbstractOperators& op, glv::Rect rect, Scene* pParent, const glv::WideString& sceneTitle, const std::vector<SelectItem>& items, std::function<void(T arg)> callback) NN_NOEXCEPT
    : Scene(rect)
    , m_Op(op)
    , m_Header(
        sceneTitle,
        glv::Label::Spec(glv::Place::TC, 0.0f, 0.0f, CommonValue::InitialFontSize))
    , m_Footer(
        GLV_TEXT_API_WIDE_STRING("↑↓:Focus  A:Select"),
        glv::Label::Spec(glv::Place::BR, 0.0f, 0.0f, CommonValue::InitialFontSize))
    , m_Container(
        "x", glv::Rect(width(), height() - (Scene::HeaderRegion + Scene::FooterRegion + 16.0f)), 5.0f, 0.0f, HeaderRegion + 16.0f)
    , m_BackButton(op, *this, glv::Rect(384.0f, 48.0f), glv::Place::CC)
    , m_pPreviousScene(pParent)
    , m_Callback(callback)
{
    *this << m_Header << m_Footer << m_Container;

    m_Container.Refresh();
    m_Container << m_BackButton;

    for( const auto& item : items )
    {
        auto button = new SelectButton(*this, item, glv::Rect(m_Container.GetInnerRect().width(), 64.0f), glv::Place::CC);
        m_SelectButtons.push_back(std::unique_ptr<SelectButton>(button));
        m_Container << button;
    }
}

template <typename T>
SelectorScene<T>::~SelectorScene() NN_NOEXCEPT
{
    Clear();
}

template <typename T>
glv::View* SelectorScene<T>::GetPrimaryView() NN_NOEXCEPT
{
    return &m_BackButton;
}

template <typename T>
void SelectorScene<T>::PopPreviousScene() NN_NOEXCEPT
{
    m_Op.PopScene(this, m_pPreviousScene);
}

template <typename T>
void SelectorScene<T>::OnSelected(T data) NN_NOEXCEPT
{
    m_Callback(data);
    PopPreviousScene();
}

template <typename T>
void SelectorScene<T>::Close() NN_NOEXCEPT
{
    disable( glv::Property::Visible );
    m_pPreviousScene->enable( glv::Property::Visible );
    m_pPreviousScene->RequireRefresh();
}

template <typename T>
void SelectorScene<T>::Clear() NN_NOEXCEPT
{
}

template <typename T>
void SelectorScene<T>::CloseScene() NN_NOEXCEPT
{
    Close();
}

template <typename T>
void SelectorScene<T>::Refresh() NN_NOEXCEPT
{
    Clear();
    ClearRefreshRequest();

    auto& glvRoot = reinterpret_cast< glv::GLV& >( this->root() );
    NN_ASSERT( glv::GLV::valid( &glvRoot ) );
    glvRoot.setFocus( GetPrimaryView() );

    m_Op.SetFocusableChild( GetPrimaryView() );
}

template <typename T>
class SelectorSceneWithKeyboardInput : public SelectorScene<T>
{
private:
    virtual void OnSelected(T data) NN_NOEXCEPT NN_OVERRIDE;

    std::function<nn::Result(const T& arg)> m_CallbackSupportingKeyboardInput;

public:
    SelectorSceneWithKeyboardInput(
        AbstractOperators& op, glv::Rect rect, Scene* pParent, const glv::WideString& sceneTitle, const std::vector<typename SelectorScene<T>::SelectItem>& items, std::function<nn::Result(const T& arg)> callback) NN_NOEXCEPT;
};

template <typename T>
SelectorSceneWithKeyboardInput<T>::SelectorSceneWithKeyboardInput(
    AbstractOperators& op, glv::Rect rect, Scene* pParent, const glv::WideString& sceneTitle, const std::vector<typename SelectorScene<T>::SelectItem>& items, std::function<nn::Result(const T& arg)> callback) NN_NOEXCEPT
    : SelectorScene<T>(op, rect, pParent, sceneTitle, items, nullptr)
    , m_CallbackSupportingKeyboardInput(callback)
{
}

template <typename T>
void SelectorSceneWithKeyboardInput<T>::OnSelected(T data) NN_NOEXCEPT
{
    if(!nn::swkbd::ResultCanceled::Includes(m_CallbackSupportingKeyboardInput(data)))
    {
        SelectorScene<T>::PopPreviousScene();
    }
}

template <typename T>
class CancellableSelectorScene : public SelectorScene<T>
{
private:
    virtual void OnSelected(T data) NN_NOEXCEPT NN_OVERRIDE;

    Page* m_pParentPage;
    std::vector<std::string> m_DialogMessages;

public:
    CancellableSelectorScene(
        AbstractOperators& op, glv::Rect rect, Page* pPage, Scene* pParent,
        const glv::WideString& sceneTitle,
        const std::vector<typename SelectorScene<T>::SelectItem>& items,
        const std::vector<std::string>& messages,
        std::function<void(T arg)> callback) NN_NOEXCEPT;

};

template <typename T>
CancellableSelectorScene<T>::CancellableSelectorScene(
    AbstractOperators& op, glv::Rect rect, Page* pPage,
    Scene* pParent, const glv::WideString& sceneTitle,
    const std::vector<typename SelectorScene<T>::SelectItem>& items,
    const std::vector<std::string>& messages,
    std::function<void(T arg)> callback) NN_NOEXCEPT
    : SelectorScene<T>( op, rect, pParent, sceneTitle, items, callback )
    , m_pParentPage( pPage )
    , m_DialogMessages( messages )
{
}

template <typename T>
void CancellableSelectorScene<T>::OnSelected(T data) NN_NOEXCEPT
{
    auto pView = new MessageView();

    for (const auto& message : m_DialogMessages )
    {
        pView->AddMessage( message.c_str() );
    }

    pView->AddButton( "Cancel" );

    pView->AddButton(
        "OK",
        [this, data](void* pParam, nn::TimeSpan& timespan)
        {
            SelectorScene<T>::OnSelected( data );
            SelectorScene<T>::PopPreviousScene();
        }
    );
    m_pParentPage->GetRootSurfaceContext()->StartModal(pView, true);
}

}} // ~namespace devmenu::accounts, ~namespace devmenu
