﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "DevMenu_AccountsScenePlayData.h"
#include "DevMenu_AccountsSdkHelper.h"
#include "../DevMenu_Config.h"
#include "../Applications/DevMenu_ApplicationsCommon.h"

#include <nn/ns/ns_ApplicationRecordApi.h>
#include <nn/pdm/pdm_QueryApiForSystem.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Country.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>

namespace devmenu { namespace accounts {

// PlayDataInfoScene::PlayStatisticsView

PlayDataInfoScene::PlayStatisticsView::PlayStatisticsView() NN_NOEXCEPT
    : glv::View(glv::Rect(640, 160), glv::Place::TC)
    , LabelSpec(glv::Place::TL, 0.0f, 0.0f, Default::FontSize)
    , m_Placer(*this, glv::Direction::E, glv::Place::CL, 0.0f, height() / 2, 0.0f)
    , m_Table("< <", 16, 16)
    , m_Icon(128, 128, 16, 16)
    , m_ApplicationIdLabel("Application ID", LabelSpec)
    , m_ApplicationIdValue("0x0000000000000000", LabelSpec)
    , m_PlayCountLabel("Play Count", LabelSpec)
    , m_PlayCountValue("0", LabelSpec)
    , m_PlayTimeLabel("Play Time", LabelSpec)
    , m_PlayTimeValue("0 minutes", LabelSpec)
{
    disable(glv::Property::Controllable | glv::Property::HitTest);
    m_Table << m_ApplicationIdLabel;
    m_Table << m_ApplicationIdValue;
    m_Table << m_PlayCountLabel;
    m_Table << m_PlayCountValue;
    m_Table << m_PlayTimeLabel;
    m_Table << m_PlayTimeValue;
    m_Table.arrange();
    m_Placer << m_Icon;
    m_Placer << m_Table;
}

void PlayDataInfoScene::PlayStatisticsView::SetPlayStatistics(const nn::pdm::PlayStatistics& statistics) NN_NOEXCEPT
{
    // アイコン
    std::unique_ptr< char[] > buffer;
    size_t dataSize = 0;

    const auto result = application::GetApplicationControlData(&dataSize, &buffer, nn::ns::ApplicationControlSource::StorageOnly, statistics.applicationId);
    if( result.IsSuccess() )
    {
        nn::ns::ApplicationControlDataAccessor accessor(buffer.get(), dataSize);
        m_Icon.SetTexture(accessor.GetIconData(), accessor.GetIconSize());
    }

    char str[128];

    // アプリケーションID
    nn::util::SNPrintf(str, sizeof(str), "0x%016llx", statistics.applicationId.value);
    m_ApplicationIdValue.setValue(std::string(str));

    // プレイ回数
    nn::util::SNPrintf(str, sizeof(str), "%u", statistics.totalPlayCount);
    m_PlayCountValue.setValue(std::string(str));

    // プレイ時間
    if( statistics.totalPlayTime >= 60 )
    {
        nn::util::SNPrintf(str, sizeof(str), "%u hours %u minutes", statistics.totalPlayTime / 60, statistics.totalPlayTime % 60);
    }
    else
    {
        nn::util::SNPrintf(str, sizeof(str), "%u minutes", statistics.totalPlayTime);
    }
    m_PlayTimeValue.setValue(std::string(str));
}

// PlayDataInfoScene::UserInfoView

PlayDataInfoScene::UserInfoView::UserInfoView() NN_NOEXCEPT
    : glv::View(glv::Rect(640, 96), glv::Place::TC)
    , m_Table("< <,| <", 8, 8)
    , m_Icon(64, 64, 8, 8)
    , m_pNickname(nullptr)
    , m_Uid("UID: XXXXXXXX_XXXXXXXX_XXXXXXXX_XXXXXXXX", glv::Label::Spec(glv::Place::TL, 0.0f, 0.0f, Default::FontSizeSmall))
{
    disable(glv::Property::Controllable | glv::Property::HitTest);
    *this << m_Table;
}

void PlayDataInfoScene::UserInfoView::SetAccount(const nn::account::Uid& uid) NN_NOEXCEPT
{
    m_Icon.remove();
    if( m_pNickname )
    {
        m_pNickname->remove();
        delete m_pNickname;
    }
    m_Uid.remove();

    glv::CharTypeU16 nickname[64];
    GetNickname(nickname, NN_ARRAY_SIZE(nickname), uid);
    m_pNickname = new glv::Label(glv::WideString(nickname), glv::Label::Spec(glv::Place::TL, 0.0f, 0.0f, Default::FontSize));

    char str[128];
    int n = 0;
    n += nn::util::SNPrintf(&str[n], sizeof(str) - n, "UID: ");
    n += UidToString(&str[n], sizeof(str) - n, uid);
    m_Uid.setValue(std::string(str));

    m_Icon.SetTexture(LoadUserIcon(uid));

    m_Table << m_Icon;
    m_Table << m_pNickname;
    m_Table << m_Uid;
    m_Table.arrange();
}

// PlayDataInfoScene

PlayDataInfoScene::PlayDataInfoScene(const AbstractOperators& op, glv::Rect rect) NN_NOEXCEPT
    : Scene(rect)
    , m_User(nn::account::InvalidUid)
    , m_Base(0, 0, rect.width(), rect.height())
    , m_Placer()
    , m_Header("Account Play Data", glv::Label::Spec(glv::Place::TC, 0.0f, 0.0f, CommonValue::InitialFontSize))
    , m_NoStatisticsMessage("Not Found", glv::Label::Spec(glv::Place::TC, 0.0f, 0.0f, CommonValue::InitialFontSize))
    , m_Closer(op, glv::Rect(384.0f, 48.0))
{
    *this << m_Base;
}
PlayDataInfoScene::~PlayDataInfoScene() NN_NOEXCEPT
{
    PlayDataInfoScene::Clear();
}
glv::View* PlayDataInfoScene::GetPrimaryView() NN_NOEXCEPT
{
    return &m_Closer;
}
void PlayDataInfoScene::SetAccount(const nn::account::Uid& user) NN_NOEXCEPT
{
    m_User = user;
    m_UserInfoView.SetAccount(user);
}

void PlayDataInfoScene::Clear() NN_NOEXCEPT
{
    auto& g = reinterpret_cast<glv::GLV&>(this->root());
    NN_ASSERT(glv::GLV::valid(&g));
    g.setFocus(nullptr);

    m_Placer = glv::Placer(m_Base, glv::Direction::S, glv::Place::TC, 0.0f, 0.0f, 16.0f);
    m_Header.remove();
    m_Closer.remove();
    m_UserInfoView.remove();
    for( int i = 0; i < PlayStatisticsCount; i++ )
    {
        m_PlayStatisticsView[i].remove();
    }
    m_NoStatisticsMessage.remove();
}
void PlayDataInfoScene::Refresh() NN_NOEXCEPT
{
    Clear();
    ClearRefreshRequest();

    m_Placer << m_Header;
    m_Placer << m_Closer;
    m_Placer << m_UserInfoView;

#if !defined(NN_CUSTOMERSUPPORTTOOL)

    nn::pdm::InitializeForQuery();
    NN_UTIL_SCOPE_EXIT{ nn::pdm::FinalizeForQuery(); };

    int recordListOffset = 0;
    int statisticsFoundCount = 0;
    while( statisticsFoundCount < PlayStatisticsCount )
    {
        const int RecordListCountMax = 8;
        std::vector<nn::ns::ApplicationRecord> recordList(RecordListCountMax);
        auto recordListCount = nn::ns::ListApplicationRecord(recordList.data(), static_cast<int>(recordList.size()), recordListOffset);
        if( recordListCount == 0 )
        {
            break;
        }
        recordListOffset += recordListCount;

        for( int i = 0; i < recordListCount && statisticsFoundCount < PlayStatisticsCount; i++ )
        {
            auto statistics = nn::pdm::QueryPlayStatistics(recordList[i].id, m_User);
            if( !statistics )
            {
                DEVMENU_LOG("PlayDataInfoScene::Refresh : PlayStatistics for 0x%016llx is not found\n", recordList[i].id.value);
                continue;
            }
            m_PlayStatisticsView[statisticsFoundCount].SetPlayStatistics(*statistics);
            m_Placer << m_PlayStatisticsView[statisticsFoundCount];
            statisticsFoundCount++;
        }
    }
    if( statisticsFoundCount == 0 )
    {
        DEVMENU_LOG("PlayDataInfoScene::Refresh : No PlayStatistics found\n");
        m_Placer << m_NoStatisticsMessage;
    }
#else
    m_Placer << m_NoStatisticsMessage;
#endif

    auto& g = reinterpret_cast<glv::GLV&>(this->root());
    NN_ASSERT(glv::GLV::valid(&g));
    g.setFocus(GetPrimaryView());
}

}} // ~namespace devmenu::accounts, ~namespace devmenu
