﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/hid/system/hid_UniquePad.h>
#include <nn/hid/debug/hid_UniquePad.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/result/result_HandlingUtility.h>

#include "ControllerSerialFlashTool_SerialFlash.h"
#include "ControllerSerialFlashTool_Utility.h"

namespace
{
//!< Serial Flash の中身をバッファに読み出す
bool ReadSerialFlash(uint8_t* pBuffer, const int size, const nn::hid::system::UniquePadId& id) NN_NOEXCEPT;

//!< Serial Flash の中身をパースし、CSV 形式に出力する
void ParseSerialFlash(SerialFlashCsvType* pOutValue, const uint8_t* pBuffer, const int size) NN_NOEXCEPT;

//!< Serial Flash への書き込みを行う
bool WriteSerialFlash(const nn::hid::system::UniquePadId& id, const uint16_t& address,  const int length, const uint8_t* pBuffer, const int bufferSize) NN_NOEXCEPT;

//!< Serial Flash の読み書きに用いるバッファ
NN_ALIGNAS(4096) uint8_t g_Buffer[SerialFlashSize];

}
bool ReadSerialFlashAsCsvText(SerialFlashCsvType* pOutValue, const nn::hid::system::UniquePadId& id) NN_NOEXCEPT
{
    COMMAND_DO(ReadSerialFlash(g_Buffer, NN_ARRAY_SIZE(g_Buffer), id));
//    PrintBuffer(g_Buffer, NN_ARRAY_SIZE(g_Buffer), SerialFlashStartAddress);
    ParseSerialFlash(pOutValue, g_Buffer, NN_ARRAY_SIZE(g_Buffer));
    return true;
}

bool WriteSerialFlashFromCsv(const SerialFlashCsvType& input, const nn::hid::system::UniquePadId& id) NN_NOEXCEPT
{
    bool commandFound = false;

    uint16_t addressToWrite = 0;
    int length = 0;
    std::memset(g_Buffer, 0, NN_ARRAY_SIZE(g_Buffer));

    uint16_t serialFlashMapAddress = 0;
    for (auto& entity : SerialFlashMap)
    {
        bool lineFound = false;
        for (int lineIndex = 0; lineIndex < input.lineCount; ++lineIndex)
        {
            auto& line = input.cell[lineIndex];
            if (entity.category == line[0] && entity.name == line[1])
            {
                // 有効な要素が見つかった
                if (commandFound == false)
                {
                    addressToWrite = serialFlashMapAddress + SerialFlashStartAddress;
                    commandFound = true;
                }
                entity.parser.Generate(&g_Buffer[length], line[2]);
                length += entity.parser.size;
                lineFound = true;
                break;
            }
        }
        serialFlashMapAddress += static_cast<uint16_t>(entity.parser.size);

        // コマンドが見つかっている際に、データの連続性が途切れたらそこで一度書き込みを行う
        if (lineFound == false && commandFound == true)
        {
            COMMAND_DO(WriteSerialFlash(id, addressToWrite, length, g_Buffer, NN_ARRAY_SIZE(g_Buffer)));
            commandFound = false;
            length = 0;
            std::memset(g_Buffer, 0, NN_ARRAY_SIZE(g_Buffer));
        }
    }


    // 書き切れていないコマンドをここで書き込み
    if (commandFound == true)
    {
        COMMAND_DO(WriteSerialFlash(id, addressToWrite, length, g_Buffer, NN_ARRAY_SIZE(g_Buffer)));
    }

    return true;
}

namespace
{
bool ReadSerialFlash(uint8_t* pBuffer, const int size, const nn::hid::system::UniquePadId& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_SDK_REQUIRES_GREATER(size, 0);
    NN_SDK_REQUIRES_ALIGNED(size, 4096);

    nn::os::SystemEventType flashEvent;
    nn::hid::debug::BindSerialFlashEventHandle(&flashEvent, nn::os::EventClearMode_AutoClear, id);
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hid::debug::ReadSerialFlash(SerialFlashStartAddress, pBuffer, size, id));
    NN_LOG("[Serial Flash Tool] Start Reading Flash for pad 0x%llx\n", id._storage);
    nn::os::WaitSystemEvent(&flashEvent);
    nn::os::DestroySystemEvent(&flashEvent);

    auto result = nn::hid::debug::GetSerialFlashResult(id);
    if (result.IsFailure())
    {
        PrintError(result, "Reading Serila Flash Failed");
    }
    return result.IsSuccess();
}

void ParseSerialFlash(SerialFlashCsvType* pOutValue, const uint8_t* pBuffer, const int size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_SDK_REQUIRES_GREATER(size, 0);
    NN_UNUSED(size);

    int address = 0;
    pOutValue->lineCount = 0;
    for (auto& entity : SerialFlashMap)
    {
        if(entity.category != "Reserved")
        {
            auto& line = pOutValue->cell[pOutValue->lineCount];
            line[0] = entity.category;
            line[1] = entity.name;
            line[2] = entity.parser.Parse(&pBuffer[address]);
            ++pOutValue->lineCount;
//            NN_LOG("[Serial Flash Tool] %s\t%s\t%s\n", entity.category.c_str(), entity.name.c_str(), entity.parser.Parse(&pBuffer[address]).c_str());
        }
        address += entity.parser.size;
    }
}

bool WriteSerialFlash(const nn::hid::system::UniquePadId& id, const uint16_t& address,  const int length, const uint8_t* pBuffer, const int bufferSize) NN_NOEXCEPT
{
    nn::os::SystemEventType flashEvent;
    nn::hid::debug::BindSerialFlashEventHandle(&flashEvent, nn::os::EventClearMode_AutoClear, id);

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hid::debug::WriteSerialFlash(address, pBuffer, bufferSize, length, id));
    NN_LOG("[Serial Flash Tool] Writing Flash for pad 0x%llx to address %04x length %d\n", id._storage, address, length);
    nn::os::WaitSystemEvent(&flashEvent);
    nn::os::DestroySystemEvent(&flashEvent);

    auto result = nn::hid::debug::GetSerialFlashResult(id);
    if (result.IsFailure())
    {
        PrintError(result, "Writing Serila Flash Failed");
    }
    return result.IsSuccess();
}

}
