﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <iostream>
#include <string>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>

struct SerialFlashParserType
{
    int size;
    std::string (*Parse)(const uint8_t* pBuffer) NN_NOEXCEPT;
    void (*Generate)(uint8_t* pBuffer, const std::string input) NN_NOEXCEPT;
};

const int IdentificationCodeLength = 16;
const SerialFlashParserType IdentificationCodeParserType =
{
    IdentificationCodeLength,
    [](const uint8_t* pBuffer) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pBuffer);
        std::string output;
        for (int charIndex = 0; charIndex < IdentificationCodeLength; ++charIndex)
        {
            if (pBuffer[charIndex] != 0)
            {
                output.push_back(pBuffer[charIndex]);
            }
        }
        return output;
    },
    [](uint8_t* pBuffer, const std::string input) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pBuffer);
        auto offset = IdentificationCodeLength - input.length();
        for (size_t charIndex = 0; charIndex < input.length(); ++charIndex)
        {
            pBuffer[charIndex + offset] = input.at(charIndex);
        }
    }
};

inline SerialFlashParserType ReservedParserType(int bytes)
{
    return
    {
    bytes,
    [](const uint8_t* pBuffer) NN_NOEXCEPT
    {
        NN_UNUSED(pBuffer);
        std::string output;
        return output;
    },
    [](uint8_t* pBuffer, const std::string input) NN_NOEXCEPT
    {
        NN_UNUSED(pBuffer);
        NN_UNUSED(input);
    }
    };
};

const SerialFlashParserType u8ParserType =
{
    1,
    [](const uint8_t* pBuffer) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pBuffer);
        std::string output = std::to_string(pBuffer[0]);
        return output;
    },
    [](uint8_t* pBuffer, const std::string input) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pBuffer);
        pBuffer[0] = static_cast<uint8_t>(std::stoi(input));
    }
};

const SerialFlashParserType s16ParserType =
{
    2,
    [](const uint8_t* pBuffer) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pBuffer);
        int16_t value = (pBuffer[0] | ((pBuffer[1] << 8) & 0xff00));
        std::string output = std::to_string(value);
        return output;
    },
    [](uint8_t* pBuffer, const std::string input) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pBuffer);
        auto value = std::stoi(input);
        pBuffer[0] = static_cast<uint8_t>(value & 0xff);
        pBuffer[1] = static_cast<uint8_t>((value >> 8) & 0xff);

    }
};

const SerialFlashParserType u16ParserType =
{
    2,
    [](const uint8_t* pBuffer) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pBuffer);
        uint16_t value = (pBuffer[0] | ((pBuffer[1] << 8) & 0xff00));
        std::string output = std::to_string(value);
        return output;
    },
    [](uint8_t* pBuffer, const std::string input) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pBuffer);
        auto value = std::stoi(input);
        pBuffer[0] = static_cast<uint8_t>(value & 0xff);
        pBuffer[1] = static_cast<uint8_t>((value >> 8) & 0xff);

    }
};

const SerialFlashParserType u12HigherParserType =
{
    1,
    [](const uint8_t* pBuffer) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pBuffer);
        int16_t value = pBuffer[0] | ((pBuffer[1] << 8) & 0x0f00);
        std::string output = std::to_string(value);
        return output;
    },
    [](uint8_t* pBuffer, const std::string input) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pBuffer);
        auto value = std::stoi(input);
        pBuffer[0] = static_cast<uint8_t>(value & 0xff);
        pBuffer[1] |= static_cast<uint8_t>((value >> 8) & 0x0f);

    }
};

const SerialFlashParserType u12LowerParserType =
{
    2,
    [](const uint8_t* pBuffer) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pBuffer);
        int16_t value = ((pBuffer[0] >> 4) &0x0f) | ((pBuffer[1] << 4) & 0x0ff0);
        std::string output = std::to_string(value);
        return output;
    },
    [](uint8_t* pBuffer, const std::string input) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pBuffer);
        auto value = std::stoi(input);
        pBuffer[0] |= static_cast<uint8_t>((value << 4) & 0xf0);
        pBuffer[1] = static_cast<uint8_t>((value >> 4) & 0xff);

    }
};
